function varargout = dwy_pnas_demo2(varargin)
%DWY_PNAS_DEMO2 Demonstration of ORN model by Dougherty, Wright and Yew (2005).
%       This program simulates the transduction current response of a single
%       olfactory receptor neuron being stimulated by an odorant plume.
%       The program is interactive in that a user can tweak parameter values 
%       and stimulus conditions.  Also, users can save a configuration
%       in a mat-file or export all aspects to a directory of text files. 
%       These text files can be read by other programs.  
%       There is also an import facility for importing text files from a
%       directory that allows the user to specify their own data, pulses 
%       and parameters.  
%
%       The names and formats of the text files are as follows:
%               dwy_pnas_data.txt
%                  Col1: Run (i.e. experiment number)
%                  Col2: Time (s)
%                  Col3:Current (pA)
%               dwy_pnas_params.txt
%                  Col1: Parameter
%                  Col2: Value  
%               dwy_pnas_pulse.txt
%                  Col1: Specifies'ton','toff',or 'conc'
%                  Col2-??: Each value is specific for a separate pulse.
%                           Multiple rows are allowed for multiple runs per 
%                           simulation. That is, rows are pulse characteristics
%                           for a particular recording.
%               dwy_pnas_prediction.txt
%                  Col1:Time(s)
%                  Col2-??: ODE system variables in the order 'bLR','aG','cAMP',
%                          'Ca','CaCaM','aCaMK','IX','V'.  These are repeated 
%                          if there are multiple runs.
%                  Col??:end  Total current,CNG current,Cl(Ca++) current,
%                         NCX current,Leak current 
%               dwy_pnas_layout.txt
%                  Col1: Different graphical layouts are possible
%                        these are:
%                        'currents_separate_figures'
%                        'concentrations_separate_panels'
%                        'all_same_panel'
%               dwy_pnas_factor.txt
%                  Col1: All predicted currents are normalized by this 
%                        amount (pA).  This is useful if the data have been 
%                        normalized since the model only predicts un-normalized 
%                        currents.
%
%
%     Please cite the Dougherty, Wright, Yew 2005, PNAS paper when refering 
%     to this modeling work. 
%
%

%
% //Copyright (C) 2005  Daniel P. Dougherty
% //
% //This program is free software; you can redistribute it and/or
% //modify it under the terms of the GNU General Public
% //License as published by the Free Software Foundation; either
% //version 2 of the License, or (at your option) any later version.
% //
% //This program is distributed in the hope that it will be useful,
% //but WITHOUT ANY WARRANTY; without even the implied warranty of
% //MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% //General Public License for more details.
% //
% //You should have received a copy of the GNU General Public
% //License along with this program; if not, write to the Free Software
% //Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


switch nargin
case 0 
        clear global STARTING_TIME;
        
	global STARTING_TIME;
        
        CONFIG_FILE = 'dwy_pnas.mat';
        
	NAMES = {'single','adaptation','prolonged','saved...','import...'};
	[s,v] = listdlg('PromptString','Select a dataset:',...
                	      'SelectionMode','single',...
                	      'ListString',NAMES);
	if (v ~= 1)
		return;
	else
                NORMALIZING_FACTOR = 1;
                
		switch NAMES{s}
                case 'import...'
                
                        EXPORTPATH = uigetdir(pwd,'Select export directory...');
                        
                        DATA_FILE       = 'dwy_pnas_data.txt';
                        PARAMS_FILE     = 'dwy_pnas_params.txt';
                        PULSE_FILE      = 'dwy_pnas_pulse.txt';
                        PREDICTION_FILE = 'dwy_pnas_prediction.txt';
                        GRAPHICS_LAYOUT_FILE = 'dwy_pnas_layout.txt';
                        NORMALIZING_FACTOR_FILE = 'dwy_pnas_factor.txt';
                        
                        
                        [DATA(:,1),DATA(:,2),DATA(:,3)] = textread(fullfile(EXPORTPATH ,DATA_FILE),'%f %f %f');
                        
                        
                        %[type,val] = textread(fullfile(EXPORTPATH,PULSE_FILE),'%s %f %f %f');
                        %
                        
                        fid = fopen(fullfile(EXPORTPATH,PULSE_FILE),'r');
                        firstline = fgetl(fid);
                        pulses = str2num(firstline(4:end));
                         
                        [type,val{1:length(pulses)}] = textread(fullfile(EXPORTPATH,PULSE_FILE),['%s',repmat('%f',1,length(pulses))]);
                        
                        V = cat(2,val{:});
                        
                        type = cellstr(type);
                        kon = 1;
                        koff = 1;
                        kconc = 1;
                        for (j=1:length(type))
                                switch type{j}
                                case 'ton'
                                        PULSE.ton(kon,:) = V(j,:);
                                        kon = kon+1;
                                case 'toff'
                                        PULSE.toff(koff,:) = V(j,:);
                                        koff = koff+1;
                                case 'conc'
                                        PULSE.conc(kconc,:) = V(j,:);
                                        kconc = kconc+1;
                                otherwise
                                        error('Unrecognized pulse identifier.');
                                end
                        end
                       
                        
                        
                        [Pms,V] = textread(fullfile(EXPORTPATH,PARAMS_FILE),'%s %f');
                        Pms = cellstr(Pms);
                        V = num2cell(V);
                        PARAMS_HAT = cell2struct(V,Pms,1);
                        
                        
                        
                        [PREDICTION{1:14}] = textread(fullfile(EXPORTPATH ,DATA_FILE),repmat('%f',1,14));
                        T = PREDICTION{1};
                        Y = cat(2,PREDICTION{2:10});
                        
                        
                        
                        GRAPHICS_LAYOUT = textread(fullfile(EXPORTPATH ,GRAPHICS_LAYOUT_FILE),'%s',1);
                        GRAPHICS_LAYOUT = GRAPHICS_LAYOUT{1}; 
                        
                        NORMALIZING_FACTOR = textread(fullfile(EXPORTPATH,NORMALIZING_FACTOR_FILE),'%f',1);
                        
                        
                case 'saved...'
                        
                        [filename, pathname] = uigetfile('*.mat', 'Pick an .mat file');
                        CONFIG_FILE = fullfile(pathname,filename);
                        
                        CONFIG = load(CONFIG_FILE);
                        
                        DATA = CONFIG.DATA;
                        
                        PULSE = CONFIG.PULSE;
                        
                        PARAMS_HAT = CONFIG.PARAMS_HAT;
                            

			GRAPHICS_LAYOUT = CONFIG.GRAPHICS_LAYOUT;
                        
                
		case 'single'

			DATA = [1.0000   -0.8714         0
				      1.0000   -0.7683         0
				      1.0000   -0.6652   -0.0553
				      1.0000   -0.5621   -0.1515
				      1.0000   -0.4590   -0.2477
				      1.0000   -0.3559   -0.3257
				      1.0000   -0.2528   -0.3257
				      1.0000   -0.1496   -0.3257
				      1.0000   -0.0465   -0.3257
				      1.0000    0.0566   -0.4985
				      1.0000    0.1597   -0.7135
				      1.0000    0.2628   -0.9286
				      1.0000    0.2857  -13.4457
				      1.0000    0.3086  -25.9629
				      1.0000    0.3315  -38.4801
				      1.0000    0.3544  -50.9973
				      1.0000    0.3773  -60.8449
				      1.0000    0.4003  -68.0229
				      1.0000    0.4232  -75.2010
				      1.0000    0.4461  -82.3790
				      1.0000    0.4690  -89.5571
				      1.0000    0.5721  -81.7881
				      1.0000    0.6752  -73.3871
				      1.0000    0.7783  -63.0887
				      1.0000    0.8814  -60.7230
				      1.0000    0.9845  -58.3574
				      1.0000    1.0876  -56.1884
				      1.0000    1.1907  -54.5010
				      1.0000    1.2938  -52.8136
				      1.0000    1.3970  -51.8790
				      1.0000    1.5001  -50.9650
				      1.0000    1.6032  -49.8150
				      1.0000    1.7063  -46.9714
				      1.0000    1.8094  -45.3393
				      1.0000    1.9125  -44.8315
				      1.0000    2.0156  -44.3237
				      1.0000    2.1187  -43.8054
				      1.0000    2.2218  -43.2638
				      1.0000    2.3249  -42.7221
				      1.0000    2.4280  -42.1805
				      1.0000    2.5311  -41.6389
				      1.0000    2.6342  -40.7111
				      1.0000    2.7373  -39.3973
				      1.0000    2.8404  -38.0834
				      1.0000    2.9435  -36.7695
				      1.0000    3.0467  -35.4557
				      1.0000    3.1498  -34.1418
				      1.0000    3.2529  -32.1260
				      1.0000    3.3560  -29.9324
				      1.0000    3.4591  -27.7388
				      1.0000    3.5622  -25.1252
				      1.0000    3.6653  -21.1368
				      1.0000    3.7684  -17.2813
				      1.0000    3.8715  -13.7776
				      1.0000    3.9746  -10.5336
				      1.0000    4.0777   -8.9440
				      1.0000    4.1808   -7.3545
				      1.0000    4.2839   -5.7649
				      1.0000    4.3870   -4.2086
				      1.0000    4.4901   -3.5122
				      1.0000    4.5933   -2.8158
				      1.0000    4.6964   -2.1195
				      1.0000    4.7995   -1.5130
				      1.0000    4.9026   -1.1213
				      1.0000    5.0057   -0.7296
				      1.0000    5.1088   -0.3379
				      1.0000    5.2119    0.0538
				      1.0000    5.3150    0.2636
				      1.0000    5.4181    0.0605
				      1.0000    5.5212   -0.1426
				      1.0000    5.6243   -0.3457
				      1.0000    5.7274   -0.5488
				      1.0000    5.8305   -0.8452
				      1.0000    5.9336   -1.2370
				      1.0000    6.0367   -1.6287
				      2.0000   -0.8714    0.3257
				      2.0000   -0.7754    0.2362
				      2.0000   -0.6794    0.1466
				      2.0000   -0.5834    0.0570
				      2.0000   -0.4874         0
				      2.0000   -0.3914         0
				      2.0000   -0.2954         0
				      2.0000   -0.1994         0
				      2.0000   -0.1034         0
				      2.0000   -0.0074         0
				      2.0000    0.0886         0
				      2.0000    0.1847         0
				      2.0000    0.2167   -0.4022
				      2.0000    0.2487   -0.8044
				      2.0000    0.2807   -1.2066
				      2.0000    0.3127  -10.7031
				      2.0000    0.3447  -20.1997
				      2.0000    0.3767  -29.6962
				      2.0000    0.4087  -45.7550
				      2.0000    0.4407  -61.8137
				      2.0000    0.4727  -77.8724
				      2.0000    0.5687  -75.0486
				      2.0000    0.6647  -69.1792
				      2.0000    0.7607  -63.7321
				      2.0000    0.8567  -60.9826
				      2.0000    0.9527  -58.2331
				      2.0000    1.0487  -56.0109
				      2.0000    1.1447  -54.0470
				      2.0000    1.2407  -52.1778
				      2.0000    1.3367  -50.7311
				      2.0000    1.4327  -49.2843
				      2.0000    1.5287  -47.8375
				      2.0000    1.6247  -46.5018
				      2.0000    1.7207  -45.3671
				      2.0000    1.8167  -44.2324
				      2.0000    1.9127  -42.3063
				      2.0000    2.0087  -39.5404
				      2.0000    2.1047  -34.0774
				      2.0000    2.2007  -29.9460
				      2.0000    2.2967  -26.7310
				      2.0000    2.3927  -22.0456
				      2.0000    2.4887  -17.9887
				      2.0000    2.5847  -14.5846
				      2.0000    2.6808  -11.8200
				      2.0000    2.7768   -9.4171
				      2.0000    2.8728   -7.0142
				      2.0000    2.9688   -4.9319
				      2.0000    3.0648   -3.3607
				      2.0000    3.1608   -1.7896
				      2.0000    3.2568   -1.5483
				      2.0000    3.3528   -1.4587
				      2.0000    3.4488   -1.3691
				      2.0000    3.5448   -1.2490
				      2.0000    3.6408   -1.0427
				      2.0000    3.7368   -0.8364
				      2.0000    3.8328   -0.6301		
				      2.0000    3.9288   -0.4237
				      2.0000    4.0248   -0.2174
				      2.0000    4.1208   -0.0111
				      2.0000    4.2168   -0.1150
				      2.0000    4.3128   -0.2366
				      2.0000    4.4088   -0.3030
				      2.0000    4.5048   -0.2179
				      2.0000    4.6008   -0.1328
				      2.0000    4.6968   -0.0477
				      2.0000    4.7928   -0.0356
				      2.0000    4.8888   -0.1167
				      2.0000    4.9848   -0.1977
				      2.0000    5.0808   -0.2788
				      2.0000    5.1769   -0.3556
				      2.0000    5.2729   -0.4265
				      2.0000    5.3689   -0.4974
				      2.0000    5.4649   -0.5683
				      2.0000    5.5609   -0.6392
				      2.0000    5.6569   -0.7923
				      2.0000    5.7529   -0.9625
				      2.0000    5.8489   -1.1327
				      2.0000    5.9449   -1.3029
				      3.0000   -0.7612    0.3257
				      3.0000   -0.6672    0.2500
				      3.0000   -0.5733    0.1743
				      3.0000   -0.4794    0.0986
				      3.0000   -0.3854    0.0229
				      3.0000   -0.2915   -0.0553
				      3.0000   -0.1976   -0.1346
				      3.0000   -0.1036   -0.2139
				      3.0000   -0.0097   -0.2932
				      3.0000    0.0842   -0.3748
				      3.0000    0.1782   -0.4581
				      3.0000    0.2721   -0.5414
				      3.0000    0.3660   -0.6246
				      3.0000    0.3869   -8.2188
				      3.0000    0.4078  -15.8130
				      3.0000    0.4287  -23.4071
				      3.0000    0.4495  -31.0013
				      3.0000    0.4704  -38.2188
				      3.0000    0.4913  -45.0597
				      3.0000    0.5122  -51.9006
				      3.0000    0.5330  -58.7415
				      3.0000    0.5539  -65.5824
				      3.0000    0.6479  -64.0283
				      3.0000    0.7418  -59.4279
				      3.0000    0.8357  -55.0596
				      3.0000    0.9297  -51.8542
				      3.0000    1.0236  -51.4101
				      3.0000    1.1175  -50.9660
				      3.0000    1.2115  -49.9371
				      3.0000    1.3054  -48.6049
				      3.0000    1.3993  -47.2726
				      3.0000    1.4933  -45.4576
				      3.0000    1.5872  -43.4592
				      3.0000    1.6811  -33.7615
				      3.0000    1.7751  -27.3191
				      3.0000    1.8690  -22.1842
				      3.0000    1.9629  -17.0494
				      3.0000    2.0569  -11.2722
				      3.0000    2.1508   -6.7899
				      3.0000    2.2447   -4.5373
				      3.0000    2.3387   -3.4271
				      3.0000    2.4326   -2.3168
				      3.0000    2.5265   -1.5834
				      3.0000    2.6205   -1.4645
				      3.0000    2.7144   -1.3455
				      3.0000    2.8083   -1.3029
				      3.0000    2.9023   -1.3029
				      3.0000    2.9962   -1.3029
				      3.0000    3.0901   -1.2421
				      3.0000    3.1841   -1.0499
				      3.0000    3.2780   -0.8577
				      3.0000    3.3719   -0.6656
				      3.0000    3.4659   -0.4734
				      3.0000    3.5598   -0.3257
				      3.0000    3.6537   -0.3257
				      3.0000    3.7477   -0.3257
				      3.0000    3.8416   -0.3257
				      3.0000    3.9355   -0.4177
				      3.0000    4.0295   -0.5961
				      3.0000    4.1234   -0.7745
				      3.0000    4.2173   -0.9529
				      3.0000    4.3113   -1.1314
				      3.0000    4.4052   -1.3029
				      3.0000    4.4991   -1.3029
				      3.0000    4.5931   -1.3029
				      3.0000    4.6870   -1.3029
				      3.0000    4.7809   -1.3029
				      3.0000    4.8749   -1.3029
				      3.0000    4.9688   -1.3029
				      3.0000    5.0627   -1.3029
				      3.0000    5.1567   -1.3029
				      3.0000    5.2506   -1.2448
				      3.0000    5.3445   -1.1523
				      3.0000    5.4385   -1.0598
				      3.0000    5.5324   -0.9942
				      3.0000    5.6263   -1.1528
				      3.0000    5.7203   -1.3115
				      3.0000    5.8142   -1.4701
				      3.0000    5.9081   -1.6287
				      4.0000   -0.8163    0.3257
				      4.0000   -0.7054    0.0450
				      4.0000   -0.5946   -0.2357
				      4.0000   -0.4838   -0.3257
				      4.0000   -0.3729   -0.3257
				      4.0000   -0.2621   -0.3257
				      4.0000   -0.1512   -0.2416
				      4.0000   -0.0404   -0.0104
				      4.0000    0.0704    0.2208
				      4.0000    0.1813   -0.7476
				      4.0000    0.2921   -2.7127
				      4.0000    0.4030   -4.6777
				      4.0000    0.5138  -31.7403
				      4.0000    0.6246  -48.4581
				      4.0000    0.7355  -48.8653
				      4.0000    0.8463  -48.0792
				      4.0000    0.9571  -47.4222
				      4.0000    1.0680  -46.8081
				      4.0000    1.1788  -46.0914
				      4.0000    1.2897  -44.4287
				      4.0000    1.4005  -42.7659
				      4.0000    1.5113  -33.2688
				      4.0000    1.6222  -16.7428
				      4.0000    1.7330   -9.0532
				      4.0000    1.8439   -5.9091
				      4.0000    1.9547   -3.8671
				      4.0000    2.0655   -3.0811
				      4.0000    2.1764   -2.6508
				      4.0000    2.2872   -2.3040
				      4.0000    2.3980   -1.9572
				      4.0000    2.5089   -1.6856
				      4.0000    2.6197   -1.4145
				      4.0000    2.7306   -1.1435
				      4.0000    2.8414   -0.8724
				      4.0000    2.9522   -0.6794
				      4.0000    3.0631   -0.8306
				      4.0000    3.1739   -0.9817
				      4.0000    3.2848   -1.1329
				      4.0000    3.3956   -1.2840
				      4.0000    3.5064   -1.3029
				      4.0000    3.6173   -1.3029
				      4.0000    3.7281   -1.3029
				      4.0000    3.8389   -1.3029
				      4.0000    3.9498   -1.2940
				      4.0000    4.0606   -1.1630
				      4.0000    4.1715   -1.0320
				      4.0000    4.2823   -0.9010
				      4.0000    4.3931   -0.7700
				      4.0000    4.5040   -0.6579
				      4.0000    4.6148   -0.7257
				      4.0000    4.7257   -0.7934
				      4.0000    4.8365   -0.8612
				      4.0000    4.9473   -0.9290
				      4.0000    5.0582   -0.9772
				      4.0000    5.1690   -0.9772
				      4.0000    5.2799   -0.9772
				      4.0000    5.3907   -0.9772
				      4.0000    5.5015   -0.9772
				      4.0000    5.6124   -0.9772
				      4.0000    5.7232   -1.0409
				      4.0000    5.8340   -1.1719
				      4.0000    5.9449   -1.3029
				      5.0000   -0.9081         0
				      5.0000   -0.8014   -0.1456
				      5.0000   -0.6947   -0.2911
				      5.0000   -0.5879   -0.1655
				      5.0000   -0.4812    0.0448
				      5.0000   -0.3745    0.2551
				      5.0000   -0.2677    0.2629
				      5.0000   -0.1610    0.1683
				      5.0000   -0.0542    0.0737
				      5.0000    0.0525    0.0182
				      5.0000    0.1592    0.1005
				      5.0000    0.2660    0.1828
				      5.0000    0.3727    0.2650
				      5.0000    0.4794   -0.4188
				      5.0000    0.5150   -2.6679
				      5.0000    0.5506   -4.9170
				      5.0000    0.5862   -7.1661
				      5.0000    0.6218  -15.4211
				      5.0000    0.6573  -23.6761
				      5.0000    0.6929  -31.9311
				      5.0000    0.7285  -34.4869
				      5.0000    0.7641  -37.0426
				      5.0000    0.7997  -39.5984
				      5.0000    0.9064  -38.7382
				      5.0000    1.0131  -37.7126
				      5.0000    1.1199  -36.2408
				      5.0000    1.2266  -34.6363
				      5.0000    1.3333  -31.9218
				      5.0000    1.4401  -25.3244
				      5.0000    1.5468  -14.7872
				      5.0000    1.6535   -5.6638
				      5.0000    1.7603   -2.9604
				      5.0000    1.8670   -1.4315
				      5.0000    1.9738   -0.9772
				      5.0000    2.0805   -0.9772
				      5.0000    2.1872   -0.9772
				      5.0000    2.2940   -0.9772
				      5.0000    2.4007   -0.9772
				      5.0000    2.5074   -0.9772
				      5.0000    2.6142   -0.9772
				      5.0000    2.7209   -0.9772
				      5.0000    2.8276   -0.9772
				      5.0000    2.9344   -0.9772
				      5.0000    3.0411   -0.9772
				      5.0000    3.1479   -0.8138
				      5.0000    3.2546   -0.5615
				      5.0000    3.3613   -0.3330
				      5.0000    3.4681   -0.4443
				      5.0000    3.5748   -0.5557
				      5.0000    3.6815   -0.6625
				      5.0000    3.7883   -0.7413
				      5.0000    3.8950   -0.8201
				      5.0000    4.0017   -0.8990
				      5.0000    4.1085   -0.9772
				      5.0000    4.2152   -0.9772
				      5.0000    4.3220   -0.9772
				      5.0000    4.4287   -0.9772
				      5.0000    4.5354   -0.9772
				      5.0000    4.6422   -0.9772
				      5.0000    4.7489   -0.9772
				      5.0000    4.8556   -0.9772
				      5.0000    4.9624   -0.9772
				      5.0000    5.0691   -0.9852
				      5.0000    5.1759   -1.1254
				      5.0000    5.2826   -1.2656
				      5.0000    5.3893   -1.4058
				      5.0000    5.4961   -1.5459
				      5.0000    5.6028   -1.5733
				      5.0000    5.7095   -1.4381
				      5.0000    5.8163   -1.3029
				      6.0000   -0.9449         0
				      6.0000   -0.8091    0.1852
				      6.0000   -0.6733    0.3257
				      6.0000   -0.5374    0.3257
				      6.0000   -0.4016    0.3257
				      6.0000   -0.2658    0.1322
				      6.0000   -0.1300   -0.2290
				      6.0000    0.0058   -0.5902
				      6.0000    0.1416   -0.6515
				      6.0000    0.2774   -0.6515
				      6.0000    0.4133   -0.7036
				      6.0000    0.5491   -0.9043
				      6.0000    0.6094   -2.1289
				      6.0000    0.6698   -3.3535
				      6.0000    0.7302   -6.4934
				      6.0000    0.7905  -10.2716
				      6.0000    0.8509  -14.0665
				      6.0000    0.9112  -17.8782
				      6.0000    0.9716  -20.8459
				      6.0000    1.0320  -21.2813
				      6.0000    1.0923  -21.7168
				      6.0000    1.2281  -19.8984
				      6.0000    1.3640  -16.0333
				      6.0000    1.4998   -8.7181
				      6.0000    1.6356   -2.6859
				      6.0000    1.7714   -1.9451
				      6.0000    1.9072   -1.3041
				      6.0000    2.0430   -0.7375
				      6.0000    2.1788   -0.2870
				      6.0000    2.3147   -0.1454
				      6.0000    2.4505   -0.0038
				      6.0000    2.5863   -0.1617
				      6.0000    2.7221   -0.3277
				      6.0000    2.8579   -0.4938
				      6.0000    2.9937   -0.6558
				      6.0000    3.1295   -0.7418
				      6.0000    3.2653   -0.8278
				      6.0000    3.4012   -0.9138
				      6.0000    3.5370   -0.9245
				      6.0000    3.6728   -0.7239
				      6.0000    3.8086   -0.5232
				      6.0000    3.9444   -0.3272
				      6.0000    4.0802   -0.4198
				      6.0000    4.2160   -0.5124
				      6.0000    4.3519   -0.6050
				      6.0000    4.4877   -0.6515
				      6.0000    4.6235   -0.6515
				      6.0000    4.7593   -0.6515
				      6.0000    4.8951   -0.6515
				      6.0000    5.0309   -0.6515
				      6.0000    5.1667   -0.6515
				      6.0000    5.3026   -0.6515
				      6.0000    5.4384   -0.6515
				      6.0000    5.5742   -0.6398
				      6.0000    5.7100   -0.5351
				      6.0000    5.8458   -0.4304
				      6.0000    5.9816   -0.3257];



      			PULSE.ton = [ 0.2000
				      0.2000
				      0.2000
				      0.2000
				      0.2000
				      0.2000];

			PULSE.toff = [1.2000
				      1.2000
				      1.2000
				      1.2000
				      1.2000
				      1.2000];

			PULSE.conc = [300
				      100
				       50
				       20
				       10
					5];


			PARAMS_HAT = struct('Sigma',2.7788,...
				   'cap',0.0043,...
				   'cc1lin',1.2242,...
				   'cc2',22.8927,...
				   'ck1lin',12.7170,...
				   'ck2',0.5564,...
				   'clmax',1.0128,...
				   'cnmax',1.2769,...
				   'cx1lin',1.1707,...
				   'cx2',16.1213,...
				   'ef',2.1622,...
				   'gl',4.5751,...
				   'hmc1',1.2298,...
				   'hmc2',2.6044,...
				   'inf',1.2601,...
				   'inhmax',1.3959,...
				   'k1',0.0235,...
				   'k2lin',9.9147,...
				   'kI',0.7037,...
				   'kinh',0.3901,...
				   'kinhcng',0.8951,...
				   'n1',1.6389,...
				   'n2',2.2759,...
				   'nI',3.7047,...
				   'ninh',1.3722,...
				   'ninhcng',1.1120,...
				   'pd',10.8842,...
				   'r1',6.9108,...
				   'r2',4.0554,...
				   'smax',91.0922,...
				   'vcl',-11.1602,...
				   'vcng',0.0027,...
				   'vl',-69.6650);

			GRAPHICS_LAYOUT = 'currents_separate_figures';

		case 'adaptation'   
                
                        NORMALIZING_FACTOR = 70;
                           
			DATA = [1.0000   -0.4495   -0.0019
				1.0000   -0.3092    0.0064
				1.0000   -0.1689   -0.0103
				1.0000    0.0181   -0.0103
				1.0000    0.2674   -0.0186
				1.0000    0.5324   -0.0103
				1.0000    0.7038   -0.0186
				1.0000    0.8908   -0.0019
				1.0000    1.1246    0.0064
				1.0000    1.4830    0.0064
				1.0000    1.7324    0.0064
				1.0000    1.9817   -0.0103
				1.0000    2.2155   -0.0103
				1.0000    2.5428   -0.0103
				1.0000    2.8077   -0.0019
				1.0000    3.1194    0.0231
				1.0000    3.2909   -0.0019
				1.0000    3.4935   -0.0103
				1.0000    3.7272   -0.0019
				1.0000    3.9298    0.0064
				1.0000    4.1168   -0.0103
				1.0000    4.2883   -0.0186
				1.0000    4.4285   -0.0186
				1.0000    4.5065   -0.0937
				1.0000    4.5844   -0.2272
				1.0000    4.6623   -0.3691
				1.0000    4.7558   -0.4692
				1.0000    4.8181   -0.5276
				1.0000    4.9896   -0.5443
				1.0000    5.0052   -0.4692
				1.0000    5.0987   -0.4275
				1.0000    5.2701   -0.3941
				1.0000    5.4104   -0.3357
				1.0000    5.5818   -0.2606
				1.0000    5.6909   -0.2022
				1.0000    5.8623   -0.1188
				1.0000    5.9870   -0.0520
				1.0000    6.1740   -0.0103
				1.0000    6.3766   -0.0103
				1.0000    6.5948    0.0064
				1.0000    6.7818    0.0064
				1.0000    6.9065    0.0064
				1.0000    6.9844    0.0148
				2.0000   -0.4376   -0.0083
				2.0000   -0.2817   -0.0167
				2.0000   -0.0946   -0.0167
				2.0000    0.0769   -0.0083
				2.0000    0.2796    0.0083
				2.0000    0.4667   -0.0083
				2.0000    0.6850   -0.0083
				2.0000    0.8410   -0.0167
				2.0000    0.9501   -0.0250
				2.0000    1.1060   -0.0417
				2.0000    1.2931   -0.0750
				2.0000    1.4335   -0.0917
				2.0000    1.6518   -0.1000
				2.0000    1.9012   -0.1000
				2.0000    2.0728   -0.0917
				2.0000    2.2443   -0.0667
				2.0000    2.4314   -0.0417
				2.0000    2.5873   -0.0333
				2.0000    2.7744   -0.0333
				2.0000    2.9459   -0.0333
				2.0000    3.2266   -0.0167
				2.0000    3.4605   -0.0250
				2.0000    3.6788   -0.0167
				2.0000    3.8503   -0.0167
				2.0000    4.0218   -0.0250
				2.0000    4.2557   -0.0083
				2.0000    4.3493   -0.0417
				2.0000    4.4740   -0.1333
				2.0000    4.5208   -0.2833
				2.0000    4.5676   -0.4500
				2.0000    4.6143   -0.5750
				2.0000    4.7391   -0.6417
				2.0000    4.9262   -0.5833
				2.0000    5.1133   -0.5250
				2.0000    5.2848   -0.4917
				2.0000    5.4719   -0.4417
				2.0000    5.6279   -0.3583
				2.0000    5.7526   -0.2500
				2.0000    5.9085   -0.1500
				2.0000    6.0644   -0.0417
				2.0000    6.2204    0.0083
				2.0000    6.4699    0.0083
				2.0000    6.7505   -0.0083
				2.0000    6.9376   -0.0083
				3.0000   -0.4453    0.0077
				3.0000   -0.3203   -0.0007
				3.0000   -0.1797    0.0161
				3.0000   -0.0078   -0.0007
				3.0000    0.1953   -0.0007
				3.0000    0.4297   -0.0259
				3.0000    0.5859   -0.0510
				3.0000    0.6953   -0.1350
				3.0000    0.7891   -0.2105
				3.0000    0.8359   -0.2944
				3.0000    0.9453   -0.4119
				3.0000    1.0547   -0.4958
				3.0000    1.1953   -0.5042
				3.0000    1.3672   -0.4371
				3.0000    1.5703   -0.3448
				3.0000    1.8047   -0.3112
				3.0000    1.9922   -0.2692
				3.0000    2.2266   -0.2273
				3.0000    2.3672   -0.1853
				3.0000    2.6484   -0.1853
				3.0000    2.8516   -0.1685
				3.0000    3.0859   -0.1098
				3.0000    3.3984   -0.0846
				3.0000    3.6328   -0.0594
				3.0000    4.0078   -0.0091
				3.0000    4.3203   -0.0091
				3.0000    4.5234   -0.0678
				3.0000    4.6328   -0.1350
				3.0000    4.7266   -0.2608
				3.0000    4.8203   -0.3448
				3.0000    4.9453   -0.4371
				3.0000    5.1328   -0.4706
				3.0000    5.3672   -0.4371
				3.0000    5.5078   -0.3448
				3.0000    5.6484   -0.2357
				3.0000    5.8203   -0.1350
				3.0000    5.9297   -0.0930
				3.0000    6.1328   -0.0259
				3.0000    6.3516   -0.0091
				3.0000    6.6172   -0.0175
				3.0000    6.7344   -0.0091
				3.0000    6.8594   -0.0091
				3.0000    6.9375   -0.0007
				4.0000   -0.4531    0.0097
				4.0000   -0.3281   -0.0070
				4.0000   -0.0469    0.0014
				4.0000    0.1719   -0.0153
				4.0000    0.3437   -0.0403
				4.0000    0.5000   -0.0904
				4.0000    0.5937   -0.2741
				4.0000    0.6406   -0.3659
				4.0000    0.7344   -0.4744
				4.0000    0.8281   -0.5245
				4.0000    0.9531   -0.5746
				4.0000    1.1250   -0.5412
				4.0000    1.2187   -0.4744
				4.0000    1.4219   -0.4661
				4.0000    1.5469   -0.4410
				4.0000    1.7031   -0.4160
				4.0000    1.8750   -0.3993
				4.0000    2.0625   -0.3743
				4.0000    2.2812   -0.3409
				4.0000    2.4844   -0.3075
				4.0000    2.7188   -0.2908
				4.0000    2.9844   -0.2490
				4.0000    3.2187   -0.2157
				4.0000    3.4844   -0.2073
				4.0000    3.7344   -0.1739
				4.0000    4.0000   -0.1739
				4.0000    4.1875   -0.1739
				4.0000    4.3594   -0.2073
				4.0000    4.6250   -0.2490
				4.0000    4.7500   -0.2824
				4.0000    4.9062   -0.3158
				4.0000    5.0938   -0.3409
				4.0000    5.2656   -0.3075
				4.0000    5.4219   -0.2407
				4.0000    5.5937   -0.1405
				4.0000    5.7188   -0.0403
				4.0000    5.9219   -0.0070
				4.0000    6.1406   -0.0070
				4.0000    6.3750    0.0014
				4.0000    6.6250    0.0181
				4.0000    6.7969    0.0014
				4.0000    6.9375    0.0097];

			PULSE.ton = [ 0.2000    4.2000
				      0.2000    4.2000
				      0.2000    4.2000
				      0.2000    4.2000];
			PULSE.toff = [4.2000    5.2000
				      4.2000    5.2000
				      4.2000    5.2000
				      4.2000    5.2000];

			PULSE.conc = [0    20
				      2    20
				      5    20
				     10    20];

			PARAMS_HAT = struct('Sigma',0.0569,...
					'cap',0.0039,...
					'cc1lin',0.7750,...
					'cc2',26.3950,...
					'ck1lin',8.5342,...
					'ck2',0.3069,...
					'clmax',0.9397,...
					'cnmax',0.9663,...
					'cx1lin',1.2307,...
					'cx2',10.9297,...
					'ef',2.7583,...
					'gl',4.9195,...
					'hmc1',1.4829,...
					'hmc2',2.7678,...
					'inf',1.7619,...
					'inhmax',3.5697,...
					'k1',0.1143,...
					'k2lin',12.9344,...
					'kI',10.0453,...
					'kinh',1.0018,...
					'kinhcng',0.5181,...
					'n1',3.1844,...
					'n2',3.1128,...
					'nI',1.9848,...
					'ninh',1.3081,...
					'ninhcng',1.4511,...
					'pd',7.5749,...
					'r1',3.1663,...
					'r2',6.5597,...
					'smax',45.5118,...
					'vcl',-7.7902,...
					'vcng',0.0106,...
					'vl',-44.0413);

			GRAPHICS_LAYOUT = 'concentrations_separate_panels';
		case 'prolonged'      


			  PARAMS_HAT = struct('Sigma',6.4270,...
				  'cap',0.0534,...
				  'cc1lin',11.4109,...
				  'cc2',10.5600,...
				  'ck1lin',0.7781,...
				  'ck2',0.3430,...
				  'clmax',2.3131,...
				  'cnmax',1.1266,...
				  'cx1lin',0.8088,...
				  'cx2',9.2763,...
				  'ef',1.5789,...
				  'gl',0.8056,...
				  'hmc1',10.5250,...
				  'hmc2',19.4752,...
				  'inf',1.0162,...
				  'inhmax',1.5105,...
				  'k1',0.0011,...
				  'k2lin',12.1312,...
				  'kI',1.8815,...
				  'kinh',1.0074,...
				  'kinhcng',2.3202,...
				  'n1',2.9406,...
				  'n2',1.2006,...
				  'nI',1.5708,...
				  'ninh',4.8356,...
				  'ninhcng',2.8548,...
				  'pd',0.6752,...
				  'r1',1.1133,...
				  'r2',5.2182,...
				  'smax',189.0821,...
				  'vcl',-0.0246,...
				  'vcng',-2.9351e-07,...
				  'vl',-78.8361);


			DATA = [1.0000   -4.9990    0.4530
				1.0000   -4.6710   -0.1455
				1.0000   -4.3440   -0.3509
				1.0000   -4.0160   -0.0496
				1.0000   -3.6880    0.0740
				1.0000   -3.3610   -0.4444
				1.0000   -3.0330   -0.1464
				1.0000   -2.7050   -1.0491
				1.0000   -2.3780   -0.6705
				1.0000   -2.0500   -0.7812
				1.0000   -1.7220   -0.6895
				1.0000   -1.3950   -1.2455
				1.0000   -1.0670   -1.0582
				1.0000   -0.7390   -1.8863
				1.0000   -0.4110   -1.9660
				1.0000   -0.0840   -1.9925
				1.0000    0.2440   -2.1175
				1.0000    0.5720  -44.3800
				1.0000    0.8990  -37.6736
				1.0000    1.2270  -30.9735
				1.0000    1.5550  -33.3799
				1.0000    1.8820  -26.1836
				1.0000    2.2100  -22.7930
				1.0000    2.5380  -21.3012
				1.0000    2.8650  -16.8166
				1.0000    3.1930  -12.0327
				1.0000    3.5210  -13.7394
				1.0000    3.8480  -13.1912
				1.0000    4.1760  -12.7852
				1.0000    4.5040  -12.3999
				1.0000    4.8310  -10.5164
				1.0000    5.1590  -10.0684
				1.0000    5.4870   -8.6712
				1.0000    5.8140   -8.1445
				1.0000    6.1420   -7.4152
				1.0000    6.4700   -7.2437
				1.0000    6.7970   -6.9193
				1.0000    7.1250   -6.1111
				1.0000    7.4530   -3.3880
				1.0000    7.7810   -2.5567
				1.0000    8.1080   -2.9389
				1.0000    8.4360   -2.5598
				1.0000    8.7640   -2.5172
				1.0000    9.0910   -2.5404
				1.0000    9.4190   -4.1932
				1.0000    9.7470   -9.9604
				1.0000   10.0740  -10.8918
				1.0000   10.4020  -17.2718
				1.0000   10.7300  -15.1820
				1.0000   11.0570  -14.0632
				1.0000   11.3850  -10.6646
				1.0000   11.7130  -10.6098
				1.0000   12.0400   -8.4265
				1.0000   12.3680   -6.8054
				1.0000   12.6960   -6.3270
				1.0000   13.0230   -4.2781
				1.0000   13.3510   -3.7554
				1.0000   13.6790   -2.7984
				1.0000   14.0060   -2.5980
				1.0000   14.3340   -2.2691
				1.0000   14.6620   -1.6528
				1.0000   14.9890   -1.7570
				1.0000   15.3170   -1.9177
				1.0000   15.6450   -1.9160
				1.0000   15.9730   -2.5176
				1.0000   16.3000   -8.1761
				1.0000   16.6280   -8.7194
				1.0000   16.9560  -11.4163
				1.0000   17.2830  -12.3020
				1.0000   17.6110  -14.1066
				1.0000   17.9390   -7.5940
				1.0000   18.2660   -6.6506
				1.0000   18.5940   -6.0592
				1.0000   18.9220   -3.5527
				1.0000   19.2490   -3.8911
				1.0000   19.5770   -3.5192
				1.0000   19.9050   -3.6055
				1.0000   20.2320   -2.9052
				1.0000   20.5600   -2.0164
				1.0000   20.8880   -1.9450
				1.0000   21.2150   -2.0987
				1.0000   21.5430   -1.8765
				1.0000   21.8710   -1.9966
				1.0000   22.1980   -1.9256
				1.0000   22.5260   -2.4254
				1.0000   22.8540   -6.6935
				1.0000   23.1810   -6.0385
				1.0000   23.5090   -6.8149
				1.0000   23.8370  -11.3721
				1.0000   24.1650  -10.3725
				1.0000   24.4920  -10.3007
				1.0000   24.8200   -8.2817
				1.0000   25.1480   -6.4648
				1.0000   25.4750   -6.2523
				1.0000   25.8030   -5.1327
				1.0000   26.1310   -3.9571
				1.0000   26.4580   -2.9111
				1.0000   26.7860   -1.8798
				1.0000   27.1140   -2.1751
				1.0000   27.4410   -1.8441
				1.0000   27.7690   -2.2837
				1.0000   28.0970   -2.1653
				1.0000   28.4240   -2.7431
				1.0000   28.7520   -3.3991
				1.0000   29.0800   -4.8679
				1.0000   29.4070   -5.4498
				1.0000   29.7350   -8.9626
				1.0000   30.0630  -10.1289
				1.0000   30.3900  -10.0789
				1.0000   30.7180   -8.4812
				1.0000   31.0460   -5.2801
				1.0000   31.3730   -6.6322
				1.0000   31.7010   -3.8467
				1.0000   32.0290   -5.0796
				1.0000   32.3570   -4.0452
				1.0000   32.6840   -3.5611
				1.0000   33.0120   -2.4896
				1.0000   33.3400   -2.4707
				1.0000   33.6670   -1.8875
				1.0000   33.9950   -2.8661
				1.0000   34.3230   -1.6875
				1.0000   34.6500   -2.1373
				1.0000   34.9780   -1.9448
				1.0000   35.3060   -2.1893
				1.0000   35.6330   -5.3411
				1.0000   35.9610  -10.2194
				1.0000   36.2890   -6.8726
				1.0000   36.6160  -10.8258
				1.0000   36.9440   -9.1356
				1.0000   37.2720   -6.8583
				1.0000   37.5990   -6.7894
				1.0000   37.9270   -4.2515
				1.0000   38.2550   -5.7219
				1.0000   38.5820   -4.3084
				1.0000   38.9100   -4.5639
				1.0000   39.2380   -4.5713
				1.0000   39.5650   -3.2159
				1.0000   39.8930   -2.9417
				1.0000   40.2210   -2.4483
				1.0000   40.5490   -2.7586
				1.0000   40.8760   -2.2966
				1.0000   41.2040   -2.6566
				1.0000   41.5320   -2.7024
				1.0000   41.8590   -2.3186
				1.0000   42.1870   -2.2603
				1.0000   42.5150   -5.0955
				1.0000   42.8420   -5.8584
				1.0000   43.1700   -6.1116
				1.0000   43.4980   -9.2250
				1.0000   43.8250   -7.8345
				1.0000   44.1530   -9.3422
				1.0000   44.4810   -7.6250
				1.0000   44.8080   -6.5974
				1.0000   45.1360   -4.7585
				1.0000   45.4640   -5.1285
				1.0000   45.7910   -3.6695
				1.0000   46.1190   -3.2349
				1.0000   46.4470   -2.4706
				1.0000   46.7740   -1.2778
				1.0000   47.1020   -0.8136
				1.0000   47.4300   -1.0630
				1.0000   47.7570   -1.3415
				1.0000   48.0850   -1.4269
				1.0000   48.4130   -0.3761
				1.0000   48.7410    0.3496
				1.0000   49.0680   -0.7743
				1.0000   49.3960   -1.8605
				1.0000   49.7240   -3.1111
				1.0000   50.0510   -3.3904
				1.0000   50.3790   -9.5082
				1.0000   50.7070   -8.3069
				1.0000   51.0340   -7.4143
				1.0000   51.3620   -3.6843
				1.0000   51.6900   -4.2977
				1.0000   52.0170   -1.1413
				1.0000   52.3450   -1.0460
				1.0000   52.6730   -0.5385
				1.0000   53.0000   -0.4032
				1.0000   53.3280   -0.0462
				1.0000   53.6560    0.5323
				1.0000   53.9830   -0.1184
				1.0000   54.3110    0.1571
				1.0000   54.6390    0.1560
				1.0000   54.9660    0.0270
				1.0000   55.2940   -0.1283
				1.0000   55.6220   -0.0764
				1.0000   55.9490   -1.6948
				1.0000   56.2770   -1.6406
				1.0000   56.6050   -4.1734
				1.0000   56.9330   -6.3281
				1.0000   57.2600   -8.8637
				1.0000   57.5880   -5.4399
				1.0000   57.9160   -6.3023
				1.0000   58.2430   -5.0151
				1.0000   58.5710   -4.1537
				1.0000   58.8990   -2.1203
				1.0000   59.2260   -1.5716
				1.0000   59.5540   -0.4763
				1.0000   59.8820   -0.5896
				1.0000   60.2090    0.1074];

			PULSE = struct('ton',0,'toff',60,'conc',100);

			GRAPHICS_LAYOUT = 'all_same_panel';
		otherwise
			error('Unknown dataset.');
		end

		PARAMS_HAT = params_gui(PARAMS_HAT);
		PULSE = pulse_gui(PULSE);


			ODEOPTS = odeset('JPattern','on','MaxStep',0.9);

			N = size(PULSE.ton,1);

			init_bLR   = 1.e-8;
			init_aG    = 1.e-8;
			init_cAMP  = 1.e-8;
			init_Ca    = 1.e-8; 
			init_CAMK = 1.e-8;
			init_CaCAM = 1.e-8;
			init_IX = 1.e-8;
			init_V = -70;

			yinit = {init_bLR, init_aG, init_cAMP, init_Ca,init_CaCAM,init_CAMK,init_IX,init_V};
			FN = {'bLR','aG','cAMP','Ca','CaCaM','aCaMK','IX','V'};
			yinit = cell2struct(yinit,FN,2);

			var_names = fieldnames(yinit);
			init_vals = struct2cell(yinit);
			init_vals = [init_vals{:}];
			init_vals = repmat(init_vals(:)',N,1);
			init_vals = init_vals(:)';		

			NVAR = length(var_names);
			NCURVE = length(unique(DATA(:,1)));
			NEQ = NVAR*NCURVE;
			JP = spdiags(ones(NEQ,2*NVAR-1),[-(NEQ-NCURVE):NCURVE:(NEQ-NCURVE)],NEQ,NEQ);


			tspan = unique(DATA(:,2));
			STARTING_TIME = clock;

			[T,Y,msg] = modeleval_od('dwy_pnas_demo2',ODEOPTS,tspan,init_vals,PARAMS_HAT,JP,PULSE,N);

			PRED = [];

			for j = 1:length(var_names)
				PRED.(var_names{j}) = Y(:,((j-1)*N+1):(j*N));
			end

			[PRED_CURRENT,ICNG,ICACL,INCX,IL] = pred_current_od_new(PARAMS_HAT,PRED);
			%if strcmp(NAMES{s},'adaptation')
				PRED_CURRENT = PRED_CURRENT/NORMALIZING_FACTOR;
				ICNG = ICNG/NORMALIZING_FACTOR;
				ICACL = ICACL/NORMALIZING_FACTOR;
				INCX = INCX/NORMALIZING_FACTOR;
				IL = IL/NORMALIZING_FACTOR;	
			%end
			%Don't want to see these currents.
			%INCX = [];
			IL = [];

			UKEY = unique(DATA(:,1));
			var_names = fieldnames(PRED);


			switch lower(GRAPHICS_LAYOUT)
			case 'currents_separate_figures'


				F(1) = figure;


				% if ~isempty(ICACL)
% 					F(2) = figure;
% 				end
% 
% 				if ~isempty(ICNG)
% 					F(3) = figure;
% 				end
% 
% 
% 				if ~isempty(IL)
% 					F(4) = figure;
% 				end

				for k = 1:length(UKEY)
					ind   = find(DATA(:,1) == UKEY(k));
					figure(F(1));
					hold on;
					hdata = plot(DATA(ind,2),DATA(ind,3),['k-'],'LineWidth',2);
					ylabel('Current (pA)','Fontsize',18);
				end	
					htotal = plot(T,real(PRED_CURRENT),'-','LineWidth',2);
					axis tight;
					AX = axis;
					AX(3) = AX(3)-0.1*abs(AX(4)-AX(3));
					AX(4) = AX(4)+0.1*abs(AX(4)-AX(3));

					hold off;

					% if ~isempty(ICNG)
% 						figure(F(3));
% 						hold on;
% 						hcng  = plot(T,real(ICNG),'-','LineWidth',2);
% 						ylabel('CNG Current (pA)','Fontsize',16');
% 						hold off;
% 					end
% 
% 					if ~isempty(ICACL)
% 						figure(F(2));
% 						hold on;
% 						hcacl  = plot(T,real(ICACL),'-','LineWidth',2);
% 						ylabel('Cl(Ca) Current (pA)','Fontsize',16');
% 						hold off;
% 					end
% 
% 					if ~isempty(IL)
% 						figure(F(4));
% 						hold on;
% 						hleak = plot(T,IL,'-','LineWidth',2);
% 						ylabel('Leak Current (pA)','Fontsize',16');
% 						hold off;
% 					end

				%end


				for k = 1:length(F)
					figure(F(k));
					set(gca,'FontSize',16);
					%ylabel('Current (pA)','Fontsize',16');			
					xlabel('Time (s)','Fontsize',18');
					axis(AX);
				end

				%Put in odorant trace.
				figure(F(1));

				AX1 = gca;
				set(AX1,'Units','normalized');
				AX2 = axes;
				set(AX2,'Units','normalized');

				POS1 = get(AX1,'Position');
				POS2 = get(AX2,'Position');
				fac = 0.05;
				set(AX2,'Position',[POS1(1), POS1(2)+POS1(4)+fac, POS1(3) 0.05]);
				TT = linspace(min(T),max(T),100);
				OD = pulse_train(TT,PULSE.ton,PULSE.toff,PULSE.conc);
				UOD = unique(OD,'rows');
				OD = OD(end,:);
				for PT = 1:size(UOD,1)
					plot(TT,UOD(PT,:),'k-','LineWidth',2);
					hold on;
				end
				hold off;
				this_ax = axis;
				this_ax(4) = this_ax(4) + 0.2*abs(this_ax(4)-this_ax(3));
				axis(this_ax);
				ax2 = axis;
				axes(AX1);
				ax1 = axis;
				axes(AX2);
				axis([ax1(1) ax1(2) ax2(3) ax2(4)]);
				axis off;
				conc = sprintf('%.4g, ',unique(PULSE.conc));
				conc = conc(1:end-2);
				text(TT(1),1.5*max(OD(:)),['Odorant ',conc,'\mu{M}'],'FontSize',16);
				box off;
				hold off;
				for k = 1:length(F)
					figure(F(k));		
					%suptitle(NAMES{s});
                                        suptitle(' ');
					switch k
					case 1
						%hgsave_alt(fullfile(PROJECTS.(NAMES{i}).graphicsdir,[NAMES{i},'_cur_total_a']));
					case 2
						%hgsave_alt(fullfile(PROJECTS.(NAMES{i}).graphicsdir,[NAMES{i},'_cur_cacl_a']));
					case 3
						%hgsave_alt(fullfile(PROJECTS.(NAMES{i}).graphicsdir,[NAMES{i},'_cur_cng_a']));
					case 4
						%hgsave_alt(fullfile(PROJECTS.(NAMES{i}).graphicsdir,[NAMES{i},'_cur_leak_a']));
					otherwise
						error('Unexpected number of figures.');
					end
				end
                                
                                axes(AX1);
				AXLIM = axis;
                                
                                F2 = figure;
				%nrows = ceil(sqrt(length(var_names)));
				%mcols = ceil(length(var_names)/nrows);
                                nrows = ceil(sqrt(9));
				mcols = ceil(9/nrows);
                                
                                
                                ww = 1;
                                for k = 1:length(var_names)
					if (strcmp(var_names{k},'bLR'))
                                                %Don't do anything.
                                        else
                                                hax = subplot(nrows,mcols,ww);% 
					        plot(T,real(PRED.(var_names{k})));
					        hl = ylabel(var_names{k});

					        box off; 
					        set(gca,'FontSize',16);
					        set(get(gca,'XLabel'),'FontSize',16);
					        set(get(gca,'YLabel'),'FontSize',16);
					        h = findobj(gcf,'Type','line');
					        set(h,'LineWidth',2);
					        axis tight;
					        AX = axis;
					        fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					        AX(3) = AX(3)-fac;
					        AX(4) = AX(4)+fac;
					        axis(AX);
					        if (ww ~= 8)
						        set(gca,'XTickLabel',[]);
					        else 
						        xlabel('Time (s)','Fontsize',18');
					        end
                                                ww = ww + 1;
                                        end
				end
                                
                                if ~isempty(ICNG)
                                        hax = subplot(nrows,mcols,ww);                                              
			                hcng  = plot(T,real(ICNG),'-','LineWidth',2);          
			                hl = ylabel('CNG Curr. (pA)','Fontsize',16');
                                        box off; 
					set(gca,'FontSize',16);
				        set(get(gca,'XLabel'),'FontSize',16);
					set(get(gca,'YLabel'),'FontSize',16);
                                        h = findobj(gcf,'Type','line');
					set(h,'LineWidth',2);
					axis tight;
					AX = axis;
					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					AX(3) = AX(3)-fac;
					AX(4) = AX(4)+fac;
					axis(AX);
                                        if (ww ~= 8)
						set(gca,'XTickLabel',[]);
					else 
						xlabel('Time (s)','Fontsize',18');
					end
					           
			                axis(AXLIM); 
                                        ww = ww + 1;                                              
			        end                                                            

			        if ~isempty(ICACL)  
                                        hax = subplot(nrows,mcols,ww);
                                        hcacl  = plot(T,real(ICACL),'-','LineWidth',2);                                              
			                hl = ylabel('Cl(Ca) Curr. (pA)');       
			               
                                        box off; 
					set(gca,'FontSize',16);
				        set(get(gca,'XLabel'),'FontSize',16);
					set(get(gca,'YLabel'),'FontSize',16);
                                        h = findobj(gcf,'Type','line');
					set(h,'LineWidth',2);
					axis tight;
					AX = axis;
					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					AX(3) = AX(3)-fac;
					AX(4) = AX(4)+fac;
					axis(AX);
                                        if (ww ~= 8)
						set(gca,'XTickLabel',[]);
					else 
						xlabel('Time (s)','Fontsize',18');
					end
					           
			                axis(AXLIM); 
                                        ww = ww + 1;  

                                           
			                                                              
			        end                                                            
                                
                               
                                
				% for k = 1:length(var_names)
% 					hax = subplot(nrows,mcols,k);% 
% 					plot(T,real(PRED.(var_names{k})));
% 					hl = ylabel(var_names{k});
% 
% 					box off; 
% 					set(gca,'FontSize',16);
% 					set(get(gca,'XLabel'),'FontSize',16);
% 					set(get(gca,'YLabel'),'FontSize',16);
% 					h = findobj(gcf,'Type','line');
% 					set(h,'LineWidth',2);
% 					axis tight;
% 					AX = axis;
% 					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
% 					AX(3) = AX(3)-fac;
% 					AX(4) = AX(4)+fac;
% 					axis(AX);
% 					if (k ~= length(var_names))
% 						%set(gca,'XTickLabel',[],'FontSize',16,'XColor',[1 1 1]);
% 					else
% 						xlabel('Time (s)','Fontsize',16');
% 					end
% 				end
				subplotspace('h',15);
				%suptitle(NAMES{s});


			case 'concentrations_separate_panels'
				F1 = figure;

				SPACE = 0.05;
				SP_WD = 0.8;
				SP_HT = 1/(length(UKEY) + 0.5) - 0.5*SPACE; 	

				OD_SP_WD = SP_WD;
				OD_SP_HT = 0.5*SP_HT;

				AX(1) = axes;%Odorant trace axis.
				set(AX(1),'Units','normalized','Position',[0.1, length(UKEY)*SP_HT+3*SPACE, OD_SP_WD, OD_SP_HT]);
				for k = 2:length(UKEY)+1
					AX(k) = axes;
					set(AX(k),'Units','normalized','Position',[0.1, (1+length(UKEY)-k)*SP_HT+3*SPACE, SP_WD, SP_HT]);
				end

                                RGB = {[0 0 1],[0 0.5 0],[1 0 0],[0 0.75 0.75]};
                                
                                AXLIM = [inf,-inf,inf,-inf];
				for k = 1:length(UKEY)
					%subplot(length(UKEY)+1,1,k+1);
					axes(AX(k+1));
					ind   = find(DATA(:,1) == UKEY(k));
					hdata = plot(DATA(ind,2),DATA(ind,3),['k-'],'LineWidth',2);
					hold on;
					htotal = plot(T,real(PRED_CURRENT(:,k)),'-','LineWidth',2,'Color',RGB{k});

					% if ~isempty(ICNG)
% 						hcng  = plot(T,real(ICNG(:,k)),'c--','LineWidth',2);
% 					end
% 
% 					if ~isempty(ICACL)
% 						hcacl  = plot(T,real(ICACL(:,k)),'m--','LineWidth',2);
% 					end
% 
% 					if ~isempty(IL)
% 						hleak = plot(T,IL(:,k),'g--','LineWidth',2);
% 					end
					hold off;
					set(gca,'FontSize',16)
					if (k ~= length(UKEY))
						set(gca,'XTickLabel',[]);
					end
					axis tight;
                                        axlim = axis;
                                        AXLIM(1) = min(AXLIM(1),axlim(1));
                                        AXLIM(2) = max(AXLIM(2),axlim(2)); 
                                        AXLIM(3) = min(AXLIM(3),axlim(3));
                                        AXLIM(4) = max(AXLIM(4),axlim(4));
                                        box off;
                                         
				end
                                for k = 1:length(UKEY)
                                        axes(AX(k+1));
                                        axis(AXLIM);
                                end
				
                                
                                axes(AX(max(2,floor(length(UKEY)/2)+1)));
				ylabel('Normalized Current','Fontsize',18');
				axes(AX(end));			
				xlabel('Time (s)','Fontsize',18');
                                

				axes(AX(1));
				%Put in odorant trace.
				hold on;
				TT = linspace(min(T),max(T),100);
				OD = pulse_train(TT,PULSE.ton,PULSE.toff,PULSE.conc);
				OD = OD(end,:);
				plot(TT,OD,'k-','LineWidth',2);
				this_ax = axis;
				this_ax(4) = this_ax(4) + 0.2*abs(this_ax(4)-this_ax(3));
				axis(this_ax);
 				axis off;
				conc = sprintf('%.4g, ',unique(PULSE.conc));
				conc = conc(1:end-2);
				text(TT(1),1.5*max(OD(:)),['Odorant ',conc,'\mu{M}'],'FontSize',16);
				box off;
				hold off;
				set(gca,'Xlim',get(AX(2),'XLim'));
                                
                                
				%suptitle(NAMES{s});
                                suptitle(' ');

				F2 = figure;
                                
                                
                                nrows = ceil(sqrt(9));
				mcols = ceil(9/nrows);
                                
                                
                                ww = 1;
                                for k = 1:length(var_names)
					if (strcmp(var_names{k},'bLR'))
                                                %Don't do anything.
                                        else
                                                hax = subplot(nrows,mcols,ww);% 
					        plot(T,real(PRED.(var_names{k})));
					        hl = ylabel(var_names{k});

					        box off; 
					        set(gca,'FontSize',16);
					        set(get(gca,'XLabel'),'FontSize',16);
					        set(get(gca,'YLabel'),'FontSize',16);
					        h = findobj(gcf,'Type','line');
					        set(h,'LineWidth',2);
					        axis tight;
					        AX = axis;
					        fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					        AX(3) = AX(3)-fac;
					        AX(4) = AX(4)+fac;
					        axis(AX);
					        if (ww ~= 8)
						        set(gca,'XTickLabel',[]);
					        else 
						        xlabel('Time (s)','Fontsize',18');
					        end
                                                ww = ww + 1;
                                        end
				end
                                
                                if ~isempty(ICNG)
                                        hax = subplot(nrows,mcols,ww);                                              
			                hcng  = plot(T,real(ICNG),'-','LineWidth',2);          
			                hl = ylabel('CNG Curr. (pA)','Fontsize',16');
                                        box off; 
					set(gca,'FontSize',16);
				        set(get(gca,'XLabel'),'FontSize',16);
					set(get(gca,'YLabel'),'FontSize',16);
                                        h = findobj(gcf,'Type','line');
					set(h,'LineWidth',2);
					axis tight;
					AX = axis;
					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					AX(3) = AX(3)-fac;
					AX(4) = AX(4)+fac;
					axis(AX);
                                        if (ww ~= 8)
						set(gca,'XTickLabel',[]);
					else 
						xlabel('Time (s)','Fontsize',18');
					end
					axis(AXLIM);             
			               
                                        ww = ww + 1;                                              
			        end                                                            

			        if ~isempty(ICACL)  
                                        hax = subplot(nrows,mcols,ww);
                                        hcacl  = plot(T,real(ICACL),'-','LineWidth',2);                                              
			                hl = ylabel('Cl(Ca) Curr. (pA)');       
			               
                                        box off; 
					set(gca,'FontSize',16);
				        set(get(gca,'XLabel'),'FontSize',16);
					set(get(gca,'YLabel'),'FontSize',16);
                                        h = findobj(gcf,'Type','line');
					set(h,'LineWidth',2);
					axis tight;
					AX = axis;
					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					AX(3) = AX(3)-fac;
					AX(4) = AX(4)+fac;
					axis(AX);
                                        if (ww ~= 8)
						set(gca,'XTickLabel',[]);
					else 
						xlabel('Time (s)','Fontsize',18');
					end
					           
			                axis(AXLIM);   
                                        ww = ww + 1;  

                                           
			                                                              
			        end                                                            
                                
                                
                                
                                
                                
				% nrows = ceil(sqrt(length(var_names)));
% 				mcols = ceil(length(var_names)/nrows);
% 
% 				for k = 1:length(var_names)
% 					hax = subplot(nrows,mcols,k);% 
% 					plot(T,real(PRED.(var_names{k})));
% 					hl = ylabel(var_names{k});
% 
% 					box off; 
% 					set(gca,'FontSize',16);
% 					set(get(gca,'XLabel'),'FontSize',16);
% 					set(get(gca,'YLabel'),'FontSize',16);
% 					h = findobj(gcf,'Type','line');
% 					set(h,'LineWidth',2);
% 					axis tight;
% 					AX = axis;
% 					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
% 					AX(3) = AX(3)-fac;
% 					AX(4) = AX(4)+fac;
% 					axis(AX);
% 					if (k ~= length(var_names))
% 						%set(gca,'XTickLabel',[],'FontSize',16,'XColor',[1 1 1]);
% 					else
% 						xlabel('Time (s)','Fontsize',16');
% 					end
% 				end
				subplotspace('h',15);
				%suptitle(NAMES{s});


			case 'all_same_panel'

				F1 = figure;

				SPACE = 0.05;
				SP_WD = 0.8;
				%SP_HT = 1/(1 + 0.5) - 0.5*SPACE; 	
				SP_HT = 0.8; 	

				OD_SP_WD = SP_WD;
				OD_SP_HT = 0.05;

				AX(1) = axes; %Odorant trace axis.
				set(AX(1),'Units','normalized','Position',[0.1, SP_HT+3*SPACE, OD_SP_WD, OD_SP_HT]);


				AX(2) = axes;
				set(AX(2),'Units','normalized','Position',[0.1, 3*SPACE, SP_WD, SP_HT]);


				axes(AX(2));
				for k = 1:length(UKEY)
					ind   = find(DATA(:,1) == UKEY(k));
					hdata = plot(DATA(ind,2),DATA(ind,3),['k-'],'LineWidth',2);
					hold on;
					htotal = plot(T,real(PRED_CURRENT(:,k)),'b-','LineWidth',2);

					% if ~isempty(ICNG)
% 						hcng  = plot(T,real(ICNG(:,k)),'c--','LineWidth',2);
% 					end
% 
% 					if ~isempty(ICACL)
% 						hcacl  = plot(T,real(ICACL(:,k)),'m--','LineWidth',2);
% 					end
% 
% 					if ~isempty(IL)
% 						hleak = plot(T,IL(:,k),'g--','LineWidth',2);
% 					end
                                        box off;
				end
				hold off;
				set(gca,'FontSize',16);
				ylabel('Current (pA)','Fontsize',18');			
				xlabel('Time (s)','Fontsize',18');
				axis tight;
				AXLIM = axis;
				%Give a little space on the vertical axis limits.
				AXLIM(3) = AXLIM(3) - 0.1*abs(AXLIM(4)-AXLIM(3));
				AXLIM(4) = AXLIM(4) + 0.1*abs(AXLIM(4)-AXLIM(3));
				axis(AXLIM);

				%Put in odorant trace.
				axes(AX(1));
				TT = linspace(min(T),max(T),100);
				OD = pulse_train(TT,PULSE.ton,PULSE.toff,PULSE.conc);
				OD = OD(end,:);
				plot(TT,OD,'k-','LineWidth',2);
				hold on;
				this_ax = axis;
				this_ax(4) = this_ax(4) + 0.2*abs(this_ax(4)-this_ax(3));
				axis(this_ax);
				axis off;
				%set(h(1),'XTickLabel',[],'FontSize',get(AX1,'FontSize'),'XColor',[1 1 1]);
				%set(get(gca,'XLabel'),'Color',[0 0 0]);
				conc = sprintf('%.4g, ',unique(PULSE.conc));
				conc = conc(1:end-2);
				text(TT(1),2.5*max(OD(:)),['Odorant ',conc,'\mu{M}'],'FontSize',16);
				box off;
				hold off;
				set(AX(1),'XLim',get(AX(2),'XLim'));
				%POS1(4) = POS1(4);
				%POS1(2) = 1.25*POS1(2);
				%set(AX1,'Position',POS1);

				%suptitle(NAMES{s});
                                suptitle(' ');
				%hgsave_alt(fullfile(PROJECTS.(NAMES{i}).graphicsdir,[NAMES{i},'_cur_a']));
                                F2 = figure;
                                
                                
                                nrows = ceil(sqrt(9));
				mcols = ceil(9/nrows);
                                
                                
                                ww = 1;
                                for k = 1:length(var_names)
					if (strcmp(var_names{k},'bLR'))
                                                %Don't do anything.
                                        else
                                                hax = subplot(nrows,mcols,ww);% 
					        plot(T,real(PRED.(var_names{k})));
					        hl = ylabel(var_names{k});

					        box off; 
					        set(gca,'FontSize',16);
					        set(get(gca,'XLabel'),'FontSize',16);
					        set(get(gca,'YLabel'),'FontSize',16);
					        h = findobj(gcf,'Type','line');
					        set(h,'LineWidth',2);
					        axis tight;
					        AX = axis;
					        fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					        AX(3) = AX(3)-fac;
					        AX(4) = AX(4)+fac;
					        axis(AX);
					        if (ww ~= 8)
						        set(gca,'XTickLabel',[]);
					        else 
						        xlabel('Time (s)','Fontsize',18');
					        end
                                                ww = ww + 1;
                                        end
				end
                                
                                if ~isempty(ICNG)
                                        hax = subplot(nrows,mcols,ww);                                              
			                hcng  = plot(T,real(ICNG),'-','LineWidth',2);          
			                hl = ylabel('CNG Curr. (pA)','Fontsize',16');
                                        box off; 
					set(gca,'FontSize',16,'YLim',[-100 0]);
				        set(get(gca,'XLabel'),'FontSize',16);
					set(get(gca,'YLabel'),'FontSize',16);
                                        h = findobj(gcf,'Type','line');
					set(h,'LineWidth',2);
					axis tight;
					AX = axis;
					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					AX(3) = AX(3)-fac;
					AX(4) = AX(4)+fac;
					axis(AX);
                                        if (ww ~= 8)
						set(gca,'XTickLabel',[]);
					else 
						xlabel('Time (s)','Fontsize',18');
					end
					axis(AXLIM);          
			               
                                        ww = ww + 1;                                              
			        end                                                            

			        if ~isempty(ICACL)  
                                        hax = subplot(nrows,mcols,ww);
                                        hcacl  = plot(T,real(ICACL),'-','LineWidth',2);                                              
			                hl = ylabel('Cl(Ca) Curr. (pA)');       
			               
                                        box off; 
					set(gca,'FontSize',16,'YLim',[-100 0]);
				        set(get(gca,'XLabel'),'FontSize',16);
					set(get(gca,'YLabel'),'FontSize',16);
                                        h = findobj(gcf,'Type','line');
					set(h,'LineWidth',2);
					axis tight;
					AX = axis;
					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
					AX(3) = AX(3)-fac;
					AX(4) = AX(4)+fac;
					axis(AX);
                                        if (ww ~= 8)
						set(gca,'XTickLabel',[]);
					else 
						xlabel('Time (s)','Fontsize',18');
					end
					           
			                axis(AXLIM);
                                        ww = ww + 1;  

                                           
			                                                              
			        end                                                            
                                
% 
% 				F2 = figure;
% 				nrows = ceil(sqrt(length(var_names)));
% 				mcols = ceil(length(var_names)/nrows);
% 
% 				for k = 1:length(var_names)
% 					hax = subplot(nrows,mcols,k);% 
% 					plot(T,real(PRED.(var_names{k})));
% 					hl = ylabel(var_names{k});
% 
% 					box off; 
% 					set(gca,'FontSize',16);
% 					set(get(gca,'XLabel'),'FontSize',16);
% 					set(get(gca,'YLabel'),'FontSize',16);
% 					h = findobj(gcf,'Type','line');
% 					set(h,'LineWidth',2);
% 					axis tight;
% 					AX = axis;
% 					fac = 0.1*(AX(4)-AX(3)); %Give a little space.
% 					AX(3) = AX(3)-fac;
% 					AX(4) = AX(4)+fac;
% 					axis(AX);
% 					if (k ~= length(var_names))
% 						%set(gca,'XTickLabel',[],'FontSize',16,'XColor',[1 1 1]);
% 					else
% 						xlabel('Time (s)','Fontsize',16');
% 					end
% 				end
				subplotspace('h',15);
				%suptitle(NAMES{s});
				%hgsave_alt(fullfile(PROJECTS.(NAMES{i}).graphicsdir,[NAMES{i},'_vars_a']));


			otherwise
				error('Unrecognized graphics layout.');
			end

                ButtonName=questdlg('Would you like to save this configuration?', ...
                       'Save Parameters and Data', ...
                       'Yes','No','Yes');
                       
                if strcmp(ButtonName,'Yes')
                        
                        [filename, pathname] = uiputfile('*.mat', 'Save configuration as...');
                        save(fullfile(pathname,filename),'DATA','PARAMS_HAT','PULSE','GRAPHICS_LAYOUT');
                end       
                
                ButtonName=questdlg('Would you like to export these results?', ...
                       'Save Parameters,Data and Predictions...', ...
                       'Yes','No','Yes');
                       
                       
                if strcmp(ButtonName,'Yes')
                        while 1
                                [pathname] = uigetdir(pwd,'Select export directory...');
                                if isnumeric(pathname)
                                        return;
                                end
                        
                                if (~exist(pathname,'dir'))
                                        ButtonName=questdlg(['Create directory ',pathname,'?'], ...
                                                'Directory does not exist!', ...
                                                'Yes','No','Yes');
                                        if strcmp(ButtonName,'Yes')
                                                break;
                                        end
                                else
                                        ButtonName=questdlg(['Overwrite contents of ',pathname,'?'], ...
                                                'Directory already exists!', ...
                                                'Yes','No','Yes');
                                                
                                        if strcmp(ButtonName,'Yes')
                                                break;
                                        end  
                                end
                        end
                        
                        DATA_FILE       = 'dwy_pnas_data.txt';
                        PARAMS_FILE     = 'dwy_pnas_params.txt';
                        PULSE_FILE      = 'dwy_pnas_pulse.txt';
                        PREDICTION_FILE = 'dwy_pnas_prediction.txt';
                        GRAPHICS_LAYOUT_FILE = 'dwy_pnas_layout.txt';
                        NORMALIZING_FACTOR_FILE = 'dwy_pnas_factor.txt';
                        
                        fid = fopen(fullfile(pathname,DATA_FILE),'w');
                        for i=1:size(DATA,1)
                                fprintf(fid,'%f\t%f\t%f\n',DATA(i,:));
                        end
                        fclose(fid);
                        
                        
                        fid = fopen(fullfile(pathname,PARAMS_FILE),'w');
                        F = fieldnames(PARAMS_HAT);
                        V = struct2cell(PARAMS_HAT);
                        for i=1:length(F)
                                fprintf(fid,'%s\t',F{i});
                                fprintf(fid,'%f\n',V{i});
                        end
                        fclose(fid);
                        
                        
                        
                        fid = fopen(fullfile(pathname,PULSE_FILE),'w');
                        for i=1:size(PULSE.ton,1)
                                fprintf(fid,'ton');
                                fprintf(fid,repmat('\t%f',1,size(PULSE.ton,2)),PULSE.ton(i,:));
                                fprintf(fid,'\n');
                        end
                        
                        
                        for i=1:size(PULSE.toff,1)
                                fprintf(fid,'toff');
                                fprintf(fid,repmat('\t%f',1,size(PULSE.toff,2)),PULSE.toff(i,:));
                                fprintf(fid,'\n');
                        end
                       
                        for i=1:size(PULSE.conc,1)
                                fprintf(fid,'conc');
                                fprintf(fid,repmat('\t%f',1,size(PULSE.conc,2)),PULSE.conc(i,:));
                                fprintf(fid,'\n');
                        end
                        fclose(fid);
                        
                        fid = fopen(fullfile(pathname,PREDICTION_FILE),'w');
                        PREDICTION = cat(2,T,Y,PRED_CURRENT,ICNG,ICACL,INCX,IL);
                        
                        
                        FMT = [repmat('\t%f',1,size(PREDICTION,2)),'\n'];
                        for i=1:size(T,1)
                                fprintf(fid,FMT,PREDICTION(i,:));
                        end
                        fclose(fid);
                        
                        fid = fopen(fullfile(pathname,GRAPHICS_LAYOUT_FILE),'w');
                        fprintf(fid,'%s\n',GRAPHICS_LAYOUT);
                        fclose(fid);
                        
                        fid = fopen(fullfile(pathname,NORMALIZING_FACTOR_FILE),'w');
                        fprintf(fid,'%f\n',NORMALIZING_FACTOR);
                        fclose(fid);
                        
                        
                        
                end       
                
                
	end
otherwise %Take on behavior of the ODE file.

%	function varargout = comb_fdbkcmk_linearv_ix(t,y,flag,PULSE,P,N,JP) 
 	
	t = varargin{1};
	y = varargin{2};
	flag = varargin{3};
	PULSE = varargin{4};
	P = varargin{5};
	N = varargin{6};
	JP = varargin{7};
	%CAMP_FDBK Ode file for the cAMP_FDBK model.
	%    
	%
	persistent ABLE;
	global STARTING_TIME;

	if ~isempty(STARTING_TIME)
		if (etime(clock,STARTING_TIME) > 120)
			error('Maximum execution time elapsed.');
		end
	end


	switch flag
	case ''
		%Define the tot variables to be 1.
		P.Rtot = 1;
		P.Gtot = 1;

		dy = zeros(size(y));

		bLR = y(1:N,1);
		aG = y(N+1:2*N,1);
		cAMP = y(2*N+1:3*N,1);
		Ca = y(3*N+1:4*N,1);
		CaCAM = y(4*N+1:5*N,1);
		CAMK = y(5*N+1:6*N,1);
		IX = y(6*N+1:7*N,1);
		V = y(7*N+1:8*N,1);

		%#####LIGAND-RECEPTOR INTERACTION#####
		k2 = P.k2lin.*bLR;
		%#####TRANSDUCTION####################
		%a1 = P.a1lin.*aG.*(P.ACtot-aAC);
		cc1 = P.cc1lin.*Ca;
		%cx1 = 10*P.ck1lin.*CaCAM.^2; 
		%cx1 = P.cx1lin.*CaCAM; 
		cx1 = P.cx1lin.*Ca;
		%ck1 = ((P.ck1lin*CAMK.^P.ckn)./(P.ckh.^P.ckn + CAMK.^P.ckn)).*CaCAM;
		ck1 = P.ck1lin.*CaCAM;
		%fca = (P.smax)./(1+(CAMK./(P.kinh)).^P.ninh);
		%P.smin = 0;

		%fca = P.smin + (P.smax-P.smin)./(1+(CAMK./(P.kinh)).^P.ninh);
		fca = (P.smax)./(1+(CAMK./(P.kinh)).^P.ninh);
		synth = aG.*fca;
		inhcng = 1+(P.inhmax-1).*CaCAM.^P.ninhcng./(CaCAM.^P.ninhcng + P.kinhcng.^P.ninhcng);	
		Icng = (P.cnmax.*cAMP.^P.n1./(cAMP.^P.n1 + (inhcng.*P.hmc1).^P.n1)).*(P.vcng-V);		
		Icacl = ((P.clmax.*Ca.^P.n2)./(Ca.^P.n2 + P.hmc2.^P.n2)).*(P.vcl-V);
		Il = P.gl.*(P.vl-V);

		Ostim = pulse_train(t,PULSE.ton,PULSE.toff,PULSE.conc);

		%#####ODOR STIMLUATION & LIGAND-RECEPTOR INTERACTION#####
		D_bLR = P.k1*Ostim.*(P.Rtot-bLR) - P.r1.*bLR;
		D_aG = k2.*(P.Gtot-aG) - P.r2.*aG;
		%#####TRANSDUCTION####################
		D_cAMP = synth - P.pd.*cAMP;
		%D_Ca = P.inf.*Icng - (P.vncx-V).*(P.ef./(1 + (IX./P.kI).^P.nck2)).*Ca + 4*(-cc1 + P.cc2.*CaCAM);

		D_Ca = P.inf.*Icng - (P.ef./(1 + (IX./P.kI).^P.nI)).*Ca + (-cc1 + P.cc2.*CaCAM);
		D_CaCAM = cc1 - P.cc2.*CaCAM;
		D_CAMK = ck1 - P.ck2.*CAMK;
		D_IX = cx1 - P.cx2.*IX;  %This has got to go back down in order for oscillations...
		D_V = (1./P.cap).*(Icng + Icacl + Il);


		varargout{1} = [D_bLR;D_aG;D_cAMP;D_Ca;D_CaCAM;D_CAMK;D_IX;D_V];

	case 'jpattern'
		varargout{1} = JP;
	otherwise
		error('Unexpected flag');
	end
end



function varargout = pred_current_od_new(P,PRED)

varargout = cell(nargout,1);
Icng = [];
Icacl = [];
Incx = [];
Il = [];

inhcng = 1+(P.inhmax-1).*PRED.CaCaM.^P.ninhcng./(PRED.CaCaM.^P.ninhcng + P.kinhcng.^P.ninhcng);
%inhcng = 1;
Icng = (P.cnmax.*PRED.cAMP.^P.n1./(PRED.cAMP.^P.n1 + (inhcng.*P.hmc1).^P.n1)).*(P.vcng-PRED.V);
Icacl = (P.clmax.*PRED.Ca.^P.n2./(PRED.Ca.^P.n2 + P.hmc2.^P.n2)).*(P.vcl-PRED.V);
Il = P.gl.*(P.vl-PRED.V);
%Icng = (P.cnmax.*PRED.cAMP.^P.n1).*(P.vcng-PRED.V);
%Icacl = (P.clmax.*PRED.Ca.^P.n2).*(P.vcl-PRED.V);
%Il = P.gl.*(P.vl-PRED.V);

varargout{1} = -Icng - Icacl;
varargout{2} = -Icng;
varargout{3} = -Icacl;
varargout{4} = -Incx;
varargout{5} = -Il;

function [T,Y,msg] = modeleval_od(mfile,ODEOPTS,tspan,yinit,PARAMS_HAT,JP,PULSE,N)

msg = '';

%odefun = PROJECTS.(NAME).odefun;
odefun = 'ode15s';

%Solve for 6 seconds so that we may come to a steady-state from our initial conditions.
tspan = [-6;tspan];


[T,Y] = feval(odefun,mfile,tspan,yinit,ODEOPTS,PULSE,PARAMS_HAT,N,JP);

%Cut off the initial solution point which was just there to get us to steady-state.
T = T(2:end);
Y = Y(2:end,:);


function varargout = params_gui(varargin)
%PARAMS_GUI Graphical interface to edit parameter struct.
%
%


if (nargin == 0)
	error('Parameter struct required as input.');
end

if (nargin > 1 )
	error('Too many inputs.');
end


if ischar(varargin{1})
	FLAG = varargin{1};
elseif isstruct(varargin{1})
	PARAMS_HAT = varargin{1};
	FLAG = 'init';
else
	error('Invalid input.');
end

switch lower(FLAG)
case 'init'
	FN = fieldnames(PARAMS_HAT);
	VALS = struct2cell(PARAMS_HAT);
	VALS = cat(1,VALS{:});
	NF = length(FN);
	
	TEXT_HT = 1.5;
	TEXT_WD = 15;
	
	FIG_WD = 2*TEXT_WD;
	FIG_HT = (NF+2)*TEXT_HT;
	
	F = figure('Name','Edit parameter values...','Units','characters','Position',[10 10 FIG_WD FIG_HT]);
	
	POS = [0 FIG_HT-TEXT_HT TEXT_WD TEXT_HT];
	h = uicontrol('Style','text','units','characters');
	set(h,'Position',POS,'String','Parameter');
	
	POS(1) = POS(1) + TEXT_WD;
	h = uicontrol('Style','text','units','characters');
	set(h,'Position',POS,'String','Value');
	
	POS(1) = POS(1) - TEXT_WD;
	POS(2) = POS(2) - TEXT_HT;
	for i = 1:NF
		h = uicontrol('Style','text','units','characters','Tag','text_params');
		set(h,'Position',POS,'String',FN{i});
		POS(1) = POS(1) + TEXT_WD;
		h = uicontrol('Style','edit','units','characters','Tag','edit_params');
		set(h,'Position',POS,'String',num2str(PARAMS_HAT.(FN{i})));
		
		POS(1) = POS(1) - TEXT_WD;
		POS(2) = POS(2) - TEXT_HT;
	end
	
	h = uicontrol('Style','push','units','characters','Tag','push_ok','UserData',PARAMS_HAT);
	set(h,'Position',POS,'String','Ok','Callback','params_gui(''OK'')');
	POS(1) = POS(1) + TEXT_WD;
	h = uicontrol('Style','push','units','characters','Tag','push_cancel');
	set(h,'Position',POS,'String','Cancel','Callback','params_gui(''Cancel'')');
	
	set(gcf,'UserData',1);
	H = findobj(gcf,'Type','uicontrol');
	set(H,'Units','normalized');
	
	waitfor(gcf,'UserData');
	F = gcf;
	H = findobj(F,'Tag','push_ok');
	varargout{1} = get(H,'UserData');
	
	delete(F);
	
	
case 'ok'

	H = findobj(gcf,'Tag','text_params');
	H = flipud(H);
	F = get(H,'String');
	H = findobj(gcf,'Tag','edit_params');
	H = flipud(H);
	NF = length(F);
	
	
	for i = 1:NF
		PARAMS_HAT.(F{i}) = str2double(get(H(i),'String'));
	end
	H = findobj(gcf,'Tag','push_ok');
	set(H,'UserData',PARAMS_HAT);
	set(gcf,'UserData',0);
	
case 'cancel'
	set(gcf,'UserData',0);	
	
otherwise
	error('Unexpected FLAG.');
end

function OUT = pulse_train(tnow,ton,toff,val,varargin)
%PULSE_TRAIN Generate a train of pulses.
%   V = PULSE_TRAIN(T,TON,TOFF,VAL) returns a pulse train
%   having pulses beginning at times TON and ending at times
%   TOFF with peak values equal to VAL.  Pulses may overlap.
% 
%   PULSE_TRAIN(T,TON,TOFF,VAL,SHARP) uses sharpness values
%   (default = 0.001) for the pulses.  Larger values make 
%   smoother pulse trains.  
%   
%   TON, TOFF, VAL and SHARP must be matrices of equal size.  
%   Each row of these matrices corresponds to a separate train.
%
%   Note:  To simulate pulse trains having different number
%   of pulses simply set the corresponding values in VAL to 
%   zero. 
%   
%   Example: Generate 2 pulse trains each with different 
%      characteristics.
%
%      t = linspace(0,10,100);
%      ton = [0.2 3 1;
%             0.2 5 6]
%      toff = [1.2 3.5 1;
%              2.2 5.5 6.5];
%      val = [1 2 0;
%             4 5 6];  
%
%      sharp = [0.001 0.01 0;
%               0.1 0.1 0.01];
%
%      T = pulse_train(t,ton,toff,val);
%      plot(t,T);
%
%

tnow = tnow(:);

if nargin == 5
	SHARPNESS = varargin{1};
else
	SHARPNESS = 0.001; 
end

NTP = length(tnow);
SIZ = size(ton);

if (NTP > 1)
	ton = repmat(ton,[1 1 length(tnow)]);
	toff = repmat(toff,[1 1 length(tnow)]);
	val = repmat(val,[1 1 length(tnow)]);
	tnow = repmat(reshape(tnow,[1 1 NTP]),[SIZ,1]);
	OUT = permute(sum(val.*(hv(tnow-ton,SHARPNESS)- hv(tnow-toff,SHARPNESS)),2),[1 3 2]);
 
else
	OUT = sum(val.*(hv(tnow-ton,SHARPNESS)- hv(tnow-toff,SHARPNESS)),2);
end

function OUT = hv(x,SHARPNESS)

OUT = 1./(1+exp(-x./SHARPNESS));


function PULSE = pulse_gui(varargin)
%PULSE_GUI Graphical interface for constructing pulses.
%   PULSE_GUI Brings up an interactive GUI for constructing
%   pulse trains.
%
%


if (nargin == 0)
	PULSE = struct('ton',0,'toff',1.2,'conc',10);
else
	PULSE = varargin{1};
end

if (nargin > 1)
	error('Too many inputs.');
end

if isempty(PULSE)
	PULSE = struct('ton',0,'toff',1.2,'conc',10);
end

NOTGOOD = logical(1);

while NOTGOOD
	prompt={'Enter the "on" times:','Enter the "off" times:','Enter the concentrations'};
	FMT = repmat('%f ',1,size(PULSE.ton,2));
	FMT = [FMT,';']; 
	TON = sprintf(FMT,PULSE.ton');
	TOFF = sprintf(FMT,PULSE.toff'); 
	CONC = sprintf(FMT,PULSE.conc');
	TON = TON(1:end-1);
	TOFF = TOFF(1:end-1);
	CONC = CONC(1:end-1);
	
	def={TON,TOFF,CONC};
	dlgTitle='Input for pulse train (row~trace,column~pulse)';
	lineNo=1;
	answer=inputdlg(prompt,dlgTitle,lineNo,def);

	if ~isempty(answer)
		PULSE.ton  = eval(['[',answer{1},']']);
		PULSE.toff = eval(['[',answer{2},']']);
		PULSE.conc = eval(['[',answer{3},']']);
	end
	
	RANGE = (max(PULSE.toff(:))-min(PULSE.ton(:)));
	
	T = linspace(min(PULSE.ton(:))-0.1*RANGE,max(PULSE.toff(:))+0.1*RANGE,100)';
	F = figure;
	plot(T,pulse_train(T,PULSE.ton,PULSE.toff,PULSE.conc),'LineWidth',2);
	xlabel('Time');
	ylabel('Concentration');
	AX = axis;
	AX(4) = AX(4) + 0.1*(AX(4)-AX(3));
	axis(AX);
	drawnow;
	

	ButtonName=questdlg('Is the pulse train correct?', ...
                	       'Accept pulse train?', ...
                	       'Yes','No','No');


	switch ButtonName
	case 'Yes' 
		NOTGOOD = logical(0);
	case 'No'
		NOTGOOD = logical(1);
	otherwise
		error('Unknown ButtonName');
	end 
	
	delete(F);
end
	
	
function hout=suptitle(str)
%SUPTITLE Puts a title above all subplots.
%	SUPTITLE('text') adds text to the top of the figure
%	above all subplots (a "super title"). Use this function
%	after all subplot commands.

% Drea Thomas 6/15/95 drea@mathworks.com

% Warning: If the figure or axis units are non-default, this
% will break.

% Parameters used to position the supertitle.

% Amount of the figure window devoted to subplots
plotregion = .92;

% Y position of title in normalized coordinates
titleypos  = .95;

% Fontsize for supertitle
fs = get(gcf,'defaultaxesfontsize')+4;

% Fudge factor to adjust y spacing between subplots
fudge=1;

haold = gca;
figunits = get(gcf,'units');

% Get the (approximate) difference between full height (plot + title
% + xlabel) and bounding rectangle.

	if (~strcmp(figunits,'pixels')),
		set(gcf,'units','pixels');
		pos = get(gcf,'position');
		set(gcf,'units',figunits);
	else,
		pos = get(gcf,'position');
	end
	ff = (fs-4)*1.27*5/pos(4)*fudge;

        % The 5 here reflects about 3 characters of height below
        % an axis and 2 above. 1.27 is pixels per point.

% Determine the bounding rectange for all the plots

% h = findobj('Type','axes');   

% findobj is a 4.2 thing.. if you don't have 4.2 comment out
% the next line and uncomment the following block.
	
h = findobj(gcf,'Type','axes');  % Change suggested by Stacy J. Hills

% If you don't have 4.2, use this code instead
%ch = get(gcf,'children');
%h=[];
%for i=1:length(ch),
%  if strcmp(get(ch(i),'type'),'axes'),
%    h=[h,ch(i)];
%  end
%end

	


max_y=0;
min_y=1;

oldtitle =0;
for i=1:length(h),
	if (~strcmp(get(h(i),'Tag'),'suptitle')),
		pos=get(h(i),'pos');
		if (pos(2) < min_y), min_y=pos(2)-ff/5*3;end;
		if (pos(4)+pos(2) > max_y), max_y=pos(4)+pos(2)+ff/5*2;end;
	else,
		oldtitle = h(i);
	end
end

if max_y > plotregion,
	scale = (plotregion-min_y)/(max_y-min_y);
	for i=1:length(h),
		pos = get(h(i),'position');
		pos(2) = (pos(2)-min_y)*scale+min_y;
		pos(4) = pos(4)*scale-(1-scale)*ff/5*3;
		set(h(i),'position',pos);
	end
end

np = get(gcf,'nextplot');
set(gcf,'nextplot','add');
if (oldtitle),
	delete(oldtitle);
end
ha=axes('pos',[0 1 1 1],'visible','off','Tag','suptitle');
ht=text(.5,titleypos-1,str);set(ht,'horizontalalignment','center','fontsize',fs);
set(gcf,'nextplot',np);
axes(haold);
if nargout,
	hout=ht;
end

function AX = subplotspace(varargin)
%SUBPLOTSPACE Spread subplots out.
%   SUBPLOTSPACE increases horizontal spacing between subplots      
%   by 1%.							    
%   SUBPLOTSPACE(DIRECTION) increases spacing between subplots      
%   by 1% where DIRECTION is either 'vertical' or 'horizontal'.     
%   SUBPLOTSPACE(DIRECTION,N) increases spacing between subplots    
%   by N%.  If N is negative then the space is reduced.				
        %   
%

	
%
%   Note, this function is useful for fixing the problem of 
%   axes labels and titles running into each other in figures 
%   using multiple subplots.
%
%   See also SUBPLOT
%

%Written by Daniel P. Dougherty 4/30/2004
%    

DIR = 'h';
N = 1;

if (nargin == 1)
	DIR = varargin{1};
elseif  (nargin == 2)
	DIR = varargin{1};
	N = varargin{2};	
elseif  (nargin > 2)
	error('Too many inputs.');
end

DIR = lower(DIR(1));

AX = findobj(gcf,'Type','axes');

if (isempty(AX) | (length(AX) <= 1))
	return;
end

UNITS = get(AX,'Units');

set(AX,'Units','pixels');

AXES = get(AX,'Position');
AXES = cat(1,AXES{:});

switch DIR
case 'h'
	UNODE = unique(AXES(:,1));
	FIGPOS = get(gcf,'Position');


	spacer = cumsum([0;diff(UNODE)]);
	UNODEnew = UNODE + (0.01*N)*spacer;

	for i = 1:length(UNODE)
		ind = find(AXES(:,1) == UNODE(i));
		for j = 1:length(ind)
			axes(AX(ind(j)));
			this_ax = get(gca,'Position');
			this_ax(1) = UNODEnew(i);
			set(gca,'Position',this_ax);
		end
	end

	FIGPOS(3) = FIGPOS(3)+ (0.01*N)*max(spacer);
	set(gcf,'Position',FIGPOS);

	for i = 1:length(AX)
		set(AX,'Units',UNITS{i});
	end
	
case 'v'
	UNODE = unique(AXES(:,2));
	FIGPOS = get(gcf,'Position');
	
	spacer = cumsum([0;diff(UNODE)]);
	UNODEnew = UNODE + (0.01*N)*spacer;

	for i = 1:length(UNODE)
		ind = find(AXES(:,2) == UNODE(i));
		for j = 1:length(ind)
			axes(AX(ind(j)));
			this_ax = get(gca,'Position');
			this_ax(2) = UNODEnew(i);
			set(gca,'Position',this_ax);
		end
	end
        
	FIGPOS(4) = FIGPOS(4)+ (0.01*N)*max(spacer);
	
	set(gcf,'Position',FIGPOS);

	for i = 1:length(AX)
		set(AX,'Units',UNITS{i});
	end
	
otherwise
	error('Unrecognized direction.');
end






