// Mammalian CA3 cell morphology
//
// Copyright 2007 John L Baker. All rights reserved.
//
// This software is provided AS IS under the terms of the Open Source
// MIT License. See http://www.opensource.org/licenses/mit-license.php.
//
// File: cell_l51_5_micron.cpp
//
// Release:		1.0.0
// Author:		John Baker
// Updated:		14 July 2006
//
// Description:
//
// Returns CA3 cell morphology in which compartment size is limited 
// to less than 5 microns (approx).
//
// SWC conversion parameters are as follows:
//
//	const float		maxCompLen = 5;		// Max compartment size (a few may be larger)
//	const float		rChgRelTol = 0.25f;	// Fractional change allowed in radius along branch
//	const float		rChgTolLen = 5;		// Minimum branch length when dividing for radius change
//	const float		rMaxForDendrite = 5.0;	// Max dendrite size next to soma (larger are merged into soma)	
//
//	const bool		useYAxisForSomaArea = false;	// Whether to assume soma aligned on Y axis
//	const bool		skipAxon = true;				// Skip writing the axon
//	const bool		debugBranchRadiusChg = false;	// Notify on each change
//
//	const float		maxXJump = 30;		// Maximum jump in x (microns)
//	const float		maxYJump = 30;		// Maximum jump in y (microns)
//	const float		maxZJump = 30;		// Maximum jump in z (microns)
//
// Conversion of soma points yields a very different area than noted
// in the original comments below. At present there is no explanation. 
//
// Cell geometry is taken from the Southampton archives.
// This file was created by the utility program swc_reader.
//
// References:
//
// See the Duke/Southamptom Cell Archive at http://neuron.duke.edu/cells/
// for information about SWC files (this was previously located at
// http://www.neuro.soton.ac.uk/cells/). The README file associated
// with the program CVAPP has further information about the SWC format.

#include "bnsf.h"
using namespace BNSF;

// Declare the morphology function as part of the namespace
namespace BAKER_2003 {
	MorphologyEntry* cell_l56a_5_micron();
}

// Provide the body of the function
MorphologyEntry* BAKER_2003::cell_l56a_5_micron() {
	
static MorphologyEntry cellMorphology[] = {

// ORIGINAL_SOURCE Neurolucida
// CREATURE rat F344
// REGION Hippocampus
// FIELD/LAYER CA3
// TYPE CA3b Pyramidal Cell in vivo young
// CONTRIBUTOR Buzsaki_G & Turner_DA
// REFERENCE J. Comp. Neurol. 356: 580-594, 1995
// RAW l56a.asc
// EXTRAS Turner_P.CA3
// SOMA_AREA 1.05E3
// SHRINKAGE_CORRECTION 1.33 1.33 2.5
// VERSION_NUMBER 2.0
// VERSION_DATE 1998-03-27
// *********************************************
// SCALE 1.33  1.33  2.5  
// 
{ 1,   0,  -1,   -1,   4.125,  63.473,    0.000,   -0.6,  -1.3,   1.0,   0.0,  0.0,  0.0},
{ 3,   1,   0,   11,   1.441,   4.936,    2.468,   -0.9,  -2.5,  -0.1,  -0.5, -2.3, -2.3},
{ 3,   2,   1,   11,   1.441,   4.936,    7.404,   -1.4,  -4.8,  -2.4,  -0.5, -2.3, -2.3},
{ 3,   3,   2,   11,   1.441,   4.936,   12.340,   -2.0,  -7.2,  -4.7,  -0.5, -2.3, -2.3},
{ 3,   4,   3,   11,   1.441,   4.936,   17.276,   -2.5,  -9.5,  -7.0,  -0.5, -2.3, -2.3},
{ 3,   5,   4,   11,   1.441,   4.936,   22.211,   -3.0, -11.9,  -9.3,  -0.5, -2.3, -2.3},
{ 3,   6,   5,   11,   1.441,   4.936,   27.147,   -3.6, -14.2, -11.6,  -0.5, -2.3, -2.3},
{ 3,   7,   6,   11,   1.441,   4.936,   32.083,   -4.1, -16.6, -13.9,  -0.5, -2.3, -2.3},
{ 3,   8,   7,   11,   1.441,   4.936,   37.019,   -4.6, -18.9, -16.2,  -0.5, -2.3, -2.3},
{ 3,   9,   8,   11,   1.441,   4.936,   41.955,   -5.2, -21.3, -18.5,  -0.5, -2.3, -2.3},
{ 3,  10,   9,   11,   1.441,   4.936,   46.891,   -5.7, -23.6, -20.8,  -0.5, -2.3, -2.3},
{ 3,  11,  10,   11,   1.441,   4.936,   51.827,   -6.2, -26.0, -23.1,  -0.5, -2.3, -2.3},
{ 3,  12,  11,   11,   1.441,   4.936,   56.763,   -6.8, -28.3, -25.4,  -0.5, -2.3, -2.3},
{ 3,  13,  12,   18,   1.009,   4.769,   61.615,   -7.9, -30.5, -27.5,  -1.7, -2.1, -2.1},
{ 3,  14,  13,   18,   1.009,   4.769,   66.385,   -9.6, -32.7, -29.6,  -1.7, -2.1, -2.1},
{ 3,  15,  14,   18,   1.009,   4.769,   71.154,  -11.4, -34.8, -31.7,  -1.7, -2.1, -2.1},
{ 3,  16,  15,   18,   1.009,   4.769,   75.923,  -13.1, -37.0, -33.8,  -1.7, -2.1, -2.1},
{ 3,  17,  16,   18,   1.009,   4.769,   80.693,  -14.9, -39.1, -35.9,  -1.7, -2.1, -2.1},
{ 3,  18,  17,   18,   1.009,   4.769,   85.462,  -16.6, -41.3, -38.0,  -1.7, -2.1, -2.1},
{ 3,  19,  18,   18,   1.009,   4.769,   90.232,  -18.4, -43.4, -40.1,  -1.7, -2.1, -2.1},
{ 3,  20,  19,   18,   1.009,   4.769,   95.001,  -20.1, -45.6, -42.2,  -1.7, -2.1, -2.1},
{ 3,  21,  20,   20,   0.720,   4.390,   99.581,  -21.1, -47.0, -41.2,  -0.2, -0.7,  4.2},
{ 3,  22,  21,   20,   0.720,   4.390,  103.971,  -21.3, -47.6, -37.0,  -0.2, -0.7,  4.2},
{ 3,  23,  22,   20,   0.720,   4.390,  108.361,  -21.5, -48.3, -32.8,  -0.2, -0.7,  4.2},
{ 3,  24,  23,   20,   0.720,   4.390,  112.752,  -21.7, -48.9, -28.6,  -0.2, -0.7,  4.2},
{ 3,  25,  24,   42,   0.400,   4.931,  117.412,  -23.4, -49.4, -26.7,  -3.0, -0.3, -0.3},
{ 3,  26,  25,   42,   0.400,   4.931,  122.343,  -26.4, -49.7, -27.0,  -3.0, -0.3, -0.3},
{ 3,  27,  26,   42,   0.400,   4.931,  127.273,  -29.4, -50.0, -27.3,  -3.0, -0.3, -0.3},
{ 3,  28,  27,   42,   0.400,   4.931,  132.204,  -32.4, -50.3, -27.6,  -3.0, -0.3, -0.3},
{ 3,  29,  28,   42,   0.400,   4.931,  137.135,  -35.4, -50.6, -28.0,  -3.0, -0.3, -0.3},
{ 3,  30,  29,   42,   0.400,   4.931,  142.065,  -38.4, -50.9, -28.3,  -3.0, -0.3, -0.3},
{ 3,  31,  30,   42,   0.400,   4.931,  146.996,  -41.4, -51.2, -28.6,  -3.0, -0.3, -0.3},
{ 3,  32,  31,   42,   0.400,   4.931,  151.926,  -44.4, -51.5, -29.0,  -3.0, -0.3, -0.3},
{ 3,  33,  32,   42,   0.400,   4.931,  156.857,  -47.4, -51.8, -29.3,  -3.0, -0.3, -0.3},
{ 3,  34,  33,   42,   0.400,   4.931,  161.788,  -50.4, -52.1, -29.6,  -3.0, -0.3, -0.3},
{ 3,  35,  34,   42,   0.400,   4.931,  166.718,  -53.4, -52.4, -29.9,  -3.0, -0.3, -0.3},
{ 3,  36,  35,   42,   0.400,   4.931,  171.649,  -56.4, -52.7, -30.3,  -3.0, -0.3, -0.3},
{ 3,  37,  36,   42,   0.400,   4.931,  176.580,  -59.4, -53.0, -30.6,  -3.0, -0.3, -0.3},
{ 3,  38,  37,   42,   0.400,   4.931,  181.510,  -62.4, -53.3, -30.9,  -3.0, -0.3, -0.3},
{ 3,  39,  38,   42,   0.400,   4.931,  186.441,  -65.4, -53.6, -31.3,  -3.0, -0.3, -0.3},
{ 3,  40,  39,   42,   0.400,   4.931,  191.371,  -68.4, -53.9, -31.6,  -3.0, -0.3, -0.3},
{ 3,  41,  40,   57,   0.247,   4.906,  196.290,  -70.1, -55.3, -31.7,  -0.3, -2.5,  0.1},
{ 3,  42,  41,   57,   0.247,   4.906,  201.196,  -70.3, -57.7, -31.6,  -0.3, -2.5,  0.1},
{ 3,  43,  42,   57,   0.247,   4.906,  206.101,  -70.6, -60.2, -31.5,  -0.3, -2.5,  0.1},
{ 3,  44,  43,   57,   0.247,   4.906,  211.007,  -70.9, -62.6, -31.4,  -0.3, -2.5,  0.1},
{ 3,  45,  44,   57,   0.247,   4.906,  215.913,  -71.2, -65.1, -31.4,  -0.3, -2.5,  0.1},
{ 3,  46,  45,   57,   0.247,   4.906,  220.819,  -71.4, -67.5, -31.3,  -0.3, -2.5,  0.1},
{ 3,  47,  46,   57,   0.247,   4.906,  225.725,  -71.7, -70.0, -31.2,  -0.3, -2.5,  0.1},
{ 3,  48,  47,   57,   0.247,   4.906,  230.631,  -72.0, -72.4, -31.1,  -0.3, -2.5,  0.1},
{ 3,  49,  48,   57,   0.247,   4.906,  235.537,  -72.3, -74.9, -31.0,  -0.3, -2.5,  0.1},
{ 3,  50,  49,   57,   0.247,   4.906,  240.443,  -72.6, -77.4, -30.9,  -0.3, -2.5,  0.1},
{ 3,  51,  50,   57,   0.247,   4.906,  245.349,  -72.8, -79.8, -30.8,  -0.3, -2.5,  0.1},
{ 3,  52,  51,   57,   0.247,   4.906,  250.255,  -73.1, -82.3, -30.8,  -0.3, -2.5,  0.1},
{ 3,  53,  52,   57,   0.247,   4.906,  255.161,  -73.4, -84.7, -30.7,  -0.3, -2.5,  0.1},
{ 3,  54,  53,   57,   0.247,   4.906,  260.067,  -73.7, -87.2, -30.6,  -0.3, -2.5,  0.1},
{ 3,  55,  54,   57,   0.247,   4.906,  264.973,  -73.9, -89.6, -30.5,  -0.3, -2.5,  0.1},
{ 3,  56,  55,   57,   0.247,   4.906,  269.878,  -74.2, -92.1, -30.4,  -0.3, -2.5,  0.1},
{ 3,  57,  56,   57,   0.247,   4.906,  274.784,  -74.5, -94.5, -30.3,  -0.3, -2.5,  0.1},
{ 3,  58,  57,   57,   0.247,   4.906,  279.690,  -74.8, -97.0, -30.2,  -0.3, -2.5,  0.1},
{ 3,  59,  58,   57,   0.247,   4.906,  284.596,  -75.1, -99.4, -30.2,  -0.3, -2.5,  0.1},
{ 3,  60,  59,   57,   0.247,   4.906,  289.502,  -75.3,-101.9, -30.1,  -0.3, -2.5,  0.1},
{ 3,  61,  60,   57,   0.247,   4.906,  294.408,  -75.6,-104.3, -30.0,  -0.3, -2.5,  0.1},
{ 3,  62,  61,   57,   0.247,   4.906,  299.314,  -75.9,-106.8, -29.9,  -0.3, -2.5,  0.1},
{ 3,  63,  62,   57,   0.247,   4.906,  304.220,  -76.2,-109.2, -29.8,  -0.3, -2.5,  0.1},
{ 3,  64,  63,   57,   0.247,   4.906,  309.126,  -76.4,-111.7, -29.7,  -0.3, -2.5,  0.1},
{ 3,  65,  64,   57,   0.247,   4.906,  314.032,  -76.7,-114.2, -29.6,  -0.3, -2.5,  0.1},
{ 3,  66,  65,   57,   0.247,   4.906,  318.938,  -77.0,-116.6, -29.6,  -0.3, -2.5,  0.1},
{ 3,  67,  66,   57,   0.247,   4.906,  323.844,  -77.3,-119.1, -29.5,  -0.3, -2.5,  0.1},
{ 3,  68,  67,   57,   0.247,   4.906,  328.750,  -77.5,-121.5, -29.4,  -0.3, -2.5,  0.1},
{ 3,  69,  68,   57,   0.247,   4.906,  333.655,  -77.8,-124.0, -29.3,  -0.3, -2.5,  0.1},
{ 3,  70,  24,   59,   0.560,   3.217,  116.555,  -23.2, -50.0, -26.5,  -2.8, -1.4,  0.0},
{ 3,  71,  70,   59,   0.560,   3.217,  119.772,  -26.0, -51.4, -26.5,  -2.8, -1.4,  0.0},
{ 3,  72,  71,   67,   0.560,   4.791,  123.776,  -27.5, -53.7, -26.8,  -0.4, -3.2, -0.5},
{ 3,  73,  72,   67,   0.560,   4.791,  128.567,  -27.9, -56.9, -27.3,  -0.4, -3.2, -0.5},
{ 3,  74,  73,   67,   0.560,   4.791,  133.358,  -28.3, -60.1, -27.8,  -0.4, -3.2, -0.5},
{ 3,  75,  74,   67,   0.560,   4.791,  138.149,  -28.7, -63.4, -28.3,  -0.4, -3.2, -0.5},
{ 3,  76,  75,   67,   0.560,   4.791,  142.940,  -29.1, -66.6, -28.8,  -0.4, -3.2, -0.5},
{ 3,  77,  76,   69,   0.524,   4.065,  147.368,  -29.9, -69.8, -29.8,  -1.4, -3.3, -1.5},
{ 3,  78,  77,   69,   0.524,   4.065,  151.432,  -31.3, -73.2, -31.3,  -1.4, -3.3, -1.5},
{ 3,  79,  78,   69,   0.524,   4.065,  155.497,  -32.7, -76.5, -32.8,  -1.4, -3.3, -1.5},
{ 3,  80,  79,   69,   0.524,   4.065,  159.561,  -34.1, -79.8, -34.3,  -1.4, -3.3, -1.5},
{ 3,  81,  80,   72,   0.400,   4.137,  163.662,  -35.5, -83.3, -35.0,  -1.5, -3.7,  0.0},
{ 3,  82,  81,   72,   0.400,   4.137,  167.800,  -37.0, -87.0, -35.0,  -1.5, -3.7,  0.0},
{ 3,  83,  82,   75,   0.400,   4.829,  172.283,  -38.5, -90.6, -33.6,  -1.6, -3.5,  2.8},
{ 3,  84,  83,   75,   0.400,   4.829,  177.113,  -40.2, -94.1, -30.7,  -1.6, -3.5,  2.8},
{ 3,  85,  84,   75,   0.400,   4.829,  181.942,  -41.8, -97.6, -27.9,  -1.6, -3.5,  2.8},
{ 3,  86,  85,  106,   0.240,   4.926,  186.820,  -42.8,-100.7, -26.6,  -0.4, -2.8, -0.3},
{ 3,  87,  86,  106,   0.240,   4.926,  191.745,  -43.2,-103.5, -26.9,  -0.4, -2.8, -0.3},
{ 3,  88,  87,  106,   0.240,   4.926,  196.671,  -43.5,-106.4, -27.2,  -0.4, -2.8, -0.3},
{ 3,  89,  88,  106,   0.240,   4.926,  201.597,  -43.9,-109.2, -27.5,  -0.4, -2.8, -0.3},
{ 3,  90,  89,  106,   0.240,   4.926,  206.523,  -44.3,-112.0, -27.8,  -0.4, -2.8, -0.3},
{ 3,  91,  90,  106,   0.240,   4.926,  211.449,  -44.6,-114.8, -28.1,  -0.4, -2.8, -0.3},
{ 3,  92,  91,  106,   0.240,   4.926,  216.375,  -45.0,-117.7, -28.4,  -0.4, -2.8, -0.3},
{ 3,  93,  92,  106,   0.240,   4.926,  221.301,  -45.4,-120.5, -28.7,  -0.4, -2.8, -0.3},
{ 3,  94,  93,  106,   0.240,   4.926,  226.227,  -45.7,-123.3, -29.0,  -0.4, -2.8, -0.3},
{ 3,  95,  94,  106,   0.240,   4.926,  231.153,  -46.1,-126.1, -29.3,  -0.4, -2.8, -0.3},
{ 3,  96,  95,  106,   0.240,   4.926,  236.079,  -46.5,-128.9, -29.6,  -0.4, -2.8, -0.3},
{ 3,  97,  96,  106,   0.240,   4.926,  241.005,  -46.8,-131.8, -29.9,  -0.4, -2.8, -0.3},
{ 3,  98,  97,  106,   0.240,   4.926,  245.931,  -47.2,-134.6, -30.2,  -0.4, -2.8, -0.3},
{ 3,  99,  98,  106,   0.240,   4.926,  250.857,  -47.6,-137.4, -30.5,  -0.4, -2.8, -0.3},
{ 3, 100,  99,  106,   0.240,   4.926,  255.782,  -47.9,-140.2, -30.8,  -0.4, -2.8, -0.3},
{ 3, 101, 100,  106,   0.240,   4.926,  260.708,  -48.3,-143.0, -31.1,  -0.4, -2.8, -0.3},
{ 3, 102, 101,  106,   0.240,   4.926,  265.634,  -48.7,-145.9, -31.4,  -0.4, -2.8, -0.3},
{ 3, 103, 102,  106,   0.240,   4.926,  270.560,  -49.0,-148.7, -31.6,  -0.4, -2.8, -0.3},
{ 3, 104, 103,  106,   0.240,   4.926,  275.486,  -49.4,-151.5, -31.9,  -0.4, -2.8, -0.3},
{ 3, 105, 104,  106,   0.240,   4.926,  280.412,  -49.8,-154.3, -32.2,  -0.4, -2.8, -0.3},
{ 3, 106, 105,  106,   0.240,   4.926,  285.338,  -50.1,-157.2, -32.5,  -0.4, -2.8, -0.3},
{ 3, 107, 106,  106,   0.240,   4.926,  290.264,  -50.5,-160.0, -32.8,  -0.4, -2.8, -0.3},
{ 3, 108, 107,  106,   0.240,   4.926,  295.190,  -50.9,-162.8, -33.1,  -0.4, -2.8, -0.3},
{ 3, 109, 108,  106,   0.240,   4.926,  300.116,  -51.2,-165.6, -33.4,  -0.4, -2.8, -0.3},
{ 3, 110, 109,  106,   0.240,   4.926,  305.042,  -51.6,-168.4, -33.7,  -0.4, -2.8, -0.3},
{ 3, 111, 110,  106,   0.240,   4.926,  309.968,  -52.0,-171.3, -34.0,  -0.4, -2.8, -0.3},
{ 3, 112, 111,  106,   0.240,   4.926,  314.894,  -52.3,-174.1, -34.3,  -0.4, -2.8, -0.3},
{ 3, 113, 112,  106,   0.240,   4.926,  319.819,  -52.7,-176.9, -34.6,  -0.4, -2.8, -0.3},
{ 3, 114, 113,  106,   0.240,   4.926,  324.745,  -53.1,-179.7, -34.9,  -0.4, -2.8, -0.3},
{ 3, 115, 114,  106,   0.240,   4.926,  329.671,  -53.4,-182.5, -35.2,  -0.4, -2.8, -0.3},
{ 3, 116, 115,  106,   0.240,   4.926,  334.597,  -53.8,-185.4, -35.5,  -0.4, -2.8, -0.3},
{ 3, 117, 116,  106,   0.240,   4.926,  339.523,  -54.2,-188.2, -35.8,  -0.4, -2.8, -0.3},
{ 3, 118, 117,  106,   0.240,   4.926,  344.449,  -54.5,-191.0, -36.1,  -0.4, -2.8, -0.3},
{ 3, 119, 118,  106,   0.240,   4.926,  349.375,  -54.9,-193.8, -36.4,  -0.4, -2.8, -0.3},
{ 3, 120,  85,  141,   0.240,   4.876,  186.795,  -43.0,-100.5, -27.0,  -0.7, -2.3, -1.0},
{ 3, 121, 120,  141,   0.240,   4.876,  191.670,  -43.7,-102.8, -28.0,  -0.7, -2.3, -1.0},
{ 3, 122, 121,  141,   0.240,   4.876,  196.546,  -44.4,-105.1, -29.0,  -0.7, -2.3, -1.0},
{ 3, 123, 122,  141,   0.240,   4.876,  201.422,  -45.1,-107.5, -29.9,  -0.7, -2.3, -1.0},
{ 3, 124, 123,  141,   0.240,   4.876,  206.298,  -45.8,-109.8, -30.9,  -0.7, -2.3, -1.0},
{ 3, 125, 124,  141,   0.240,   4.876,  211.174,  -46.5,-112.1, -31.9,  -0.7, -2.3, -1.0},
{ 3, 126, 125,  141,   0.240,   4.876,  216.050,  -47.2,-114.5, -32.9,  -0.7, -2.3, -1.0},
{ 3, 127, 126,  141,   0.240,   4.876,  220.926,  -47.9,-116.8, -33.9,  -0.7, -2.3, -1.0},
{ 3, 128, 127,  141,   0.240,   4.876,  225.802,  -48.6,-119.1, -34.9,  -0.7, -2.3, -1.0},
{ 3, 129, 128,  141,   0.240,   4.876,  230.678,  -49.3,-121.5, -35.9,  -0.7, -2.3, -1.0},
{ 3, 130, 129,  141,   0.240,   4.876,  235.554,  -50.0,-123.8, -36.8,  -0.7, -2.3, -1.0},
{ 3, 131, 130,  141,   0.240,   4.876,  240.430,  -50.7,-126.1, -37.8,  -0.7, -2.3, -1.0},
{ 3, 132, 131,  141,   0.240,   4.876,  245.305,  -51.5,-128.5, -38.8,  -0.7, -2.3, -1.0},
{ 3, 133, 132,  141,   0.240,   4.876,  250.181,  -52.2,-130.8, -39.8,  -0.7, -2.3, -1.0},
{ 3, 134, 133,  141,   0.240,   4.876,  255.057,  -52.9,-133.1, -40.8,  -0.7, -2.3, -1.0},
{ 3, 135, 134,  141,   0.240,   4.876,  259.933,  -53.6,-135.5, -41.8,  -0.7, -2.3, -1.0},
{ 3, 136, 135,  141,   0.240,   4.876,  264.809,  -54.3,-137.8, -42.7,  -0.7, -2.3, -1.0},
{ 3, 137, 136,  141,   0.240,   4.876,  269.685,  -55.0,-140.1, -43.7,  -0.7, -2.3, -1.0},
{ 3, 138, 137,  141,   0.240,   4.876,  274.561,  -55.7,-142.4, -44.7,  -0.7, -2.3, -1.0},
{ 3, 139, 138,  141,   0.240,   4.876,  279.437,  -56.4,-144.8, -45.7,  -0.7, -2.3, -1.0},
{ 3, 140, 139,  141,   0.240,   4.876,  284.313,  -57.1,-147.1, -46.7,  -0.7, -2.3, -1.0},
{ 3, 141, 140,  141,   0.240,   4.876,  289.189,  -57.8,-149.4, -47.7,  -0.7, -2.3, -1.0},
{ 3, 142, 141,  141,   0.240,   4.876,  294.065,  -58.5,-151.8, -48.6,  -0.7, -2.3, -1.0},
{ 3, 143, 142,  141,   0.240,   4.876,  298.940,  -59.2,-154.1, -49.6,  -0.7, -2.3, -1.0},
{ 3, 144, 143,  141,   0.240,   4.876,  303.816,  -60.0,-156.4, -50.6,  -0.7, -2.3, -1.0},
{ 3, 145, 144,  141,   0.240,   4.876,  308.692,  -60.7,-158.8, -51.6,  -0.7, -2.3, -1.0},
{ 3, 146, 145,  141,   0.240,   4.876,  313.568,  -61.4,-161.1, -52.6,  -0.7, -2.3, -1.0},
{ 3, 147, 146,  141,   0.240,   4.876,  318.444,  -62.1,-163.4, -53.6,  -0.7, -2.3, -1.0},
{ 3, 148, 147,  141,   0.240,   4.876,  323.320,  -62.8,-165.8, -54.6,  -0.7, -2.3, -1.0},
{ 3, 149, 148,  141,   0.240,   4.876,  328.196,  -63.5,-168.1, -55.5,  -0.7, -2.3, -1.0},
{ 3, 150, 149,  141,   0.240,   4.876,  333.072,  -64.2,-170.4, -56.5,  -0.7, -2.3, -1.0},
{ 3, 151, 150,  141,   0.240,   4.876,  337.948,  -64.9,-172.8, -57.5,  -0.7, -2.3, -1.0},
{ 3, 152,  82,  144,   0.400,   4.942,  172.339,  -37.0, -90.6, -34.4,   1.4, -3.5,  1.3},
{ 3, 153, 152,  144,   0.400,   4.942,  177.281,  -35.7, -94.1, -33.1,   1.4, -3.5,  1.3},
{ 3, 154, 153,  170,   0.245,   4.892,  182.198,  -34.6, -97.4, -33.1,   0.8, -3.1, -1.3},
{ 3, 155, 154,  170,   0.245,   4.892,  187.089,  -33.8,-100.5, -34.4,   0.8, -3.1, -1.3},
{ 3, 156, 155,  170,   0.245,   4.892,  191.981,  -33.0,-103.7, -35.7,   0.8, -3.1, -1.3},
{ 3, 157, 156,  170,   0.245,   4.892,  196.872,  -32.2,-106.8, -37.0,   0.8, -3.1, -1.3},
{ 3, 158, 157,  170,   0.245,   4.892,  201.764,  -31.4,-109.9, -38.3,   0.8, -3.1, -1.3},
{ 3, 159, 158,  170,   0.245,   4.892,  206.656,  -30.6,-113.1, -39.5,   0.8, -3.1, -1.3},
{ 3, 160, 159,  170,   0.245,   4.892,  211.547,  -29.8,-116.2, -40.8,   0.8, -3.1, -1.3},
{ 3, 161, 160,  170,   0.245,   4.892,  216.439,  -29.0,-119.3, -42.1,   0.8, -3.1, -1.3},
{ 3, 162, 161,  170,   0.245,   4.892,  221.330,  -28.2,-122.5, -43.4,   0.8, -3.1, -1.3},
{ 3, 163, 162,  170,   0.245,   4.892,  226.222,  -27.4,-125.6, -44.6,   0.8, -3.1, -1.3},
{ 3, 164, 163,  170,   0.245,   4.892,  231.113,  -26.6,-128.7, -45.9,   0.8, -3.1, -1.3},
{ 3, 165, 164,  170,   0.245,   4.892,  236.005,  -25.8,-131.9, -47.2,   0.8, -3.1, -1.3},
{ 3, 166, 165,  170,   0.245,   4.892,  240.897,  -25.0,-135.0, -48.5,   0.8, -3.1, -1.3},
{ 3, 167, 166,  170,   0.245,   4.892,  245.788,  -24.2,-138.1, -49.8,   0.8, -3.1, -1.3},
{ 3, 168, 167,  170,   0.245,   4.892,  250.680,  -23.4,-141.3, -51.0,   0.8, -3.1, -1.3},
{ 3, 169, 168,  170,   0.245,   4.892,  255.571,  -22.6,-144.4, -52.3,   0.8, -3.1, -1.3},
{ 3, 170, 169,  170,   0.245,   4.892,  260.463,  -21.8,-147.5, -53.6,   0.8, -3.1, -1.3},
{ 3, 171, 170,  170,   0.245,   4.892,  265.354,  -21.0,-150.7, -54.9,   0.8, -3.1, -1.3},
{ 3, 172, 171,  170,   0.245,   4.892,  270.246,  -20.2,-153.8, -56.1,   0.8, -3.1, -1.3},
{ 3, 173, 172,  170,   0.245,   4.892,  275.138,  -19.4,-156.9, -57.4,   0.8, -3.1, -1.3},
{ 3, 174, 173,  170,   0.245,   4.892,  280.029,  -18.6,-160.1, -58.7,   0.8, -3.1, -1.3},
{ 3, 175, 174,  170,   0.245,   4.892,  284.921,  -17.8,-163.2, -60.0,   0.8, -3.1, -1.3},
{ 3, 176, 175,  170,   0.245,   4.892,  289.812,  -17.0,-166.3, -61.3,   0.8, -3.1, -1.3},
{ 3, 177, 176,  170,   0.245,   4.892,  294.704,  -16.2,-169.5, -62.5,   0.8, -3.1, -1.3},
{ 3, 178, 177,  170,   0.245,   4.892,  299.595,  -15.4,-172.6, -63.8,   0.8, -3.1, -1.3},
{ 3, 179, 178,  170,   0.245,   4.892,  304.487,  -14.6,-175.8, -65.1,   0.8, -3.1, -1.3},
{ 3, 180, 179,  170,   0.245,   4.892,  309.379,  -13.8,-178.9, -66.4,   0.8, -3.1, -1.3},
{ 3, 181,  76,  183,   0.240,   4.695,  147.683,  -30.2, -69.5, -28.8,  -1.8, -2.7,  0.4},
{ 3, 182, 181,  183,   0.240,   4.695,  152.377,  -32.0, -72.2, -28.5,  -1.8, -2.7,  0.4},
{ 3, 183, 182,  183,   0.240,   4.695,  157.072,  -33.9, -74.9, -28.1,  -1.8, -2.7,  0.4},
{ 3, 184, 183,  183,   0.240,   4.695,  161.767,  -35.7, -77.5, -27.7,  -1.8, -2.7,  0.4},
{ 3, 185, 184,  183,   0.240,   4.695,  166.461,  -37.6, -80.2, -27.4,  -1.8, -2.7,  0.4},
{ 3, 186, 185,  183,   0.240,   4.695,  171.156,  -39.4, -82.9, -27.0,  -1.8, -2.7,  0.4},
{ 3, 187, 186,  183,   0.240,   4.695,  175.850,  -41.3, -85.6, -26.7,  -1.8, -2.7,  0.4},
{ 3, 188, 187,  219,   0.398,   4.913,  180.654,  -42.7, -88.3, -26.8,  -1.1, -2.8, -0.6},
{ 3, 189, 188,  219,   0.398,   4.913,  185.568,  -43.8, -91.2, -27.4,  -1.1, -2.8, -0.6},
{ 3, 190, 189,  219,   0.398,   4.913,  190.481,  -44.8, -94.0, -27.9,  -1.1, -2.8, -0.6},
{ 3, 191, 190,  219,   0.398,   4.913,  195.394,  -45.9, -96.8, -28.5,  -1.1, -2.8, -0.6},
{ 3, 192, 191,  219,   0.398,   4.913,  200.308,  -46.9, -99.7, -29.1,  -1.1, -2.8, -0.6},
{ 3, 193, 192,  219,   0.398,   4.913,  205.221,  -48.0,-102.5, -29.7,  -1.1, -2.8, -0.6},
{ 3, 194, 193,  219,   0.398,   4.913,  210.135,  -49.0,-105.3, -30.2,  -1.1, -2.8, -0.6},
{ 3, 195, 194,  219,   0.398,   4.913,  215.048,  -50.1,-108.2, -30.8,  -1.1, -2.8, -0.6},
{ 3, 196, 195,  219,   0.398,   4.913,  219.961,  -51.1,-111.0, -31.4,  -1.1, -2.8, -0.6},
{ 3, 197, 196,  219,   0.398,   4.913,  224.875,  -52.2,-113.8, -32.0,  -1.1, -2.8, -0.6},
{ 3, 198, 197,  219,   0.398,   4.913,  229.788,  -53.2,-116.7, -32.5,  -1.1, -2.8, -0.6},
{ 3, 199, 198,  219,   0.398,   4.913,  234.701,  -54.3,-119.5, -33.1,  -1.1, -2.8, -0.6},
{ 3, 200, 199,  219,   0.398,   4.913,  239.615,  -55.3,-122.3, -33.7,  -1.1, -2.8, -0.6},
{ 3, 201, 200,  219,   0.398,   4.913,  244.528,  -56.4,-125.2, -34.3,  -1.1, -2.8, -0.6},
{ 3, 202, 201,  219,   0.398,   4.913,  249.441,  -57.4,-128.0, -34.8,  -1.1, -2.8, -0.6},
{ 3, 203, 202,  219,   0.398,   4.913,  254.355,  -58.5,-130.8, -35.4,  -1.1, -2.8, -0.6},
{ 3, 204, 203,  219,   0.398,   4.913,  259.268,  -59.6,-133.7, -36.0,  -1.1, -2.8, -0.6},
{ 3, 205, 204,  219,   0.398,   4.913,  264.182,  -60.6,-136.5, -36.6,  -1.1, -2.8, -0.6},
{ 3, 206, 205,  219,   0.398,   4.913,  269.095,  -61.7,-139.3, -37.1,  -1.1, -2.8, -0.6},
{ 3, 207, 206,  219,   0.398,   4.913,  274.008,  -62.7,-142.2, -37.7,  -1.1, -2.8, -0.6},
{ 3, 208, 207,  219,   0.398,   4.913,  278.922,  -63.8,-145.0, -38.3,  -1.1, -2.8, -0.6},
{ 3, 209, 208,  219,   0.398,   4.913,  283.835,  -64.8,-147.8, -38.8,  -1.1, -2.8, -0.6},
{ 3, 210, 209,  219,   0.398,   4.913,  288.748,  -65.9,-150.7, -39.4,  -1.1, -2.8, -0.6},
{ 3, 211, 210,  219,   0.398,   4.913,  293.662,  -66.9,-153.5, -40.0,  -1.1, -2.8, -0.6},
{ 3, 212, 211,  219,   0.398,   4.913,  298.575,  -68.0,-156.3, -40.6,  -1.1, -2.8, -0.6},
{ 3, 213, 212,  219,   0.398,   4.913,  303.488,  -69.0,-159.2, -41.1,  -1.1, -2.8, -0.6},
{ 3, 214, 213,  219,   0.398,   4.913,  308.402,  -70.1,-162.0, -41.7,  -1.1, -2.8, -0.6},
{ 3, 215, 214,  219,   0.398,   4.913,  313.315,  -71.1,-164.8, -42.3,  -1.1, -2.8, -0.6},
{ 3, 216, 215,  219,   0.398,   4.913,  318.229,  -72.2,-167.7, -42.9,  -1.1, -2.8, -0.6},
{ 3, 217, 216,  219,   0.398,   4.913,  323.142,  -73.2,-170.5, -43.4,  -1.1, -2.8, -0.6},
{ 3, 218, 217,  219,   0.398,   4.913,  328.055,  -74.3,-173.3, -44.0,  -1.1, -2.8, -0.6},
{ 3, 219, 218,  219,   0.398,   4.913,  332.969,  -75.4,-176.2, -44.6,  -1.1, -2.8, -0.6},
{ 3, 220, 219,  219,   0.398,   4.913,  337.882,  -76.4,-179.0, -45.2,  -1.1, -2.8, -0.6},
{ 3, 221, 220,  219,   0.398,   4.913,  342.795,  -77.5,-181.8, -45.7,  -1.1, -2.8, -0.6},
{ 3, 222, 221,  219,   0.398,   4.913,  347.709,  -78.5,-184.7, -46.3,  -1.1, -2.8, -0.6},
{ 3, 223, 222,  219,   0.398,   4.913,  352.622,  -79.6,-187.5, -46.9,  -1.1, -2.8, -0.6},
{ 3, 224, 223,  219,   0.398,   4.913,  357.535,  -80.6,-190.3, -47.5,  -1.1, -2.8, -0.6},
{ 3, 225,  71,  222,   0.523,   4.192,  123.476,  -28.2, -53.0, -25.9,  -1.7, -1.8,  1.1},
{ 3, 226, 225,  222,   0.523,   4.192,  127.668,  -29.9, -54.8, -24.8,  -1.7, -1.8,  1.1},
{ 3, 227, 226,  222,   0.523,   4.192,  131.861,  -31.6, -56.6, -23.7,  -1.7, -1.8,  1.1},
{ 3, 228, 227,  222,   0.523,   4.192,  136.053,  -33.3, -58.4, -22.6,  -1.7, -1.8,  1.1},
{ 3, 229, 228,  223,   0.400,   0.488,  138.393,  -34.3, -59.6, -22.0,  -0.2, -0.4,  0.0},
{ 3, 230, 229,  225,   0.400,   5.148,  141.211,  -33.7, -61.0, -19.8,   1.4, -2.4,  4.3},
{ 3, 231, 230,  225,   0.400,   5.148,  146.359,  -32.3, -63.4, -15.5,   1.4, -2.4,  4.3},
{ 3, 232, 231,  225,   0.400,   5.148,  151.507,  -30.9, -65.7, -11.2,   1.4, -2.4,  4.3},
{ 3, 233, 232,  253,   0.242,   5.012,  156.587,  -30.6, -68.8,  -8.9,  -0.8, -3.6,  0.2},
{ 3, 234, 233,  253,   0.242,   5.012,  161.599,  -31.3, -72.4,  -8.7,  -0.8, -3.6,  0.2},
{ 3, 235, 234,  253,   0.242,   5.012,  166.611,  -32.1, -76.0,  -8.6,  -0.8, -3.6,  0.2},
{ 3, 236, 235,  253,   0.242,   5.012,  171.623,  -32.9, -79.7,  -8.4,  -0.8, -3.6,  0.2},
{ 3, 237, 236,  253,   0.242,   5.012,  176.635,  -33.7, -83.3,  -8.2,  -0.8, -3.6,  0.2},
{ 3, 238, 237,  253,   0.242,   5.012,  181.646,  -34.5, -87.0,  -8.1,  -0.8, -3.6,  0.2},
{ 3, 239, 238,  253,   0.242,   5.012,  186.658,  -35.3, -90.6,  -7.9,  -0.8, -3.6,  0.2},
{ 3, 240, 239,  253,   0.242,   5.012,  191.670,  -36.1, -94.3,  -7.7,  -0.8, -3.6,  0.2},
{ 3, 241, 240,  253,   0.242,   5.012,  196.682,  -36.9, -97.9,  -7.5,  -0.8, -3.6,  0.2},
{ 3, 242, 241,  253,   0.242,   5.012,  201.694,  -37.7,-101.5,  -7.4,  -0.8, -3.6,  0.2},
{ 3, 243, 242,  253,   0.242,   5.012,  206.706,  -38.5,-105.2,  -7.2,  -0.8, -3.6,  0.2},
{ 3, 244, 243,  253,   0.242,   5.012,  211.718,  -39.3,-108.8,  -7.0,  -0.8, -3.6,  0.2},
{ 3, 245, 244,  253,   0.242,   5.012,  216.730,  -40.1,-112.5,  -6.9,  -0.8, -3.6,  0.2},
{ 3, 246, 245,  253,   0.242,   5.012,  221.742,  -40.9,-116.1,  -6.7,  -0.8, -3.6,  0.2},
{ 3, 247, 246,  253,   0.242,   5.012,  226.754,  -41.7,-119.8,  -6.5,  -0.8, -3.6,  0.2},
{ 3, 248, 247,  253,   0.242,   5.012,  231.765,  -42.5,-123.4,  -6.4,  -0.8, -3.6,  0.2},
{ 3, 249, 248,  253,   0.242,   5.012,  236.777,  -43.3,-127.0,  -6.2,  -0.8, -3.6,  0.2},
{ 3, 250, 249,  253,   0.242,   5.012,  241.789,  -44.1,-130.7,  -6.0,  -0.8, -3.6,  0.2},
{ 3, 251, 250,  253,   0.242,   5.012,  246.801,  -44.9,-134.3,  -5.8,  -0.8, -3.6,  0.2},
{ 3, 252, 251,  253,   0.242,   5.012,  251.813,  -45.7,-138.0,  -5.7,  -0.8, -3.6,  0.2},
{ 3, 253, 252,  253,   0.242,   5.012,  256.825,  -46.5,-141.6,  -5.5,  -0.8, -3.6,  0.2},
{ 3, 254, 253,  253,   0.242,   5.012,  261.837,  -47.3,-145.2,  -5.3,  -0.8, -3.6,  0.2},
{ 3, 255, 254,  253,   0.242,   5.012,  266.849,  -48.1,-148.9,  -5.2,  -0.8, -3.6,  0.2},
{ 3, 256, 255,  253,   0.242,   5.012,  271.861,  -48.9,-152.5,  -5.0,  -0.8, -3.6,  0.2},
{ 3, 257, 256,  253,   0.242,   5.012,  276.873,  -49.7,-156.2,  -4.8,  -0.8, -3.6,  0.2},
{ 3, 258, 257,  253,   0.242,   5.012,  281.884,  -50.5,-159.8,  -4.6,  -0.8, -3.6,  0.2},
{ 3, 259, 258,  253,   0.242,   5.012,  286.896,  -51.3,-163.5,  -4.5,  -0.8, -3.6,  0.2},
{ 3, 260, 259,  253,   0.242,   5.012,  291.908,  -52.1,-167.1,  -4.3,  -0.8, -3.6,  0.2},
{ 3, 261, 260,  253,   0.242,   5.012,  296.920,  -52.9,-170.7,  -4.1,  -0.8, -3.6,  0.2},
{ 3, 262, 261,  253,   0.242,   5.012,  301.932,  -53.7,-174.4,  -4.0,  -0.8, -3.6,  0.2},
{ 3, 263, 262,  253,   0.242,   5.012,  306.944,  -54.5,-178.0,  -3.8,  -0.8, -3.6,  0.2},
{ 3, 264, 263,  253,   0.242,   5.012,  311.956,  -55.3,-181.7,  -3.6,  -0.8, -3.6,  0.2},
{ 3, 265, 264,  253,   0.242,   5.012,  316.968,  -56.1,-185.3,  -3.5,  -0.8, -3.6,  0.2},
{ 3, 266, 265,  253,   0.242,   5.012,  321.980,  -56.9,-189.0,  -3.3,  -0.8, -3.6,  0.2},
{ 3, 267, 266,  253,   0.242,   5.012,  326.992,  -57.7,-192.6,  -3.1,  -0.8, -3.6,  0.2},
{ 3, 268, 267,  253,   0.242,   5.012,  332.003,  -58.5,-196.2,  -2.9,  -0.8, -3.6,  0.2},
{ 3, 269, 268,  253,   0.242,   5.012,  337.015,  -59.3,-199.9,  -2.8,  -0.8, -3.6,  0.2},
{ 3, 270, 269,  253,   0.242,   5.012,  342.027,  -60.1,-203.5,  -2.6,  -0.8, -3.6,  0.2},
{ 3, 271, 270,  253,   0.242,   5.012,  347.039,  -60.9,-207.2,  -2.4,  -0.8, -3.6,  0.2},
{ 3, 272, 271,  253,   0.242,   5.012,  352.051,  -61.7,-210.8,  -2.3,  -0.8, -3.6,  0.2},
{ 3, 273, 272,  253,   0.242,   5.012,  357.063,  -62.5,-214.4,  -2.1,  -0.8, -3.6,  0.2},
{ 3, 274, 229,  279,   0.400,   4.981,  141.127,  -35.3, -61.6, -21.6,  -1.8, -3.7,  0.7},
{ 3, 275, 274,  279,   0.400,   4.981,  146.108,  -37.1, -65.3, -20.9,  -1.8, -3.7,  0.7},
{ 3, 276, 275,  279,   0.400,   4.981,  151.088,  -38.9, -68.9, -20.2,  -1.8, -3.7,  0.7},
{ 3, 277, 276,  279,   0.400,   4.981,  156.069,  -40.6, -72.6, -19.5,  -1.8, -3.7,  0.7},
{ 3, 278, 277,  279,   0.400,   4.981,  161.050,  -42.4, -76.2, -18.8,  -1.8, -3.7,  0.7},
{ 3, 279, 278,  279,   0.400,   4.981,  166.030,  -44.2, -79.9, -18.1,  -1.8, -3.7,  0.7},
{ 3, 280, 279,  279,   0.400,   4.981,  171.011,  -46.0, -83.6, -17.4,  -1.8, -3.7,  0.7},
{ 3, 281, 280,  279,   0.400,   4.981,  175.991,  -47.8, -87.2, -16.7,  -1.8, -3.7,  0.7},
{ 3, 282, 281,  279,   0.400,   4.981,  180.972,  -49.6, -90.9, -16.0,  -1.8, -3.7,  0.7},
{ 3, 283, 282,  279,   0.400,   4.981,  185.952,  -51.3, -94.5, -15.2,  -1.8, -3.7,  0.7},
{ 3, 284, 283,  279,   0.400,   4.981,  190.933,  -53.1, -98.2, -14.5,  -1.8, -3.7,  0.7},
{ 3, 285, 284,  279,   0.400,   4.981,  195.914,  -54.9,-101.8, -13.8,  -1.8, -3.7,  0.7},
{ 3, 286, 285,  279,   0.400,   4.981,  200.894,  -56.7,-105.5, -13.1,  -1.8, -3.7,  0.7},
{ 3, 287, 286,  279,   0.400,   4.981,  205.875,  -58.5,-109.2, -12.4,  -1.8, -3.7,  0.7},
{ 3, 288, 287,  279,   0.400,   4.981,  210.855,  -60.3,-112.8, -11.7,  -1.8, -3.7,  0.7},
{ 3, 289, 288,  279,   0.400,   4.981,  215.836,  -62.0,-116.5, -11.0,  -1.8, -3.7,  0.7},
{ 3, 290, 289,  279,   0.400,   4.981,  220.817,  -63.8,-120.1, -10.3,  -1.8, -3.7,  0.7},
{ 3, 291, 290,  279,   0.400,   4.981,  225.797,  -65.6,-123.8,  -9.6,  -1.8, -3.7,  0.7},
{ 3, 292, 291,  279,   0.400,   4.981,  230.778,  -67.4,-127.5,  -8.9,  -1.8, -3.7,  0.7},
{ 3, 293, 292,  279,   0.400,   4.981,  235.758,  -69.2,-131.1,  -8.1,  -1.8, -3.7,  0.7},
{ 3, 294, 293,  279,   0.400,   4.981,  240.739,  -71.0,-134.8,  -7.4,  -1.8, -3.7,  0.7},
{ 3, 295, 294,  279,   0.400,   4.981,  245.719,  -72.7,-138.4,  -6.7,  -1.8, -3.7,  0.7},
{ 3, 296, 295,  279,   0.400,   4.981,  250.700,  -74.5,-142.1,  -6.0,  -1.8, -3.7,  0.7},
{ 3, 297, 296,  279,   0.400,   4.981,  255.681,  -76.3,-145.8,  -5.3,  -1.8, -3.7,  0.7},
{ 3, 298, 297,  279,   0.400,   4.981,  260.661,  -78.1,-149.4,  -4.6,  -1.8, -3.7,  0.7},
{ 3, 299, 298,  279,   0.400,   4.981,  265.642,  -79.9,-153.1,  -3.9,  -1.8, -3.7,  0.7},
{ 3, 300, 299,  279,   0.400,   4.981,  270.622,  -81.7,-156.7,  -3.2,  -1.8, -3.7,  0.7},
{ 3, 301, 300,  279,   0.400,   4.981,  275.603,  -83.4,-160.4,  -2.5,  -1.8, -3.7,  0.7},
{ 3, 302, 301,  279,   0.400,   4.981,  280.583,  -85.2,-164.0,  -1.7,  -1.8, -3.7,  0.7},
{ 3, 303, 302,  279,   0.400,   4.981,  285.564,  -87.0,-167.7,  -1.0,  -1.8, -3.7,  0.7},
{ 3, 304, 303,  279,   0.400,   4.981,  290.545,  -88.8,-171.4,  -0.3,  -1.8, -3.7,  0.7},
{ 3, 305, 304,  279,   0.400,   4.981,  295.525,  -90.6,-175.0,   0.4,  -1.8, -3.7,  0.7},
{ 3, 306, 305,  279,   0.400,   4.981,  300.506,  -92.4,-178.7,   1.1,  -1.8, -3.7,  0.7},
{ 3, 307,  20,  299,   0.240,   4.786,   99.779,  -21.1, -48.2, -42.1,  -0.2, -3.0,  2.2},
{ 3, 308, 307,  299,   0.240,   4.786,  104.565,  -21.3, -51.2, -39.9,  -0.2, -3.0,  2.2},
{ 3, 309, 308,  299,   0.240,   4.786,  109.351,  -21.5, -54.2, -37.7,  -0.2, -3.0,  2.2},
{ 3, 310, 309,  299,   0.240,   4.786,  114.136,  -21.7, -57.2, -35.4,  -0.2, -3.0,  2.2},
{ 3, 311, 310,  299,   0.240,   4.786,  118.922,  -21.9, -60.2, -33.2,  -0.2, -3.0,  2.2},
{ 3, 312, 311,  299,   0.240,   4.786,  123.708,  -22.1, -63.2, -31.0,  -0.2, -3.0,  2.2},
{ 3, 313, 312,  299,   0.240,   4.786,  128.494,  -22.2, -66.2, -28.7,  -0.2, -3.0,  2.2},
{ 3, 314, 313,  299,   0.240,   4.786,  133.280,  -22.4, -69.2, -26.5,  -0.2, -3.0,  2.2},
{ 3, 315, 314,  299,   0.240,   4.786,  138.066,  -22.6, -72.2, -24.3,  -0.2, -3.0,  2.2},
{ 3, 316, 315,  299,   0.240,   4.786,  142.852,  -22.8, -75.2, -22.0,  -0.2, -3.0,  2.2},
{ 3, 317, 316,  299,   0.240,   4.786,  147.638,  -23.0, -78.2, -19.8,  -0.2, -3.0,  2.2},
{ 3, 318, 317,  299,   0.240,   4.786,  152.424,  -23.2, -81.2, -17.5,  -0.2, -3.0,  2.2},
{ 3, 319, 318,  299,   0.240,   4.786,  157.209,  -23.4, -84.2, -15.3,  -0.2, -3.0,  2.2},
{ 3, 320, 319,  299,   0.240,   4.786,  161.995,  -23.6, -87.2, -13.1,  -0.2, -3.0,  2.2},
{ 3, 321, 320,  299,   0.240,   4.786,  166.781,  -23.8, -90.2, -10.8,  -0.2, -3.0,  2.2},
{ 3, 322, 321,  299,   0.240,   4.786,  171.567,  -24.0, -93.2,  -8.6,  -0.2, -3.0,  2.2},
{ 3, 323, 322,  299,   0.240,   4.786,  176.353,  -24.2, -96.2,  -6.4,  -0.2, -3.0,  2.2},
{ 3, 324, 323,  299,   0.240,   4.786,  181.139,  -24.3, -99.2,  -4.1,  -0.2, -3.0,  2.2},
{ 3, 325, 324,  299,   0.240,   4.786,  185.925,  -24.5,-102.2,  -1.9,  -0.2, -3.0,  2.2},
{ 3, 326, 325,  299,   0.240,   4.786,  190.711,  -24.7,-105.2,   0.3,  -0.2, -3.0,  2.2},
{ 3, 327, 326,  316,   0.240,   4.998,  195.602,  -23.9,-108.8,   1.3,   1.8, -4.3, -0.2},
{ 3, 328, 327,  316,   0.240,   4.998,  200.600,  -22.1,-113.1,   1.1,   1.8, -4.3, -0.2},
{ 3, 329, 328,  316,   0.240,   4.998,  205.598,  -20.3,-117.4,   0.9,   1.8, -4.3, -0.2},
{ 3, 330, 329,  316,   0.240,   4.998,  210.595,  -18.5,-121.7,   0.7,   1.8, -4.3, -0.2},
{ 3, 331, 330,  316,   0.240,   4.998,  215.593,  -16.7,-126.0,   0.5,   1.8, -4.3, -0.2},
{ 3, 332, 331,  316,   0.240,   4.998,  220.591,  -14.8,-130.2,   0.3,   1.8, -4.3, -0.2},
{ 3, 333, 332,  316,   0.240,   4.998,  225.588,  -13.0,-134.5,   0.0,   1.8, -4.3, -0.2},
{ 3, 334, 333,  316,   0.240,   4.998,  230.586,  -11.2,-138.8,  -0.2,   1.8, -4.3, -0.2},
{ 3, 335, 334,  316,   0.240,   4.998,  235.584,   -9.4,-143.1,  -0.4,   1.8, -4.3, -0.2},
{ 3, 336, 335,  316,   0.240,   4.998,  240.582,   -7.6,-147.3,  -0.6,   1.8, -4.3, -0.2},
{ 3, 337, 336,  316,   0.240,   4.998,  245.579,   -5.8,-151.6,  -0.8,   1.8, -4.3, -0.2},
{ 3, 338, 337,  316,   0.240,   4.998,  250.577,   -4.0,-155.9,  -1.0,   1.8, -4.3, -0.2},
{ 3, 339, 338,  316,   0.240,   4.998,  255.575,   -2.1,-160.2,  -1.3,   1.8, -4.3, -0.2},
{ 3, 340, 339,  316,   0.240,   4.998,  260.572,   -0.3,-164.5,  -1.5,   1.8, -4.3, -0.2},
{ 3, 341, 340,  316,   0.240,   4.998,  265.570,    1.5,-168.7,  -1.7,   1.8, -4.3, -0.2},
{ 3, 342, 341,  316,   0.240,   4.998,  270.568,    3.3,-173.0,  -1.9,   1.8, -4.3, -0.2},
{ 3, 343, 342,  316,   0.240,   4.998,  275.565,    5.1,-177.3,  -2.1,   1.8, -4.3, -0.2},
{ 3, 344, 343,  316,   0.240,   4.998,  280.563,    6.9,-181.6,  -2.3,   1.8, -4.3, -0.2},
{ 3, 345, 344,  316,   0.240,   4.998,  285.561,    8.7,-185.9,  -2.6,   1.8, -4.3, -0.2},
{ 3, 346, 345,  316,   0.240,   4.998,  290.558,   10.6,-190.1,  -2.8,   1.8, -4.3, -0.2},
{ 3, 347, 346,  316,   0.240,   4.998,  295.556,   12.4,-194.4,  -3.0,   1.8, -4.3, -0.2},
{ 3, 348, 347,  316,   0.240,   4.998,  300.554,   14.2,-198.7,  -3.2,   1.8, -4.3, -0.2},
{ 3, 349, 348,  316,   0.240,   4.998,  305.551,   16.0,-203.0,  -3.4,   1.8, -4.3, -0.2},
{ 3, 350, 349,  316,   0.240,   4.998,  310.549,   17.8,-207.2,  -3.6,   1.8, -4.3, -0.2},
{ 3, 351, 350,  316,   0.240,   4.998,  315.547,   19.6,-211.5,  -3.9,   1.8, -4.3, -0.2},
{ 3, 352, 351,  316,   0.240,   4.998,  320.545,   21.4,-215.8,  -4.1,   1.8, -4.3, -0.2},
{ 3, 353, 352,  316,   0.240,   4.998,  325.542,   23.3,-220.1,  -4.3,   1.8, -4.3, -0.2},
{ 3, 354, 353,  316,   0.240,   4.998,  330.540,   25.1,-224.4,  -4.5,   1.8, -4.3, -0.2},
{ 3, 355, 354,  316,   0.240,   4.998,  335.538,   26.9,-228.6,  -4.7,   1.8, -4.3, -0.2},
{ 3, 356, 355,  316,   0.240,   4.998,  340.535,   28.7,-232.9,  -4.9,   1.8, -4.3, -0.2},
{ 3, 357, 326,  323,   0.400,   4.753,  195.480,  -25.0,-108.6,   2.0,  -0.4, -3.8,  1.0},
{ 3, 358, 357,  323,   0.400,   4.753,  200.233,  -25.4,-112.4,   3.0,  -0.4, -3.8,  1.0},
{ 3, 359, 358,  323,   0.400,   4.753,  204.986,  -25.8,-116.2,   4.1,  -0.4, -3.8,  1.0},
{ 3, 360, 359,  323,   0.400,   4.753,  209.740,  -26.2,-120.0,   5.1,  -0.4, -3.8,  1.0},
{ 3, 361, 360,  323,   0.400,   4.753,  214.493,  -26.6,-123.8,   6.1,  -0.4, -3.8,  1.0},
{ 3, 362, 361,  323,   0.400,   4.753,  219.246,  -27.0,-127.6,   7.2,  -0.4, -3.8,  1.0},
{ 3, 363, 362,  342,   0.247,   4.895,  224.070,  -27.0,-131.5,   7.4,   0.3, -3.9, -0.6},
{ 3, 364, 363,  342,   0.247,   4.895,  228.965,  -26.7,-135.4,   6.8,   0.3, -3.9, -0.6},
{ 3, 365, 364,  342,   0.247,   4.895,  233.861,  -26.4,-139.2,   6.2,   0.3, -3.9, -0.6},
{ 3, 366, 365,  342,   0.247,   4.895,  238.756,  -26.1,-143.1,   5.6,   0.3, -3.9, -0.6},
{ 3, 367, 366,  342,   0.247,   4.895,  243.652,  -25.7,-147.0,   5.0,   0.3, -3.9, -0.6},
{ 3, 368, 367,  342,   0.247,   4.895,  248.547,  -25.4,-150.8,   4.4,   0.3, -3.9, -0.6},
{ 3, 369, 368,  342,   0.247,   4.895,  253.442,  -25.1,-154.7,   3.9,   0.3, -3.9, -0.6},
{ 3, 370, 369,  342,   0.247,   4.895,  258.338,  -24.8,-158.6,   3.3,   0.3, -3.9, -0.6},
{ 3, 371, 370,  342,   0.247,   4.895,  263.233,  -24.5,-162.5,   2.7,   0.3, -3.9, -0.6},
{ 3, 372, 371,  342,   0.247,   4.895,  268.128,  -24.2,-166.3,   2.1,   0.3, -3.9, -0.6},
{ 3, 373, 372,  342,   0.247,   4.895,  273.024,  -23.9,-170.2,   1.5,   0.3, -3.9, -0.6},
{ 3, 374, 373,  342,   0.247,   4.895,  277.919,  -23.6,-174.1,   0.9,   0.3, -3.9, -0.6},
{ 3, 375, 374,  342,   0.247,   4.895,  282.814,  -23.3,-177.9,   0.3,   0.3, -3.9, -0.6},
{ 3, 376, 375,  342,   0.247,   4.895,  287.710,  -22.9,-181.8,  -0.3,   0.3, -3.9, -0.6},
{ 3, 377, 376,  342,   0.247,   4.895,  292.605,  -22.6,-185.7,  -0.9,   0.3, -3.9, -0.6},
{ 3, 378, 377,  342,   0.247,   4.895,  297.500,  -22.3,-189.6,  -1.5,   0.3, -3.9, -0.6},
{ 3, 379, 378,  342,   0.247,   4.895,  302.396,  -22.0,-193.4,  -2.1,   0.3, -3.9, -0.6},
{ 3, 380, 379,  342,   0.247,   4.895,  307.291,  -21.7,-197.3,  -2.7,   0.3, -3.9, -0.6},
{ 3, 381, 380,  342,   0.247,   4.895,  312.187,  -21.4,-201.2,  -3.2,   0.3, -3.9, -0.6},
{ 3, 382, 381,  342,   0.247,   4.895,  317.082,  -21.1,-205.1,  -3.8,   0.3, -3.9, -0.6},
{ 3, 383, 382,  342,   0.247,   4.895,  321.977,  -20.8,-208.9,  -4.4,   0.3, -3.9, -0.6},
{ 3, 384, 383,  342,   0.247,   4.895,  326.873,  -20.5,-212.8,  -5.0,   0.3, -3.9, -0.6},
{ 3, 385, 384,  342,   0.247,   4.895,  331.768,  -20.2,-216.7,  -5.6,   0.3, -3.9, -0.6},
{ 3, 386, 385,  342,   0.247,   4.895,  336.663,  -19.8,-220.5,  -6.2,   0.3, -3.9, -0.6},
{ 3, 387, 386,  342,   0.247,   4.895,  341.559,  -19.5,-224.4,  -6.8,   0.3, -3.9, -0.6},
{ 3, 388, 387,  342,   0.247,   4.895,  346.454,  -19.2,-228.3,  -7.4,   0.3, -3.9, -0.6},
{ 3, 389, 388,  342,   0.247,   4.895,  351.349,  -18.9,-232.2,  -8.0,   0.3, -3.9, -0.6},
{ 3, 390, 389,  342,   0.247,   4.895,  356.245,  -18.6,-236.0,  -8.6,   0.3, -3.9, -0.6},
{ 3, 391, 390,  342,   0.247,   4.895,  361.140,  -18.3,-239.9,  -9.2,   0.3, -3.9, -0.6},
{ 3, 392, 391,  342,   0.247,   4.895,  366.035,  -18.0,-243.8,  -9.8,   0.3, -3.9, -0.6},
{ 3, 393,  12,  349,   0.784,   5.024,   61.743,   -6.3, -29.9, -27.9,   1.5, -0.9, -2.9},
{ 3, 394, 393,  349,   0.784,   5.024,   66.767,   -4.7, -30.9, -30.8,   1.5, -0.9, -2.9},
{ 3, 395, 394,  349,   0.784,   5.024,   71.791,   -3.2, -31.8, -33.6,   1.5, -0.9, -2.9},
{ 3, 396, 395,  349,   0.784,   5.024,   76.815,   -1.6, -32.7, -36.5,   1.5, -0.9, -2.9},
{ 3, 397, 396,  349,   0.784,   5.024,   81.839,   -0.1, -33.6, -39.3,   1.5, -0.9, -2.9},
{ 3, 398, 397,  349,   0.784,   5.024,   86.864,    1.4, -34.6, -42.2,   1.5, -0.9, -2.9},
{ 3, 399, 398,  349,   0.784,   5.024,   91.888,    3.0, -35.5, -45.1,   1.5, -0.9, -2.9},
{ 3, 400, 399,  349,   0.784,   5.024,   96.912,    4.5, -36.4, -47.9,   1.5, -0.9, -2.9},
{ 3, 401, 400,  349,   0.784,   5.024,  101.936,    6.0, -37.3, -50.8,   1.5, -0.9, -2.9},
{ 3, 402, 401,  349,   0.784,   5.024,  106.960,    7.6, -38.3, -53.6,   1.5, -0.9, -2.9},
{ 3, 403, 402,  349,   0.784,   5.024,  111.985,    9.1, -39.2, -56.5,   1.5, -0.9, -2.9},
{ 3, 404, 403,  349,   0.784,   5.024,  117.009,   10.6, -40.1, -59.3,   1.5, -0.9, -2.9},
{ 3, 405, 404,  355,   0.565,   4.883,  121.963,   11.6, -41.6, -61.2,   0.5, -2.1, -1.0},
{ 3, 406, 405,  355,   0.565,   4.883,  126.846,   12.1, -43.8, -62.2,   0.5, -2.1, -1.0},
{ 3, 407, 406,  355,   0.565,   4.883,  131.729,   12.5, -45.9, -63.2,   0.5, -2.1, -1.0},
{ 3, 408, 407,  355,   0.565,   4.883,  136.612,   13.0, -48.1, -64.1,   0.5, -2.1, -1.0},
{ 3, 409, 408,  355,   0.565,   4.883,  141.495,   13.4, -50.2, -65.1,   0.5, -2.1, -1.0},
{ 3, 410, 409,  355,   0.565,   4.883,  146.378,   13.9, -52.4, -66.1,   0.5, -2.1, -1.0},
{ 3, 411, 410,  355,   0.565,   4.883,  151.261,   14.3, -54.5, -67.0,   0.5, -2.1, -1.0},
{ 3, 412, 411,  361,   0.717,   4.624,  156.015,   14.7, -56.3, -66.8,   0.3, -1.3,  1.4},
{ 3, 413, 412,  361,   0.717,   4.624,  160.639,   15.0, -57.6, -65.4,   0.3, -1.3,  1.4},
{ 3, 414, 413,  361,   0.717,   4.624,  165.263,   15.2, -58.9, -64.1,   0.3, -1.3,  1.4},
{ 3, 415, 414,  361,   0.717,   4.624,  169.888,   15.5, -60.3, -62.7,   0.3, -1.3,  1.4},
{ 3, 416, 415,  361,   0.717,   4.624,  174.512,   15.7, -61.6, -61.3,   0.3, -1.3,  1.4},
{ 3, 417, 416,  361,   0.717,   4.624,  179.136,   16.0, -63.0, -59.9,   0.3, -1.3,  1.4},
{ 3, 418, 417,  361,   0.717,   4.624,  183.761,   16.3, -64.3, -58.6,   0.3, -1.3,  1.4},
{ 3, 419, 418,  361,   0.717,   4.624,  188.385,   16.5, -65.6, -57.2,   0.3, -1.3,  1.4},
{ 3, 420, 419,  363,   0.560,   4.617,  193.006,   15.9, -66.9, -58.5,  -1.4, -1.3, -4.1},
{ 3, 421, 420,  363,   0.560,   4.617,  197.622,   14.5, -68.2, -62.6,  -1.4, -1.3, -4.1},
{ 3, 422, 421,  363,   0.560,   4.617,  202.239,   13.1, -69.4, -66.8,  -1.4, -1.3, -4.1},
{ 3, 423, 422,  363,   0.560,   4.617,  206.855,   11.7, -70.7, -70.8,  -1.4, -1.3, -4.1},
{ 3, 424, 423,  363,   0.560,   4.617,  211.472,   10.3, -71.9, -74.9,  -1.4, -1.3, -4.1},
{ 3, 425, 424,  381,   0.560,   4.906,  216.233,    9.3, -73.9, -77.2,  -0.5, -2.6, -0.3},
{ 3, 426, 425,  381,   0.560,   4.906,  221.139,    8.8, -76.5, -77.5,  -0.5, -2.6, -0.3},
{ 3, 427, 426,  381,   0.560,   4.906,  226.045,    8.3, -79.2, -77.9,  -0.5, -2.6, -0.3},
{ 3, 428, 427,  381,   0.560,   4.906,  230.951,    7.8, -81.8, -78.2,  -0.5, -2.6, -0.3},
{ 3, 429, 428,  381,   0.560,   4.906,  235.857,    7.4, -84.4, -78.6,  -0.5, -2.6, -0.3},
{ 3, 430, 429,  381,   0.560,   4.906,  240.762,    6.9, -87.1, -78.9,  -0.5, -2.6, -0.3},
{ 3, 431, 430,  381,   0.560,   4.906,  245.668,    6.4, -89.7, -79.3,  -0.5, -2.6, -0.3},
{ 3, 432, 431,  381,   0.560,   4.906,  250.574,    5.9, -92.4, -79.6,  -0.5, -2.6, -0.3},
{ 3, 433, 432,  381,   0.560,   4.906,  255.480,    5.4, -95.0, -80.0,  -0.5, -2.6, -0.3},
{ 3, 434, 433,  381,   0.560,   4.906,  260.386,    4.9, -97.7, -80.3,  -0.5, -2.6, -0.3},
{ 3, 435, 434,  381,   0.560,   4.906,  265.291,    4.4,-100.3, -80.7,  -0.5, -2.6, -0.3},
{ 3, 436, 435,  381,   0.560,   4.906,  270.197,    3.9,-102.9, -81.0,  -0.5, -2.6, -0.3},
{ 3, 437, 436,  381,   0.560,   4.906,  275.103,    3.4,-105.6, -81.4,  -0.5, -2.6, -0.3},
{ 3, 438, 437,  381,   0.560,   4.906,  280.009,    2.9,-108.2, -81.7,  -0.5, -2.6, -0.3},
{ 3, 439, 438,  381,   0.560,   4.906,  284.915,    2.4,-110.9, -82.1,  -0.5, -2.6, -0.3},
{ 3, 440, 439,  381,   0.560,   4.906,  289.821,    1.9,-113.5, -82.4,  -0.5, -2.6, -0.3},
{ 3, 441, 440,  381,   0.560,   4.906,  294.726,    1.4,-116.1, -82.8,  -0.5, -2.6, -0.3},
{ 3, 442, 441,  381,   0.560,   4.906,  299.632,    0.9,-118.8, -83.1,  -0.5, -2.6, -0.3},
{ 3, 443, 442,  381,   0.560,   4.906,  304.538,    0.4,-121.4, -83.5,  -0.5, -2.6, -0.3},
{ 3, 444, 443,  381,   0.560,   4.906,  309.444,   -0.1,-124.1, -83.8,  -0.5, -2.6, -0.3},
{ 3, 445, 424,  410,   0.400,   4.994,  216.277,   10.6, -73.5, -78.0,   2.1, -1.8, -2.1},
{ 3, 446, 445,  410,   0.400,   4.994,  221.272,   12.8, -75.3, -80.1,   2.1, -1.8, -2.1},
{ 3, 447, 446,  410,   0.400,   4.994,  226.266,   14.9, -77.1, -82.2,   2.1, -1.8, -2.1},
{ 3, 448, 447,  410,   0.400,   4.994,  231.260,   17.1, -78.9, -84.3,   2.1, -1.8, -2.1},
{ 3, 449, 448,  410,   0.400,   4.994,  236.255,   19.2, -80.7, -86.4,   2.1, -1.8, -2.1},
{ 3, 450, 449,  410,   0.400,   4.994,  241.249,   21.3, -82.5, -88.5,   2.1, -1.8, -2.1},
{ 3, 451, 450,  410,   0.400,   4.994,  246.243,   23.5, -84.3, -90.6,   2.1, -1.8, -2.1},
{ 3, 452, 451,  410,   0.400,   4.994,  251.237,   25.6, -86.1, -92.7,   2.1, -1.8, -2.1},
{ 3, 453, 452,  410,   0.400,   4.994,  256.232,   27.8, -87.9, -94.8,   2.1, -1.8, -2.1},
{ 3, 454, 453,  410,   0.400,   4.994,  261.226,   29.9, -89.7, -96.9,   2.1, -1.8, -2.1},
{ 3, 455, 454,  410,   0.400,   4.994,  266.220,   32.0, -91.5, -99.0,   2.1, -1.8, -2.1},
{ 3, 456, 455,  410,   0.400,   4.994,  271.214,   34.2, -93.3,-101.1,   2.1, -1.8, -2.1},
{ 3, 457, 456,  410,   0.400,   4.994,  276.209,   36.3, -95.1,-103.1,   2.1, -1.8, -2.1},
{ 3, 458, 457,  410,   0.400,   4.994,  281.203,   38.5, -96.9,-105.2,   2.1, -1.8, -2.1},
{ 3, 459, 458,  410,   0.400,   4.994,  286.197,   40.6, -98.7,-107.3,   2.1, -1.8, -2.1},
{ 3, 460, 459,  410,   0.400,   4.994,  291.191,   42.7,-100.5,-109.4,   2.1, -1.8, -2.1},
{ 3, 461, 460,  410,   0.400,   4.994,  296.186,   44.9,-102.3,-111.5,   2.1, -1.8, -2.1},
{ 3, 462, 461,  410,   0.400,   4.994,  301.180,   47.0,-104.1,-113.6,   2.1, -1.8, -2.1},
{ 3, 463, 462,  410,   0.400,   4.994,  306.174,   49.1,-105.9,-115.7,   2.1, -1.8, -2.1},
{ 3, 464, 463,  410,   0.400,   4.994,  311.169,   51.3,-107.7,-117.8,   2.1, -1.8, -2.1},
{ 3, 465, 464,  410,   0.400,   4.994,  316.163,   53.4,-109.5,-119.9,   2.1, -1.8, -2.1},
{ 3, 466, 465,  410,   0.400,   4.994,  321.157,   55.6,-111.3,-122.0,   2.1, -1.8, -2.1},
{ 3, 467, 466,  410,   0.400,   4.994,  326.151,   57.7,-113.1,-124.1,   2.1, -1.8, -2.1},
{ 3, 468, 467,  410,   0.400,   4.994,  331.146,   59.8,-114.9,-126.2,   2.1, -1.8, -2.1},
{ 3, 469, 468,  410,   0.400,   4.994,  336.140,   62.0,-116.7,-128.2,   2.1, -1.8, -2.1},
{ 3, 470, 469,  410,   0.400,   4.994,  341.134,   64.1,-118.5,-130.3,   2.1, -1.8, -2.1},
{ 3, 471, 470,  410,   0.400,   4.994,  346.128,   66.3,-120.3,-132.4,   2.1, -1.8, -2.1},
{ 3, 472, 471,  410,   0.400,   4.994,  351.123,   68.4,-122.1,-134.5,   2.1, -1.8, -2.1},
{ 3, 473, 472,  410,   0.400,   4.994,  356.117,   70.5,-124.0,-136.6,   2.1, -1.8, -2.1},
{ 3, 474, 473,  410,   0.400,   4.994,  361.111,   72.7,-125.8,-138.7,   2.1, -1.8, -2.1},
{ 3, 475, 419,  418,   0.560,   4.888,  193.141,   17.4, -67.0, -57.3,   1.5, -1.4, -1.6},
{ 3, 476, 475,  418,   0.560,   4.888,  198.030,   18.9, -68.4, -58.9,   1.5, -1.4, -1.6},
{ 3, 477, 476,  418,   0.560,   4.888,  202.918,   20.4, -69.8, -60.4,   1.5, -1.4, -1.6},
{ 3, 478, 477,  418,   0.560,   4.888,  207.806,   21.9, -71.2, -62.0,   1.5, -1.4, -1.6},
{ 3, 479, 478,  418,   0.560,   4.888,  212.695,   23.4, -72.6, -63.6,   1.5, -1.4, -1.6},
{ 3, 480, 479,  418,   0.560,   4.888,  217.583,   24.9, -74.0, -65.1,   1.5, -1.4, -1.6},
{ 3, 481, 480,  418,   0.560,   4.888,  222.471,   26.4, -75.4, -66.7,   1.5, -1.4, -1.6},
{ 3, 482, 481,  450,   0.400,   5.001,  227.416,   28.3, -77.1, -68.2,   2.4, -1.9, -1.5},
{ 3, 483, 482,  450,   0.400,   5.001,  232.417,   30.7, -78.9, -69.7,   2.4, -1.9, -1.5},
{ 3, 484, 483,  450,   0.400,   5.001,  237.418,   33.0, -80.8, -71.2,   2.4, -1.9, -1.5},
{ 3, 485, 484,  450,   0.400,   5.001,  242.419,   35.4, -82.7, -72.7,   2.4, -1.9, -1.5},
{ 3, 486, 485,  450,   0.400,   5.001,  247.420,   37.8, -84.5, -74.2,   2.4, -1.9, -1.5},
{ 3, 487, 486,  450,   0.400,   5.001,  252.421,   40.1, -86.4, -75.7,   2.4, -1.9, -1.5},
{ 3, 488, 487,  450,   0.400,   5.001,  257.422,   42.5, -88.3, -77.2,   2.4, -1.9, -1.5},
{ 3, 489, 488,  450,   0.400,   5.001,  262.423,   44.8, -90.1, -78.7,   2.4, -1.9, -1.5},
{ 3, 490, 489,  450,   0.400,   5.001,  267.424,   47.2, -92.0, -80.2,   2.4, -1.9, -1.5},
{ 3, 491, 490,  450,   0.400,   5.001,  272.425,   49.6, -93.9, -81.6,   2.4, -1.9, -1.5},
{ 3, 492, 491,  450,   0.400,   5.001,  277.426,   51.9, -95.7, -83.1,   2.4, -1.9, -1.5},
{ 3, 493, 492,  450,   0.400,   5.001,  282.427,   54.3, -97.6, -84.6,   2.4, -1.9, -1.5},
{ 3, 494, 493,  450,   0.400,   5.001,  287.428,   56.6, -99.5, -86.1,   2.4, -1.9, -1.5},
{ 3, 495, 494,  450,   0.400,   5.001,  292.429,   59.0,-101.3, -87.6,   2.4, -1.9, -1.5},
{ 3, 496, 495,  450,   0.400,   5.001,  297.430,   61.4,-103.2, -89.1,   2.4, -1.9, -1.5},
{ 3, 497, 496,  450,   0.400,   5.001,  302.431,   63.7,-105.1, -90.6,   2.4, -1.9, -1.5},
{ 3, 498, 497,  450,   0.400,   5.001,  307.432,   66.1,-106.9, -92.1,   2.4, -1.9, -1.5},
{ 3, 499, 498,  450,   0.400,   5.001,  312.433,   68.4,-108.8, -93.6,   2.4, -1.9, -1.5},
{ 3, 500, 499,  450,   0.400,   5.001,  317.434,   70.8,-110.7, -95.0,   2.4, -1.9, -1.5},
{ 3, 501, 500,  450,   0.400,   5.001,  322.435,   73.2,-112.5, -96.5,   2.4, -1.9, -1.5},
{ 3, 502, 501,  450,   0.400,   5.001,  327.436,   75.5,-114.4, -98.0,   2.4, -1.9, -1.5},
{ 3, 503, 502,  450,   0.400,   5.001,  332.437,   77.9,-116.3, -99.5,   2.4, -1.9, -1.5},
{ 3, 504, 503,  450,   0.400,   5.001,  337.438,   80.2,-118.1,-101.0,   2.4, -1.9, -1.5},
{ 3, 505, 504,  450,   0.400,   5.001,  342.439,   82.6,-120.0,-102.5,   2.4, -1.9, -1.5},
{ 3, 506, 505,  450,   0.400,   5.001,  347.440,   85.0,-121.9,-104.0,   2.4, -1.9, -1.5},
{ 3, 507, 506,  450,   0.400,   5.001,  352.442,   87.3,-123.7,-105.5,   2.4, -1.9, -1.5},
{ 3, 508, 507,  450,   0.400,   5.001,  357.443,   89.7,-125.6,-107.0,   2.4, -1.9, -1.5},
{ 3, 509, 508,  450,   0.400,   5.001,  362.444,   92.1,-127.5,-108.5,   2.4, -1.9, -1.5},
{ 3, 510, 509,  450,   0.400,   5.001,  367.445,   94.4,-129.3,-109.9,   2.4, -1.9, -1.5},
{ 3, 511, 510,  450,   0.400,   5.001,  372.446,   96.8,-131.2,-111.4,   2.4, -1.9, -1.5},
{ 3, 512, 511,  450,   0.400,   5.001,  377.447,   99.1,-133.1,-112.9,   2.4, -1.9, -1.5},
{ 3, 513, 512,  450,   0.400,   5.001,  382.448,  101.5,-134.9,-114.4,   2.4, -1.9, -1.5},
{ 3, 514, 513,  450,   0.400,   5.001,  387.449,  103.9,-136.8,-115.9,   2.4, -1.9, -1.5},
{ 3, 515, 514,  450,   0.400,   5.001,  392.450,  106.2,-138.7,-117.4,   2.4, -1.9, -1.5},
{ 3, 516, 515,  450,   0.400,   5.001,  397.451,  108.6,-140.5,-118.9,   2.4, -1.9, -1.5},
{ 3, 517, 516,  450,   0.400,   5.001,  402.452,  110.9,-142.4,-120.4,   2.4, -1.9, -1.5},
{ 3, 518, 517,  450,   0.400,   5.001,  407.453,  113.3,-144.3,-121.9,   2.4, -1.9, -1.5},
{ 3, 519, 518,  450,   0.400,   5.001,  412.454,  115.7,-146.1,-123.3,   2.4, -1.9, -1.5},
{ 3, 520, 519,  450,   0.400,   5.001,  417.455,  118.0,-148.0,-124.8,   2.4, -1.9, -1.5},
{ 3, 521, 520,  450,   0.400,   5.001,  422.456,  120.4,-149.9,-126.3,   2.4, -1.9, -1.5},
{ 3, 522, 521,  450,   0.400,   5.001,  427.457,  122.7,-151.7,-127.8,   2.4, -1.9, -1.5},
{ 3, 523, 522,  450,   0.400,   5.001,  432.458,  125.1,-153.6,-129.3,   2.4, -1.9, -1.5},
{ 3, 524, 523,  450,   0.400,   5.001,  437.459,  127.5,-155.5,-130.8,   2.4, -1.9, -1.5},
{ 3, 525, 524,  450,   0.400,   5.001,  442.460,  129.8,-157.3,-132.3,   2.4, -1.9, -1.5},
{ 3, 526, 525,  450,   0.400,   5.001,  447.461,  132.2,-159.2,-133.8,   2.4, -1.9, -1.5},
{ 3, 527, 526,  450,   0.400,   5.001,  452.462,  134.5,-161.1,-135.3,   2.4, -1.9, -1.5},
{ 3, 528, 481,  482,   0.400,   4.991,  227.411,   27.9, -77.2, -68.3,   1.5, -2.2, -1.6},
{ 3, 529, 528,  482,   0.400,   4.991,  232.401,   29.4, -79.4, -70.0,   1.5, -2.2, -1.6},
{ 3, 530, 529,  482,   0.400,   4.991,  237.392,   30.9, -81.6, -71.6,   1.5, -2.2, -1.6},
{ 3, 531, 530,  482,   0.400,   4.991,  242.383,   32.4, -83.7, -73.2,   1.5, -2.2, -1.6},
{ 3, 532, 531,  482,   0.400,   4.991,  247.374,   34.0, -85.9, -74.9,   1.5, -2.2, -1.6},
{ 3, 533, 532,  482,   0.400,   4.991,  252.364,   35.5, -88.1, -76.5,   1.5, -2.2, -1.6},
{ 3, 534, 533,  482,   0.400,   4.991,  257.355,   37.0, -90.2, -78.1,   1.5, -2.2, -1.6},
{ 3, 535, 534,  482,   0.400,   4.991,  262.346,   38.5, -92.4, -79.8,   1.5, -2.2, -1.6},
{ 3, 536, 535,  482,   0.400,   4.991,  267.336,   40.0, -94.6, -81.4,   1.5, -2.2, -1.6},
{ 3, 537, 536,  482,   0.400,   4.991,  272.327,   41.5, -96.7, -83.0,   1.5, -2.2, -1.6},
{ 3, 538, 537,  482,   0.400,   4.991,  277.318,   43.1, -98.9, -84.7,   1.5, -2.2, -1.6},
{ 3, 539, 538,  482,   0.400,   4.991,  282.309,   44.6,-101.1, -86.3,   1.5, -2.2, -1.6},
{ 3, 540, 539,  482,   0.400,   4.991,  287.299,   46.1,-103.2, -87.9,   1.5, -2.2, -1.6},
{ 3, 541, 540,  482,   0.400,   4.991,  292.290,   47.6,-105.4, -89.6,   1.5, -2.2, -1.6},
{ 3, 542, 541,  482,   0.400,   4.991,  297.281,   49.1,-107.6, -91.2,   1.5, -2.2, -1.6},
{ 3, 543, 542,  482,   0.400,   4.991,  302.271,   50.6,-109.7, -92.8,   1.5, -2.2, -1.6},
{ 3, 544, 543,  482,   0.400,   4.991,  307.262,   52.2,-111.9, -94.5,   1.5, -2.2, -1.6},
{ 3, 545, 544,  482,   0.400,   4.991,  312.253,   53.7,-114.0, -96.1,   1.5, -2.2, -1.6},
{ 3, 546, 545,  482,   0.400,   4.991,  317.244,   55.2,-116.2, -97.7,   1.5, -2.2, -1.6},
{ 3, 547, 546,  482,   0.400,   4.991,  322.234,   56.7,-118.4, -99.4,   1.5, -2.2, -1.6},
{ 3, 548, 547,  482,   0.400,   4.991,  327.225,   58.2,-120.5,-101.0,   1.5, -2.2, -1.6},
{ 3, 549, 548,  482,   0.400,   4.991,  332.216,   59.7,-122.7,-102.6,   1.5, -2.2, -1.6},
{ 3, 550, 549,  482,   0.400,   4.991,  337.206,   61.3,-124.9,-104.3,   1.5, -2.2, -1.6},
{ 3, 551, 550,  482,   0.400,   4.991,  342.197,   62.8,-127.0,-105.9,   1.5, -2.2, -1.6},
{ 3, 552, 551,  482,   0.400,   4.991,  347.188,   64.3,-129.2,-107.5,   1.5, -2.2, -1.6},
{ 3, 553, 552,  482,   0.400,   4.991,  352.178,   65.8,-131.4,-109.2,   1.5, -2.2, -1.6},
{ 3, 554, 553,  482,   0.400,   4.991,  357.169,   67.3,-133.5,-110.8,   1.5, -2.2, -1.6},
{ 3, 555, 554,  482,   0.400,   4.991,  362.160,   68.8,-135.7,-112.4,   1.5, -2.2, -1.6},
{ 3, 556, 555,  482,   0.400,   4.991,  367.151,   70.4,-137.9,-114.1,   1.5, -2.2, -1.6},
{ 3, 557, 556,  482,   0.400,   4.991,  372.141,   71.9,-140.0,-115.7,   1.5, -2.2, -1.6},
{ 3, 558, 557,  482,   0.400,   4.991,  377.132,   73.4,-142.2,-117.3,   1.5, -2.2, -1.6},
{ 3, 559, 558,  482,   0.400,   4.991,  382.123,   74.9,-144.4,-119.0,   1.5, -2.2, -1.6},
{ 3, 560, 559,  482,   0.400,   4.991,  387.113,   76.4,-146.5,-120.6,   1.5, -2.2, -1.6},
{ 3, 561, 560,  482,   0.400,   4.991,  392.104,   77.9,-148.7,-122.2,   1.5, -2.2, -1.6},
{ 3, 562, 561,  482,   0.400,   4.991,  397.095,   79.5,-150.9,-123.9,   1.5, -2.2, -1.6},
{ 3, 563, 562,  482,   0.400,   4.991,  402.086,   81.0,-153.0,-125.5,   1.5, -2.2, -1.6},
{ 3, 564, 563,  482,   0.400,   4.991,  407.076,   82.5,-155.2,-127.1,   1.5, -2.2, -1.6},
{ 3, 565, 564,  482,   0.400,   4.991,  412.067,   84.0,-157.4,-128.8,   1.5, -2.2, -1.6},
{ 3, 566, 565,  482,   0.400,   4.991,  417.058,   85.5,-159.5,-130.4,   1.5, -2.2, -1.6},
{ 3, 567, 566,  482,   0.400,   4.991,  422.048,   87.0,-161.7,-132.0,   1.5, -2.2, -1.6},
{ 3, 568, 567,  482,   0.400,   4.991,  427.039,   88.6,-163.9,-133.7,   1.5, -2.2, -1.6},
{ 3, 569, 568,  482,   0.400,   4.991,  432.030,   90.1,-166.0,-135.3,   1.5, -2.2, -1.6},
{ 3, 570, 569,  482,   0.400,   4.991,  437.020,   91.6,-168.2,-136.9,   1.5, -2.2, -1.6},
{ 4, 571,   0,  506,   2.150,   4.335,    2.168,  -11.7,  44.0,   4.3,  -0.5,  1.1,  4.2},
{ 4, 572, 571,  506,   2.150,   4.335,    6.503,  -12.2,  45.1,   8.5,  -0.5,  1.1,  4.2},
{ 4, 573, 572,  506,   2.150,   4.335,   10.838,  -12.7,  46.2,  12.7,  -0.5,  1.1,  4.2},
{ 4, 574, 573,  509,   2.384,   4.838,   15.424,  -13.9,  48.4,  13.8,  -2.0,  3.2, -1.9},
{ 4, 575, 574,  509,   2.384,   4.838,   20.262,  -15.9,  51.6,  11.9,  -2.0,  3.2, -1.9},
{ 4, 576, 575,  509,   2.384,   4.838,   25.099,  -17.9,  54.8,  10.0,  -2.0,  3.2, -1.9},
{ 4, 577, 576,  510,   1.510,   0.925,   27.981,  -19.0,  56.8,   9.1,  -0.2,  0.9,  0.3},
{ 4, 578, 577,  522,   0.901,   4.915,   30.901,  -20.2,  58.7,   9.9,  -2.3,  3.0,  1.3},
{ 4, 579, 578,  522,   0.901,   4.915,   35.815,  -22.4,  61.8,  11.2,  -2.3,  3.0,  1.3},
{ 4, 580, 579,  522,   0.901,   4.915,   40.730,  -24.7,  64.8,  12.5,  -2.3,  3.0,  1.3},
{ 4, 581, 580,  522,   0.901,   4.915,   45.645,  -26.9,  67.8,  13.8,  -2.3,  3.0,  1.3},
{ 4, 582, 581,  522,   0.901,   4.915,   50.559,  -29.2,  70.9,  15.1,  -2.3,  3.0,  1.3},
{ 4, 583, 582,  522,   0.901,   4.915,   55.474,  -31.5,  73.9,  16.4,  -2.3,  3.0,  1.3},
{ 4, 584, 583,  522,   0.901,   4.915,   60.388,  -33.7,  76.9,  17.7,  -2.3,  3.0,  1.3},
{ 4, 585, 584,  522,   0.901,   4.915,   65.303,  -36.0,  79.9,  19.0,  -2.3,  3.0,  1.3},
{ 4, 586, 585,  522,   0.901,   4.915,   70.218,  -38.2,  83.0,  20.3,  -2.3,  3.0,  1.3},
{ 4, 587, 586,  522,   0.901,   4.915,   75.132,  -40.5,  86.0,  21.6,  -2.3,  3.0,  1.3},
{ 4, 588, 587,  522,   0.901,   4.915,   80.047,  -42.7,  89.0,  22.9,  -2.3,  3.0,  1.3},
{ 4, 589, 588,  522,   0.901,   4.915,   84.962,  -45.0,  92.1,  24.2,  -2.3,  3.0,  1.3},
{ 4, 590, 589,  522,   0.901,   4.915,   89.876,  -47.2,  95.1,  25.5,  -2.3,  3.0,  1.3},
{ 4, 591, 590,  522,   0.901,   4.915,   94.791,  -49.5,  98.1,  26.8,  -2.3,  3.0,  1.3},
{ 4, 592, 591,  525,   0.880,   4.694,   99.595,  -51.6, 101.6,  28.0,  -2.0,  4.0,  0.9},
{ 4, 593, 592,  525,   0.880,   4.694,  104.289,  -53.5, 105.6,  28.9,  -2.0,  4.0,  0.9},
{ 4, 594, 593,  525,   0.880,   4.694,  108.982,  -55.5, 109.6,  29.8,  -2.0,  4.0,  0.9},
{ 4, 595, 594,  525,   0.880,   4.694,  113.676,  -57.5, 113.5,  30.8,  -2.0,  4.0,  0.9},
{ 4, 596, 595,  535,   0.400,   4.973,  118.509,  -59.4, 117.3,  31.4,  -2.0,  3.6,  0.3},
{ 4, 597, 596,  535,   0.400,   4.973,  123.483,  -61.4, 121.0,  31.7,  -2.0,  3.6,  0.3},
{ 4, 598, 597,  535,   0.400,   4.973,  128.456,  -63.3, 124.6,  31.9,  -2.0,  3.6,  0.3},
{ 4, 599, 598,  535,   0.400,   4.973,  133.429,  -65.3, 128.2,  32.2,  -2.0,  3.6,  0.3},
{ 4, 600, 599,  535,   0.400,   4.973,  138.402,  -67.3, 131.8,  32.5,  -2.0,  3.6,  0.3},
{ 4, 601, 600,  535,   0.400,   4.973,  143.375,  -69.2, 135.5,  32.7,  -2.0,  3.6,  0.3},
{ 4, 602, 601,  535,   0.400,   4.973,  148.348,  -71.2, 139.1,  33.0,  -2.0,  3.6,  0.3},
{ 4, 603, 602,  535,   0.400,   4.973,  153.321,  -73.1, 142.7,  33.3,  -2.0,  3.6,  0.3},
{ 4, 604, 603,  535,   0.400,   4.973,  158.295,  -75.1, 146.3,  33.6,  -2.0,  3.6,  0.3},
{ 4, 605, 604,  535,   0.400,   4.973,  163.268,  -77.0, 150.0,  33.8,  -2.0,  3.6,  0.3},
{ 4, 606, 605,  535,   0.400,   4.973,  168.241,  -79.0, 153.6,  34.1,  -2.0,  3.6,  0.3},
{ 4, 607, 606,  535,   0.400,   4.973,  173.214,  -81.0, 157.2,  34.4,  -2.0,  3.6,  0.3},
{ 4, 608, 595,  538,   0.400,   3.810,  117.928,  -59.2, 115.6,  32.8,  -1.5,  0.1,  3.1},
{ 4, 609, 608,  538,   0.400,   3.810,  121.737,  -60.7, 115.7,  35.9,  -1.5,  0.1,  3.1},
{ 4, 610, 609,  561,   0.242,   4.848,  126.066,  -62.2, 116.0,  37.3,  -1.6,  0.5, -0.3},
{ 4, 611, 610,  561,   0.242,   4.848,  130.915,  -63.9, 116.6,  37.0,  -1.6,  0.5, -0.3},
{ 4, 612, 611,  561,   0.242,   4.848,  135.763,  -65.5, 117.1,  36.7,  -1.6,  0.5, -0.3},
{ 4, 613, 612,  561,   0.242,   4.848,  140.611,  -67.1, 117.6,  36.4,  -1.6,  0.5, -0.3},
{ 4, 614, 613,  561,   0.242,   4.848,  145.460,  -68.8, 118.2,  36.1,  -1.6,  0.5, -0.3},
{ 4, 615, 614,  561,   0.242,   4.848,  150.308,  -70.4, 118.7,  35.8,  -1.6,  0.5, -0.3},
{ 4, 616, 615,  561,   0.242,   4.848,  155.157,  -72.0, 119.3,  35.4,  -1.6,  0.5, -0.3},
{ 4, 617, 616,  561,   0.242,   4.848,  160.005,  -73.7, 119.8,  35.1,  -1.6,  0.5, -0.3},
{ 4, 618, 617,  561,   0.242,   4.848,  164.853,  -75.3, 120.4,  34.8,  -1.6,  0.5, -0.3},
{ 4, 619, 618,  561,   0.242,   4.848,  169.702,  -77.0, 120.9,  34.5,  -1.6,  0.5, -0.3},
{ 4, 620, 619,  561,   0.242,   4.848,  174.550,  -78.6, 121.4,  34.2,  -1.6,  0.5, -0.3},
{ 4, 621, 620,  561,   0.242,   4.848,  179.399,  -80.2, 122.0,  33.9,  -1.6,  0.5, -0.3},
{ 4, 622, 621,  561,   0.242,   4.848,  184.247,  -81.9, 122.5,  33.5,  -1.6,  0.5, -0.3},
{ 4, 623, 622,  561,   0.242,   4.848,  189.095,  -83.5, 123.1,  33.2,  -1.6,  0.5, -0.3},
{ 4, 624, 623,  561,   0.242,   4.848,  193.944,  -85.1, 123.6,  32.9,  -1.6,  0.5, -0.3},
{ 4, 625, 624,  561,   0.242,   4.848,  198.792,  -86.8, 124.2,  32.6,  -1.6,  0.5, -0.3},
{ 4, 626, 625,  561,   0.242,   4.848,  203.640,  -88.4, 124.7,  32.3,  -1.6,  0.5, -0.3},
{ 4, 627, 626,  561,   0.242,   4.848,  208.489,  -90.1, 125.3,  31.9,  -1.6,  0.5, -0.3},
{ 4, 628, 627,  561,   0.242,   4.848,  213.337,  -91.7, 125.8,  31.6,  -1.6,  0.5, -0.3},
{ 4, 629, 628,  561,   0.242,   4.848,  218.186,  -93.3, 126.3,  31.3,  -1.6,  0.5, -0.3},
{ 4, 630, 629,  561,   0.242,   4.848,  223.034,  -95.0, 126.9,  31.0,  -1.6,  0.5, -0.3},
{ 4, 631, 630,  561,   0.242,   4.848,  227.882,  -96.6, 127.4,  30.7,  -1.6,  0.5, -0.3},
{ 4, 632, 631,  561,   0.242,   4.848,  232.731,  -98.2, 128.0,  30.4,  -1.6,  0.5, -0.3},
{ 4, 633, 632,  561,   0.242,   4.848,  237.579,  -99.9, 128.5,  30.0,  -1.6,  0.5, -0.3},
{ 4, 634, 633,  561,   0.242,   4.848,  242.428, -101.5, 129.1,  29.7,  -1.6,  0.5, -0.3},
{ 4, 635, 634,  561,   0.242,   4.848,  247.276, -103.1, 129.6,  29.4,  -1.6,  0.5, -0.3},
{ 4, 636, 591,  564,   0.560,   3.717,   99.107,  -49.9, 100.0,  28.9,   1.5,  0.8,  2.8},
{ 4, 637, 636,  564,   0.560,   3.717,  102.824,  -48.4, 100.8,  31.6,   1.5,  0.8,  2.8},
{ 4, 638, 637,  564,   0.560,   3.717,  106.542,  -46.9, 101.6,  34.4,   1.5,  0.8,  2.8},
{ 4, 639, 638,  566,   0.696,   4.772,  110.787,  -45.6, 102.8,  36.8,   1.1,  1.5,  2.0},
{ 4, 640, 639,  566,   0.696,   4.772,  115.559,  -44.6, 104.2,  38.8,   1.1,  1.5,  2.0},
{ 4, 641, 640,  566,   0.696,   4.772,  120.331,  -43.5, 105.7,  40.8,   1.1,  1.5,  2.0},
{ 4, 642, 641,  569,   0.720,   3.949,  124.692,  -42.7, 106.6,  41.5,   0.5,  0.4, -0.5},
{ 4, 643, 642,  569,   0.720,   3.949,  128.641,  -42.2, 107.0,  41.0,   0.5,  0.4, -0.5},
{ 4, 644, 643,  569,   0.720,   3.949,  132.590,  -41.7, 107.4,  40.5,   0.5,  0.4, -0.5},
{ 4, 645, 644,  569,   0.720,   3.949,  136.539,  -41.1, 107.8,  40.0,   0.5,  0.4, -0.5},
{ 4, 646, 645,  576,   0.593,   4.563,  140.795,  -40.9, 108.9,  40.6,  -0.0,  1.8,  1.6},
{ 4, 647, 646,  576,   0.593,   4.563,  145.358,  -40.9, 110.7,  42.2,  -0.0,  1.8,  1.6},
{ 4, 648, 647,  576,   0.593,   4.563,  149.922,  -40.9, 112.5,  43.8,  -0.0,  1.8,  1.6},
{ 4, 649, 648,  576,   0.593,   4.563,  154.485,  -40.9, 114.3,  45.5,  -0.0,  1.8,  1.6},
{ 4, 650, 649,  576,   0.593,   4.563,  159.048,  -40.9, 116.1,  47.1,  -0.0,  1.8,  1.6},
{ 4, 651, 650,  576,   0.593,   4.563,  163.611,  -40.9, 117.9,  48.8,  -0.0,  1.8,  1.6},
{ 4, 652, 651,  576,   0.593,   4.563,  168.174,  -40.9, 119.8,  50.4,  -0.0,  1.8,  1.6},
{ 4, 653, 652,  576,   0.593,   4.563,  172.737,  -40.9, 121.6,  52.0,  -0.0,  1.8,  1.6},
{ 4, 654, 653,  576,   0.593,   4.563,  177.301,  -40.9, 123.4,  53.7,  -0.0,  1.8,  1.6},
{ 4, 655, 654,  584,   0.560,   4.998,  182.081,  -41.2, 125.1,  55.3,  -0.7,  1.6,  1.5},
{ 4, 656, 655,  584,   0.560,   4.998,  187.079,  -41.9, 126.7,  56.8,  -0.7,  1.6,  1.5},
{ 4, 657, 656,  584,   0.560,   4.998,  192.077,  -42.6, 128.4,  58.3,  -0.7,  1.6,  1.5},
{ 4, 658, 657,  584,   0.560,   4.998,  197.074,  -43.3, 130.0,  59.8,  -0.7,  1.6,  1.5},
{ 4, 659, 658,  584,   0.560,   4.998,  202.072,  -44.0, 131.7,  61.3,  -0.7,  1.6,  1.5},
{ 4, 660, 659,  584,   0.560,   4.998,  207.070,  -44.6, 133.3,  62.8,  -0.7,  1.6,  1.5},
{ 4, 661, 660,  584,   0.560,   4.998,  212.068,  -45.3, 135.0,  64.3,  -0.7,  1.6,  1.5},
{ 4, 662, 661,  584,   0.560,   4.998,  217.065,  -46.0, 136.6,  65.8,  -0.7,  1.6,  1.5},
{ 4, 663, 662,  584,   0.560,   4.998,  222.063,  -46.7, 138.3,  67.3,  -0.7,  1.6,  1.5},
{ 4, 664, 663,  584,   0.560,   4.998,  227.061,  -47.3, 139.9,  68.8,  -0.7,  1.6,  1.5},
{ 4, 665, 664,  603,   0.403,   4.984,  232.052,  -47.6, 141.4,  70.2,   0.1,  1.4,  1.4},
{ 4, 666, 665,  603,   0.403,   4.984,  237.035,  -47.5, 142.8,  71.5,   0.1,  1.4,  1.4},
{ 4, 667, 666,  603,   0.403,   4.984,  242.019,  -47.5, 144.1,  72.9,   0.1,  1.4,  1.4},
{ 4, 668, 667,  603,   0.403,   4.984,  247.003,  -47.4, 145.5,  74.3,   0.1,  1.4,  1.4},
{ 4, 669, 668,  603,   0.403,   4.984,  251.986,  -47.3, 146.9,  75.6,   0.1,  1.4,  1.4},
{ 4, 670, 669,  603,   0.403,   4.984,  256.970,  -47.2, 148.2,  77.0,   0.1,  1.4,  1.4},
{ 4, 671, 670,  603,   0.403,   4.984,  261.954,  -47.2, 149.6,  78.3,   0.1,  1.4,  1.4},
{ 4, 672, 671,  603,   0.403,   4.984,  266.937,  -47.1, 151.0,  79.7,   0.1,  1.4,  1.4},
{ 4, 673, 672,  603,   0.403,   4.984,  271.921,  -47.0, 152.3,  81.0,   0.1,  1.4,  1.4},
{ 4, 674, 673,  603,   0.403,   4.984,  276.905,  -46.9, 153.7,  82.4,   0.1,  1.4,  1.4},
{ 4, 675, 674,  603,   0.403,   4.984,  281.888,  -46.8, 155.0,  83.8,   0.1,  1.4,  1.4},
{ 4, 676, 675,  603,   0.403,   4.984,  286.872,  -46.8, 156.4,  85.1,   0.1,  1.4,  1.4},
{ 4, 677, 676,  603,   0.403,   4.984,  291.856,  -46.7, 157.8,  86.5,   0.1,  1.4,  1.4},
{ 4, 678, 677,  603,   0.403,   4.984,  296.839,  -46.6, 159.1,  87.8,   0.1,  1.4,  1.4},
{ 4, 679, 678,  603,   0.403,   4.984,  301.823,  -46.5, 160.5,  89.2,   0.1,  1.4,  1.4},
{ 4, 680, 679,  603,   0.403,   4.984,  306.807,  -46.4, 161.9,  90.6,   0.1,  1.4,  1.4},
{ 4, 681, 680,  603,   0.403,   4.984,  311.790,  -46.4, 163.2,  91.9,   0.1,  1.4,  1.4},
{ 4, 682, 681,  603,   0.403,   4.984,  316.774,  -46.3, 164.6,  93.3,   0.1,  1.4,  1.4},
{ 4, 683, 654,  609,   0.560,   4.430,  181.797,  -41.2, 126.2,  54.0,  -0.5,  3.9, -0.9},
{ 4, 684, 683,  609,   0.560,   4.430,  186.227,  -41.6, 130.1,  53.1,  -0.5,  3.9, -0.9},
{ 4, 685, 684,  609,   0.560,   4.430,  190.657,  -42.1, 134.1,  52.2,  -0.5,  3.9, -0.9},
{ 4, 686, 685,  609,   0.560,   4.430,  195.087,  -42.6, 138.0,  51.3,  -0.5,  3.9, -0.9},
{ 4, 687, 686,  609,   0.560,   4.430,  199.517,  -43.1, 141.9,  50.3,  -0.5,  3.9, -0.9},
{ 4, 688, 687,  609,   0.560,   4.430,  203.947,  -43.6, 145.8,  49.4,  -0.5,  3.9, -0.9},
{ 4, 689, 688,  609,   0.560,   4.430,  208.377,  -44.1, 149.7,  48.5,  -0.5,  3.9, -0.9},
{ 4, 690, 689,  619,   0.400,   4.667,  212.925,  -44.5, 153.1,  49.1,  -0.3,  2.8,  2.2},
{ 4, 691, 690,  619,   0.400,   4.667,  217.592,  -44.9, 155.8,  51.3,  -0.3,  2.8,  2.2},
{ 4, 692, 691,  619,   0.400,   4.667,  222.258,  -45.2, 158.6,  53.5,  -0.3,  2.8,  2.2},
{ 4, 693, 692,  619,   0.400,   4.667,  226.925,  -45.5, 161.3,  55.7,  -0.3,  2.8,  2.2},
{ 4, 694, 693,  619,   0.400,   4.667,  231.592,  -45.8, 164.1,  58.0,  -0.3,  2.8,  2.2},
{ 4, 695, 694,  619,   0.400,   4.667,  236.258,  -46.2, 166.8,  60.2,  -0.3,  2.8,  2.2},
{ 4, 696, 695,  619,   0.400,   4.667,  240.925,  -46.5, 169.6,  62.4,  -0.3,  2.8,  2.2},
{ 4, 697, 696,  619,   0.400,   4.667,  245.591,  -46.8, 172.4,  64.6,  -0.3,  2.8,  2.2},
{ 4, 698, 697,  642,   0.244,   5.000,  250.424,  -47.3, 175.7,  66.6,  -0.5,  3.9,  1.7},
{ 4, 699, 698,  642,   0.244,   5.000,  255.424,  -47.8, 179.6,  68.3,  -0.5,  3.9,  1.7},
{ 4, 700, 699,  642,   0.244,   5.000,  260.423,  -48.4, 183.5,  70.0,  -0.5,  3.9,  1.7},
{ 4, 701, 700,  642,   0.244,   5.000,  265.423,  -48.9, 187.5,  71.7,  -0.5,  3.9,  1.7},
{ 4, 702, 701,  642,   0.244,   5.000,  270.423,  -49.5, 191.4,  73.5,  -0.5,  3.9,  1.7},
{ 4, 703, 702,  642,   0.244,   5.000,  275.422,  -50.0, 195.3,  75.2,  -0.5,  3.9,  1.7},
{ 4, 704, 703,  642,   0.244,   5.000,  280.422,  -50.5, 199.2,  76.9,  -0.5,  3.9,  1.7},
{ 4, 705, 704,  642,   0.244,   5.000,  285.421,  -51.1, 203.2,  78.6,  -0.5,  3.9,  1.7},
{ 4, 706, 705,  642,   0.244,   5.000,  290.421,  -51.6, 207.1,  80.4,  -0.5,  3.9,  1.7},
{ 4, 707, 706,  642,   0.244,   5.000,  295.421,  -52.2, 211.0,  82.1,  -0.5,  3.9,  1.7},
{ 4, 708, 707,  642,   0.244,   5.000,  300.420,  -52.7, 214.9,  83.8,  -0.5,  3.9,  1.7},
{ 4, 709, 708,  642,   0.244,   5.000,  305.420,  -53.3, 218.8,  85.5,  -0.5,  3.9,  1.7},
{ 4, 710, 709,  642,   0.244,   5.000,  310.419,  -53.8, 222.8,  87.3,  -0.5,  3.9,  1.7},
{ 4, 711, 710,  642,   0.244,   5.000,  315.419,  -54.4, 226.7,  89.0,  -0.5,  3.9,  1.7},
{ 4, 712, 711,  642,   0.244,   5.000,  320.419,  -54.9, 230.6,  90.7,  -0.5,  3.9,  1.7},
{ 4, 713, 712,  642,   0.244,   5.000,  325.418,  -55.4, 234.5,  92.4,  -0.5,  3.9,  1.7},
{ 4, 714, 713,  642,   0.244,   5.000,  330.418,  -56.0, 238.5,  94.2,  -0.5,  3.9,  1.7},
{ 4, 715, 714,  642,   0.244,   5.000,  335.417,  -56.5, 242.4,  95.9,  -0.5,  3.9,  1.7},
{ 4, 716, 715,  642,   0.244,   5.000,  340.417,  -57.1, 246.3,  97.6,  -0.5,  3.9,  1.7},
{ 4, 717, 716,  642,   0.244,   5.000,  345.417,  -57.6, 250.2,  99.3,  -0.5,  3.9,  1.7},
{ 4, 718, 689,  674,   0.400,   4.938,  213.061,  -43.7, 153.0,  48.9,   1.3,  2.6,  1.8},
{ 4, 719, 718,  674,   0.400,   4.938,  218.000,  -42.4, 155.5,  50.6,   1.3,  2.6,  1.8},
{ 4, 720, 719,  674,   0.400,   4.938,  222.938,  -41.1, 158.1,  52.4,   1.3,  2.6,  1.8},
{ 4, 721, 720,  674,   0.400,   4.938,  227.877,  -39.8, 160.6,  54.1,   1.3,  2.6,  1.8},
{ 4, 722, 721,  674,   0.400,   4.938,  232.815,  -38.5, 163.2,  55.9,   1.3,  2.6,  1.8},
{ 4, 723, 722,  674,   0.400,   4.938,  237.753,  -37.1, 165.7,  57.6,   1.3,  2.6,  1.8},
{ 4, 724, 723,  674,   0.400,   4.938,  242.692,  -35.8, 168.3,  59.4,   1.3,  2.6,  1.8},
{ 4, 725, 724,  674,   0.400,   4.938,  247.630,  -34.5, 170.9,  61.1,   1.3,  2.6,  1.8},
{ 4, 726, 725,  674,   0.400,   4.938,  252.569,  -33.2, 173.4,  62.9,   1.3,  2.6,  1.8},
{ 4, 727, 726,  674,   0.400,   4.938,  257.507,  -31.9, 176.0,  64.6,   1.3,  2.6,  1.8},
{ 4, 728, 727,  674,   0.400,   4.938,  262.446,  -30.6, 178.5,  66.4,   1.3,  2.6,  1.8},
{ 4, 729, 728,  674,   0.400,   4.938,  267.384,  -29.3, 181.1,  68.1,   1.3,  2.6,  1.8},
{ 4, 730, 729,  674,   0.400,   4.938,  272.322,  -28.0, 183.6,  69.9,   1.3,  2.6,  1.8},
{ 4, 731, 730,  674,   0.400,   4.938,  277.261,  -26.6, 186.2,  71.6,   1.3,  2.6,  1.8},
{ 4, 732, 731,  674,   0.400,   4.938,  282.199,  -25.3, 188.8,  73.4,   1.3,  2.6,  1.8},
{ 4, 733, 732,  674,   0.400,   4.938,  287.138,  -24.0, 191.3,  75.1,   1.3,  2.6,  1.8},
{ 4, 734, 733,  674,   0.400,   4.938,  292.076,  -22.7, 193.9,  76.9,   1.3,  2.6,  1.8},
{ 4, 735, 734,  674,   0.400,   4.938,  297.015,  -21.4, 196.4,  78.6,   1.3,  2.6,  1.8},
{ 4, 736, 735,  674,   0.400,   4.938,  301.953,  -20.1, 199.0,  80.4,   1.3,  2.6,  1.8},
{ 4, 737, 736,  674,   0.400,   4.938,  306.891,  -18.8, 201.5,  82.1,   1.3,  2.6,  1.8},
{ 4, 738, 737,  674,   0.400,   4.938,  311.830,  -17.5, 204.1,  83.9,   1.3,  2.6,  1.8},
{ 4, 739, 738,  674,   0.400,   4.938,  316.768,  -16.1, 206.6,  85.6,   1.3,  2.6,  1.8},
{ 4, 740, 739,  674,   0.400,   4.938,  321.707,  -14.8, 209.2,  87.4,   1.3,  2.6,  1.8},
{ 4, 741, 740,  674,   0.400,   4.938,  326.645,  -13.5, 211.8,  89.1,   1.3,  2.6,  1.8},
{ 4, 742, 741,  674,   0.400,   4.938,  331.584,  -12.2, 214.3,  90.9,   1.3,  2.6,  1.8},
{ 4, 743, 742,  674,   0.400,   4.938,  336.522,  -10.9, 216.9,  92.6,   1.3,  2.6,  1.8},
{ 4, 744, 743,  674,   0.400,   4.938,  341.460,   -9.6, 219.4,  94.4,   1.3,  2.6,  1.8},
{ 4, 745, 645,  689,   0.240,   4.812,  140.920,  -41.4, 108.8,  40.6,  -1.1,  1.5,  1.7},
{ 4, 746, 745,  689,   0.240,   4.812,  145.731,  -42.5, 110.3,  42.3,  -1.1,  1.5,  1.7},
{ 4, 747, 746,  689,   0.240,   4.812,  150.543,  -43.7, 111.8,  43.9,  -1.1,  1.5,  1.7},
{ 4, 748, 747,  689,   0.240,   4.812,  155.355,  -44.8, 113.3,  45.6,  -1.1,  1.5,  1.7},
{ 4, 749, 748,  689,   0.240,   4.812,  160.167,  -45.9, 114.8,  47.3,  -1.1,  1.5,  1.7},
{ 4, 750, 749,  689,   0.240,   4.812,  164.979,  -47.0, 116.3,  48.9,  -1.1,  1.5,  1.7},
{ 4, 751, 750,  689,   0.240,   4.812,  169.791,  -48.1, 117.8,  50.6,  -1.1,  1.5,  1.7},
{ 4, 752, 751,  689,   0.240,   4.812,  174.603,  -49.2, 119.3,  52.3,  -1.1,  1.5,  1.7},
{ 4, 753, 752,  689,   0.240,   4.812,  179.415,  -50.4, 120.8,  54.0,  -1.1,  1.5,  1.7},
{ 4, 754, 753,  689,   0.240,   4.812,  184.226,  -51.5, 122.3,  55.6,  -1.1,  1.5,  1.7},
{ 4, 755, 754,  689,   0.240,   4.812,  189.038,  -52.6, 123.8,  57.3,  -1.1,  1.5,  1.7},
{ 4, 756, 755,  689,   0.240,   4.812,  193.850,  -53.7, 125.3,  59.0,  -1.1,  1.5,  1.7},
{ 4, 757, 756,  689,   0.240,   4.812,  198.662,  -54.8, 126.8,  60.6,  -1.1,  1.5,  1.7},
{ 4, 758, 757,  689,   0.240,   4.812,  203.474,  -55.9, 128.3,  62.3,  -1.1,  1.5,  1.7},
{ 4, 759, 758,  689,   0.240,   4.812,  208.286,  -57.0, 129.8,  64.0,  -1.1,  1.5,  1.7},
{ 4, 760, 759,  689,   0.240,   4.812,  213.098,  -58.2, 131.4,  65.7,  -1.1,  1.5,  1.7},
{ 4, 761, 760,  689,   0.240,   4.812,  217.910,  -59.3, 132.9,  67.3,  -1.1,  1.5,  1.7},
{ 4, 762, 761,  689,   0.240,   4.812,  222.721,  -60.4, 134.4,  69.0,  -1.1,  1.5,  1.7},
{ 4, 763, 762,  689,   0.240,   4.812,  227.533,  -61.5, 135.9,  70.7,  -1.1,  1.5,  1.7},
{ 4, 764, 763,  721,   0.400,   4.865,  232.372,  -63.1, 137.3,  72.2,  -2.1,  1.4,  1.4},
{ 4, 765, 764,  721,   0.400,   4.865,  237.236,  -65.2, 138.8,  73.6,  -2.1,  1.4,  1.4},
{ 4, 766, 765,  721,   0.400,   4.865,  242.101,  -67.3, 140.2,  75.1,  -2.1,  1.4,  1.4},
{ 4, 767, 766,  721,   0.400,   4.865,  246.966,  -69.4, 141.6,  76.5,  -2.1,  1.4,  1.4},
{ 4, 768, 767,  721,   0.400,   4.865,  251.831,  -71.5, 143.1,  77.9,  -2.1,  1.4,  1.4},
{ 4, 769, 768,  721,   0.400,   4.865,  256.695,  -73.6, 144.5,  79.3,  -2.1,  1.4,  1.4},
{ 4, 770, 769,  721,   0.400,   4.865,  261.560,  -75.8, 145.9,  80.8,  -2.1,  1.4,  1.4},
{ 4, 771, 770,  721,   0.400,   4.865,  266.425,  -77.9, 147.3,  82.2,  -2.1,  1.4,  1.4},
{ 4, 772, 771,  721,   0.400,   4.865,  271.290,  -80.0, 148.8,  83.6,  -2.1,  1.4,  1.4},
{ 4, 773, 772,  721,   0.400,   4.865,  276.154,  -82.1, 150.2,  85.0,  -2.1,  1.4,  1.4},
{ 4, 774, 773,  721,   0.400,   4.865,  281.019,  -84.2, 151.6,  86.4,  -2.1,  1.4,  1.4},
{ 4, 775, 774,  721,   0.400,   4.865,  285.884,  -86.3, 153.1,  87.9,  -2.1,  1.4,  1.4},
{ 4, 776, 775,  721,   0.400,   4.865,  290.749,  -88.4, 154.5,  89.3,  -2.1,  1.4,  1.4},
{ 4, 777, 776,  721,   0.400,   4.865,  295.613,  -90.5, 155.9,  90.7,  -2.1,  1.4,  1.4},
{ 4, 778, 777,  721,   0.400,   4.865,  300.478,  -92.6, 157.4,  92.1,  -2.1,  1.4,  1.4},
{ 4, 779, 778,  721,   0.400,   4.865,  305.343,  -94.7, 158.8,  93.6,  -2.1,  1.4,  1.4},
{ 4, 780, 779,  721,   0.400,   4.865,  310.208,  -96.8, 160.2,  95.0,  -2.1,  1.4,  1.4},
{ 4, 781, 780,  721,   0.400,   4.865,  315.072,  -98.9, 161.6,  96.4,  -2.1,  1.4,  1.4},
{ 4, 782, 781,  721,   0.400,   4.865,  319.937, -101.0, 163.1,  97.8,  -2.1,  1.4,  1.4},
{ 4, 783, 782,  721,   0.400,   4.865,  324.802, -103.1, 164.5,  99.3,  -2.1,  1.4,  1.4},
{ 4, 784, 783,  721,   0.400,   4.865,  329.667, -105.2, 165.9, 100.7,  -2.1,  1.4,  1.4},
{ 4, 785, 784,  721,   0.400,   4.865,  334.531, -107.3, 167.4, 102.1,  -2.1,  1.4,  1.4},
{ 4, 786, 785,  721,   0.400,   4.865,  339.396, -109.4, 168.8, 103.5,  -2.1,  1.4,  1.4},
{ 4, 787, 786,  721,   0.400,   4.865,  344.261, -111.5, 170.2, 104.9,  -2.1,  1.4,  1.4},
{ 4, 788, 787,  721,   0.400,   4.865,  349.126, -113.6, 171.7, 106.4,  -2.1,  1.4,  1.4},
{ 4, 789, 788,  721,   0.400,   4.865,  353.990, -115.7, 173.1, 107.8,  -2.1,  1.4,  1.4},
{ 4, 790, 763,  723,   0.318,   3.357,  231.618,  -61.8, 137.4,  70.1,   0.5,  1.6, -2.8},
{ 4, 791, 790,  723,   0.318,   3.357,  234.975,  -61.3, 139.0,  67.4,   0.5,  1.6, -2.8},
{ 4, 792, 791,  759,   0.400,   4.953,  239.129,  -61.2, 141.0,  66.6,  -0.3,  2.4,  1.2},
{ 4, 793, 792,  759,   0.400,   4.953,  244.082,  -61.5, 143.4,  67.8,  -0.3,  2.4,  1.2},
{ 4, 794, 793,  759,   0.400,   4.953,  249.035,  -61.8, 145.8,  68.9,  -0.3,  2.4,  1.2},
{ 4, 795, 794,  759,   0.400,   4.953,  253.988,  -62.1, 148.2,  70.1,  -0.3,  2.4,  1.2},
{ 4, 796, 795,  759,   0.400,   4.953,  258.941,  -62.5, 150.6,  71.3,  -0.3,  2.4,  1.2},
{ 4, 797, 796,  759,   0.400,   4.953,  263.894,  -62.8, 153.0,  72.5,  -0.3,  2.4,  1.2},
{ 4, 798, 797,  759,   0.400,   4.953,  268.847,  -63.1, 155.4,  73.6,  -0.3,  2.4,  1.2},
{ 4, 799, 798,  759,   0.400,   4.953,  273.800,  -63.4, 157.8,  74.8,  -0.3,  2.4,  1.2},
{ 4, 800, 799,  759,   0.400,   4.953,  278.753,  -63.8, 160.2,  76.0,  -0.3,  2.4,  1.2},
{ 4, 801, 800,  759,   0.400,   4.953,  283.706,  -64.1, 162.6,  77.2,  -0.3,  2.4,  1.2},
{ 4, 802, 801,  759,   0.400,   4.953,  288.659,  -64.4, 165.0,  78.3,  -0.3,  2.4,  1.2},
{ 4, 803, 802,  759,   0.400,   4.953,  293.611,  -64.7, 167.4,  79.5,  -0.3,  2.4,  1.2},
{ 4, 804, 803,  759,   0.400,   4.953,  298.564,  -65.1, 169.8,  80.7,  -0.3,  2.4,  1.2},
{ 4, 805, 804,  759,   0.400,   4.953,  303.517,  -65.4, 172.2,  81.9,  -0.3,  2.4,  1.2},
{ 4, 806, 805,  759,   0.400,   4.953,  308.470,  -65.7, 174.5,  83.0,  -0.3,  2.4,  1.2},
{ 4, 807, 806,  759,   0.400,   4.953,  313.423,  -66.0, 176.9,  84.2,  -0.3,  2.4,  1.2},
{ 4, 808, 807,  759,   0.400,   4.953,  318.376,  -66.4, 179.3,  85.4,  -0.3,  2.4,  1.2},
{ 4, 809, 808,  759,   0.400,   4.953,  323.329,  -66.7, 181.7,  86.6,  -0.3,  2.4,  1.2},
{ 4, 810, 809,  759,   0.400,   4.953,  328.282,  -67.0, 184.1,  87.7,  -0.3,  2.4,  1.2},
{ 4, 811, 810,  759,   0.400,   4.953,  333.235,  -67.4, 186.5,  88.9,  -0.3,  2.4,  1.2},
{ 4, 812, 811,  759,   0.400,   4.953,  338.188,  -67.7, 188.9,  90.1,  -0.3,  2.4,  1.2},
{ 4, 813, 812,  759,   0.400,   4.953,  343.141,  -68.0, 191.3,  91.2,  -0.3,  2.4,  1.2},
{ 4, 814, 813,  759,   0.400,   4.953,  348.093,  -68.3, 193.7,  92.4,  -0.3,  2.4,  1.2},
{ 4, 815, 814,  759,   0.400,   4.953,  353.046,  -68.7, 196.1,  93.6,  -0.3,  2.4,  1.2},
{ 4, 816, 815,  759,   0.400,   4.953,  357.999,  -69.0, 198.5,  94.8,  -0.3,  2.4,  1.2},
{ 4, 817, 816,  759,   0.400,   4.953,  362.952,  -69.3, 200.9,  95.9,  -0.3,  2.4,  1.2},
{ 4, 818, 817,  759,   0.400,   4.953,  367.905,  -69.6, 203.3,  97.1,  -0.3,  2.4,  1.2},
{ 4, 819, 818,  759,   0.400,   4.953,  372.858,  -70.0, 205.7,  98.3,  -0.3,  2.4,  1.2},
{ 4, 820, 819,  759,   0.400,   4.953,  377.811,  -70.3, 208.1,  99.5,  -0.3,  2.4,  1.2},
{ 4, 821, 820,  759,   0.400,   4.953,  382.764,  -70.6, 210.5, 100.6,  -0.3,  2.4,  1.2},
{ 4, 822, 821,  759,   0.400,   4.953,  387.717,  -70.9, 212.9, 101.8,  -0.3,  2.4,  1.2},
{ 4, 823, 822,  759,   0.400,   4.953,  392.670,  -71.3, 215.3, 103.0,  -0.3,  2.4,  1.2},
{ 4, 824, 823,  759,   0.400,   4.953,  397.622,  -71.6, 217.7, 104.2,  -0.3,  2.4,  1.2},
{ 4, 825, 824,  759,   0.400,   4.953,  402.575,  -71.9, 220.0, 105.3,  -0.3,  2.4,  1.2},
{ 4, 826, 825,  759,   0.400,   4.953,  407.528,  -72.2, 222.4, 106.5,  -0.3,  2.4,  1.2},
{ 4, 827, 826,  759,   0.400,   4.953,  412.481,  -72.6, 224.8, 107.7,  -0.3,  2.4,  1.2},
{ 4, 828, 827,  759,   0.400,   4.953,  417.434,  -72.9, 227.2, 108.9,  -0.3,  2.4,  1.2},
{ 4, 829, 641,  762,   0.481,   3.507,  124.471,  -43.6, 107.4,  41.6,  -1.2,  2.0, -0.3},
{ 4, 830, 829,  762,   0.481,   3.507,  127.978,  -44.8, 109.4,  41.4,  -1.2,  2.0, -0.3},
{ 4, 831, 830,  762,   0.481,   3.507,  131.484,  -46.0, 111.3,  41.1,  -1.2,  2.0, -0.3},
{ 4, 832, 831,  764,   0.560,   4.124,  135.300,  -47.3, 113.3,  41.5,  -1.2,  2.0,  1.1},
{ 4, 833, 832,  764,   0.560,   4.124,  139.424,  -48.5, 115.2,  42.6,  -1.2,  2.0,  1.1},
{ 4, 834, 833,  764,   0.560,   4.124,  143.548,  -49.7, 117.2,  43.7,  -1.2,  2.0,  1.1},
{ 4, 835, 834,  770,   0.560,   4.541,  147.881,  -51.1, 120.1,  44.1,  -1.4,  3.8, -0.2},
{ 4, 836, 835,  770,   0.560,   4.541,  152.422,  -52.5, 123.9,  43.9,  -1.4,  3.8, -0.2},
{ 4, 837, 836,  770,   0.560,   4.541,  156.963,  -53.9, 127.8,  43.7,  -1.4,  3.8, -0.2},
{ 4, 838, 837,  770,   0.560,   4.541,  161.504,  -55.4, 131.6,  43.5,  -1.4,  3.8, -0.2},
{ 4, 839, 838,  770,   0.560,   4.541,  166.045,  -56.8, 135.5,  43.3,  -1.4,  3.8, -0.2},
{ 4, 840, 839,  770,   0.560,   4.541,  170.586,  -58.3, 139.3,  43.1,  -1.4,  3.8, -0.2},
{ 4, 841, 840,  815,   0.400,   4.942,  175.328,  -60.0, 142.6,  42.3,  -2.0,  2.6, -1.5},
{ 4, 842, 841,  815,   0.400,   4.942,  180.270,  -61.9, 145.2,  40.8,  -2.0,  2.6, -1.5},
{ 4, 843, 842,  815,   0.400,   4.942,  185.212,  -63.9, 147.8,  39.3,  -2.0,  2.6, -1.5},
{ 4, 844, 843,  815,   0.400,   4.942,  190.154,  -65.8, 150.5,  37.8,  -2.0,  2.6, -1.5},
{ 4, 845, 844,  815,   0.400,   4.942,  195.096,  -67.8, 153.1,  36.3,  -2.0,  2.6, -1.5},
{ 4, 846, 845,  815,   0.400,   4.942,  200.038,  -69.7, 155.8,  34.8,  -2.0,  2.6, -1.5},
{ 4, 847, 846,  815,   0.400,   4.942,  204.981,  -71.7, 158.4,  33.3,  -2.0,  2.6, -1.5},
{ 4, 848, 847,  815,   0.400,   4.942,  209.923,  -73.6, 161.0,  31.8,  -2.0,  2.6, -1.5},
{ 4, 849, 848,  815,   0.400,   4.942,  214.865,  -75.6, 163.7,  30.3,  -2.0,  2.6, -1.5},
{ 4, 850, 849,  815,   0.400,   4.942,  219.807,  -77.6, 166.3,  28.8,  -2.0,  2.6, -1.5},
{ 4, 851, 850,  815,   0.400,   4.942,  224.749,  -79.5, 169.0,  27.3,  -2.0,  2.6, -1.5},
{ 4, 852, 851,  815,   0.400,   4.942,  229.691,  -81.5, 171.6,  25.8,  -2.0,  2.6, -1.5},
{ 4, 853, 852,  815,   0.400,   4.942,  234.633,  -83.4, 174.2,  24.3,  -2.0,  2.6, -1.5},
{ 4, 854, 853,  815,   0.400,   4.942,  239.575,  -85.4, 176.9,  22.8,  -2.0,  2.6, -1.5},
{ 4, 855, 854,  815,   0.400,   4.942,  244.518,  -87.3, 179.5,  21.3,  -2.0,  2.6, -1.5},
{ 4, 856, 855,  815,   0.400,   4.942,  249.460,  -89.3, 182.2,  19.8,  -2.0,  2.6, -1.5},
{ 4, 857, 856,  815,   0.400,   4.942,  254.402,  -91.3, 184.8,  18.3,  -2.0,  2.6, -1.5},
{ 4, 858, 857,  815,   0.400,   4.942,  259.344,  -93.2, 187.4,  16.8,  -2.0,  2.6, -1.5},
{ 4, 859, 858,  815,   0.400,   4.942,  264.286,  -95.2, 190.1,  15.4,  -2.0,  2.6, -1.5},
{ 4, 860, 859,  815,   0.400,   4.942,  269.228,  -97.1, 192.7,  13.9,  -2.0,  2.6, -1.5},
{ 4, 861, 860,  815,   0.400,   4.942,  274.170,  -99.1, 195.4,  12.4,  -2.0,  2.6, -1.5},
{ 4, 862, 861,  815,   0.400,   4.942,  279.112, -101.0, 198.0,  10.9,  -2.0,  2.6, -1.5},
{ 4, 863, 862,  815,   0.400,   4.942,  284.055, -103.0, 200.6,   9.4,  -2.0,  2.6, -1.5},
{ 4, 864, 863,  815,   0.400,   4.942,  288.997, -104.9, 203.3,   7.9,  -2.0,  2.6, -1.5},
{ 4, 865, 864,  815,   0.400,   4.942,  293.939, -106.9, 205.9,   6.4,  -2.0,  2.6, -1.5},
{ 4, 866, 865,  815,   0.400,   4.942,  298.881, -108.9, 208.6,   4.9,  -2.0,  2.6, -1.5},
{ 4, 867, 866,  815,   0.400,   4.942,  303.823, -110.8, 211.2,   3.4,  -2.0,  2.6, -1.5},
{ 4, 868, 867,  815,   0.400,   4.942,  308.765, -112.8, 213.8,   1.9,  -2.0,  2.6, -1.5},
{ 4, 869, 868,  815,   0.400,   4.942,  313.707, -114.7, 216.5,   0.4,  -2.0,  2.6, -1.5},
{ 4, 870, 869,  815,   0.400,   4.942,  318.650, -116.7, 219.1,  -1.1,  -2.0,  2.6, -1.5},
{ 4, 871, 870,  815,   0.400,   4.942,  323.592, -118.6, 221.8,  -2.6,  -2.0,  2.6, -1.5},
{ 4, 872, 871,  815,   0.400,   4.942,  328.534, -120.6, 224.4,  -4.1,  -2.0,  2.6, -1.5},
{ 4, 873, 872,  815,   0.400,   4.942,  333.476, -122.6, 227.0,  -5.6,  -2.0,  2.6, -1.5},
{ 4, 874, 873,  815,   0.400,   4.942,  338.418, -124.5, 229.7,  -7.1,  -2.0,  2.6, -1.5},
{ 4, 875, 874,  815,   0.400,   4.942,  343.360, -126.5, 232.3,  -8.6,  -2.0,  2.6, -1.5},
{ 4, 876, 875,  815,   0.400,   4.942,  348.302, -128.4, 235.0, -10.0,  -2.0,  2.6, -1.5},
{ 4, 877, 876,  815,   0.400,   4.942,  353.244, -130.4, 237.6, -11.5,  -2.0,  2.6, -1.5},
{ 4, 878, 877,  815,   0.400,   4.942,  358.187, -132.3, 240.2, -13.0,  -2.0,  2.6, -1.5},
{ 4, 879, 878,  815,   0.400,   4.942,  363.129, -134.3, 242.9, -14.5,  -2.0,  2.6, -1.5},
{ 4, 880, 879,  815,   0.400,   4.942,  368.071, -136.2, 245.5, -16.0,  -2.0,  2.6, -1.5},
{ 4, 881, 880,  815,   0.400,   4.942,  373.013, -138.2, 248.2, -17.5,  -2.0,  2.6, -1.5},
{ 4, 882, 881,  815,   0.400,   4.942,  377.955, -140.2, 250.8, -19.0,  -2.0,  2.6, -1.5},
{ 4, 883, 882,  815,   0.400,   4.942,  382.897, -142.1, 253.4, -20.5,  -2.0,  2.6, -1.5},
{ 4, 884, 883,  815,   0.400,   4.942,  387.839, -144.1, 256.1, -22.0,  -2.0,  2.6, -1.5},
{ 4, 885, 840,  826,   0.560,   4.868,  175.291,  -59.8, 143.1,  42.7,  -1.6,  3.6, -0.6},
{ 4, 886, 885,  826,   0.560,   4.868,  180.159,  -61.3, 146.7,  42.1,  -1.6,  3.6, -0.6},
{ 4, 887, 886,  826,   0.560,   4.868,  185.027,  -62.9, 150.3,  41.5,  -1.6,  3.6, -0.6},
{ 4, 888, 887,  826,   0.560,   4.868,  189.895,  -64.4, 154.0,  40.9,  -1.6,  3.6, -0.6},
{ 4, 889, 888,  826,   0.560,   4.868,  194.763,  -66.0, 157.6,  40.3,  -1.6,  3.6, -0.6},
{ 4, 890, 889,  826,   0.560,   4.868,  199.631,  -67.6, 161.3,  39.6,  -1.6,  3.6, -0.6},
{ 4, 891, 890,  826,   0.560,   4.868,  204.499,  -69.1, 164.9,  39.0,  -1.6,  3.6, -0.6},
{ 4, 892, 891,  826,   0.560,   4.868,  209.367,  -70.7, 168.5,  38.4,  -1.6,  3.6, -0.6},
{ 4, 893, 892,  826,   0.560,   4.868,  214.235,  -72.3, 172.2,  37.8,  -1.6,  3.6, -0.6},
{ 4, 894, 893,  826,   0.560,   4.868,  219.103,  -73.8, 175.8,  37.2,  -1.6,  3.6, -0.6},
{ 4, 895, 894,  826,   0.560,   4.868,  223.971,  -75.4, 179.5,  36.6,  -1.6,  3.6, -0.6},
{ 4, 896, 895,  826,   0.560,   4.868,  228.839,  -77.0, 183.1,  36.0,  -1.6,  3.6, -0.6},
{ 4, 897, 896,  826,   0.560,   4.868,  233.707,  -78.5, 186.8,  35.4,  -1.6,  3.6, -0.6},
{ 4, 898, 897,  826,   0.560,   4.868,  238.575,  -80.1, 190.4,  34.8,  -1.6,  3.6, -0.6},
{ 4, 899, 898,  826,   0.560,   4.868,  243.443,  -81.7, 194.0,  34.1,  -1.6,  3.6, -0.6},
{ 4, 900, 899,  826,   0.560,   4.868,  248.311,  -83.2, 197.7,  33.5,  -1.6,  3.6, -0.6},
{ 4, 901, 900,  826,   0.560,   4.868,  253.179,  -84.8, 201.3,  32.9,  -1.6,  3.6, -0.6},
{ 4, 902, 901,  826,   0.560,   4.868,  258.047,  -86.4, 205.0,  32.3,  -1.6,  3.6, -0.6},
{ 4, 903, 902,  829,   0.560,   4.675,  262.819,  -87.2, 209.1,  32.0,  -0.1,  4.7,  0.0},
{ 4, 904, 903,  829,   0.560,   4.675,  267.493,  -87.3, 213.8,  32.0,  -0.1,  4.7,  0.0},
{ 4, 905, 904,  829,   0.560,   4.675,  272.168,  -87.3, 218.5,  32.0,  -0.1,  4.7,  0.0},
{ 4, 906, 905,  857,   0.240,   4.906,  276.958,  -88.2, 222.0,  31.9,  -1.7,  2.4, -0.1},
{ 4, 907, 906,  857,   0.240,   4.906,  281.865,  -90.0, 224.3,  31.8,  -1.7,  2.4, -0.1},
{ 4, 908, 907,  857,   0.240,   4.906,  286.771,  -91.7, 226.7,  31.6,  -1.7,  2.4, -0.1},
{ 4, 909, 908,  857,   0.240,   4.906,  291.677,  -93.4, 229.1,  31.5,  -1.7,  2.4, -0.1},
{ 4, 910, 909,  857,   0.240,   4.906,  296.583,  -95.2, 231.4,  31.4,  -1.7,  2.4, -0.1},
{ 4, 911, 910,  857,   0.240,   4.906,  301.490,  -96.9, 233.8,  31.2,  -1.7,  2.4, -0.1},
{ 4, 912, 911,  857,   0.240,   4.906,  306.396,  -98.6, 236.2,  31.1,  -1.7,  2.4, -0.1},
{ 4, 913, 912,  857,   0.240,   4.906,  311.302, -100.4, 238.5,  30.9,  -1.7,  2.4, -0.1},
{ 4, 914, 913,  857,   0.240,   4.906,  316.208, -102.1, 240.9,  30.8,  -1.7,  2.4, -0.1},
{ 4, 915, 914,  857,   0.240,   4.906,  321.114, -103.8, 243.3,  30.7,  -1.7,  2.4, -0.1},
{ 4, 916, 915,  857,   0.240,   4.906,  326.021, -105.5, 245.6,  30.5,  -1.7,  2.4, -0.1},
{ 4, 917, 916,  857,   0.240,   4.906,  330.927, -107.3, 248.0,  30.4,  -1.7,  2.4, -0.1},
{ 4, 918, 917,  857,   0.240,   4.906,  335.833, -109.0, 250.4,  30.2,  -1.7,  2.4, -0.1},
{ 4, 919, 918,  857,   0.240,   4.906,  340.739, -110.7, 252.7,  30.1,  -1.7,  2.4, -0.1},
{ 4, 920, 919,  857,   0.240,   4.906,  345.646, -112.5, 255.1,  29.9,  -1.7,  2.4, -0.1},
{ 4, 921, 920,  857,   0.240,   4.906,  350.552, -114.2, 257.5,  29.8,  -1.7,  2.4, -0.1},
{ 4, 922, 921,  857,   0.240,   4.906,  355.458, -115.9, 259.8,  29.7,  -1.7,  2.4, -0.1},
{ 4, 923, 922,  857,   0.240,   4.906,  360.364, -117.7, 262.2,  29.5,  -1.7,  2.4, -0.1},
{ 4, 924, 923,  857,   0.240,   4.906,  365.270, -119.4, 264.6,  29.4,  -1.7,  2.4, -0.1},
{ 4, 925, 924,  857,   0.240,   4.906,  370.177, -121.1, 266.9,  29.2,  -1.7,  2.4, -0.1},
{ 4, 926, 925,  857,   0.240,   4.906,  375.083, -122.8, 269.3,  29.1,  -1.7,  2.4, -0.1},
{ 4, 927, 926,  857,   0.240,   4.906,  379.989, -124.6, 271.6,  28.9,  -1.7,  2.4, -0.1},
{ 4, 928, 927,  857,   0.240,   4.906,  384.895, -126.3, 274.0,  28.8,  -1.7,  2.4, -0.1},
{ 4, 929, 928,  857,   0.240,   4.906,  389.802, -128.0, 276.4,  28.7,  -1.7,  2.4, -0.1},
{ 4, 930, 929,  857,   0.240,   4.906,  394.708, -129.8, 278.7,  28.5,  -1.7,  2.4, -0.1},
{ 4, 931, 930,  857,   0.240,   4.906,  399.614, -131.5, 281.1,  28.4,  -1.7,  2.4, -0.1},
{ 4, 932, 931,  857,   0.240,   4.906,  404.520, -133.2, 283.5,  28.2,  -1.7,  2.4, -0.1},
{ 4, 933, 932,  857,   0.240,   4.906,  409.426, -134.9, 285.8,  28.1,  -1.7,  2.4, -0.1},
{ 4, 934, 933,  857,   0.240,   4.906,  414.333, -136.7, 288.2,  28.0,  -1.7,  2.4, -0.1},
{ 4, 935, 934,  857,   0.240,   4.906,  419.239, -138.4, 290.6,  27.8,  -1.7,  2.4, -0.1},
{ 4, 936, 935,  857,   0.240,   4.906,  424.145, -140.1, 292.9,  27.7,  -1.7,  2.4, -0.1},
{ 4, 937, 936,  857,   0.240,   4.906,  429.051, -141.9, 295.3,  27.5,  -1.7,  2.4, -0.1},
{ 4, 938, 937,  857,   0.240,   4.906,  433.958, -143.6, 297.7,  27.4,  -1.7,  2.4, -0.1},
{ 4, 939, 938,  857,   0.240,   4.906,  438.864, -145.3, 300.0,  27.2,  -1.7,  2.4, -0.1},
{ 4, 940, 939,  857,   0.240,   4.906,  443.770, -147.1, 302.4,  27.1,  -1.7,  2.4, -0.1},
{ 4, 941, 940,  857,   0.240,   4.906,  448.676, -148.8, 304.8,  27.0,  -1.7,  2.4, -0.1},
{ 4, 942, 941,  857,   0.240,   4.906,  453.582, -150.5, 307.1,  26.8,  -1.7,  2.4, -0.1},
{ 4, 943, 905,  860,   0.560,   4.692,  276.851,  -87.0, 223.0,  32.0,   0.8,  4.5,  0.0},
{ 4, 944, 943,  860,   0.560,   4.692,  281.543,  -86.2, 227.5,  32.0,   0.8,  4.5,  0.0},
{ 4, 945, 944,  860,   0.560,   4.692,  286.234,  -85.4, 232.0,  32.0,   0.8,  4.5,  0.0},
{ 4, 946, 945,  860,   0.560,   4.692,  290.926,  -84.6, 236.5,  32.0,   0.8,  4.5,  0.0},
{ 4, 947, 946,  863,   0.684,   5.079,  295.811,  -84.0, 241.2,  32.0,   0.6,  5.0,  0.0},
{ 4, 948, 947,  863,   0.684,   5.079,  300.890,  -83.4, 246.2,  32.0,   0.6,  5.0,  0.0},
{ 4, 949, 948,  863,   0.684,   5.079,  305.968,  -82.9, 251.2,  32.0,   0.6,  5.0,  0.0},
{ 4, 950, 949,  863,   0.684,   5.079,  311.047,  -82.3, 256.2,  32.0,   0.6,  5.0,  0.0},
{ 4, 951, 950,  908,   0.400,   4.920,  316.047,  -82.6, 260.2,  31.9,  -1.1,  3.0, -0.1},
{ 4, 952, 951,  908,   0.400,   4.920,  320.967,  -83.8, 263.3,  31.8,  -1.1,  3.0, -0.1},
{ 4, 953, 952,  908,   0.400,   4.920,  325.887,  -84.9, 266.3,  31.7,  -1.1,  3.0, -0.1},
{ 4, 954, 953,  908,   0.400,   4.920,  330.808,  -86.0, 269.4,  31.6,  -1.1,  3.0, -0.1},
{ 4, 955, 954,  908,   0.400,   4.920,  335.728,  -87.2, 272.4,  31.5,  -1.1,  3.0, -0.1},
{ 4, 956, 955,  908,   0.400,   4.920,  340.649,  -88.3, 275.4,  31.4,  -1.1,  3.0, -0.1},
{ 4, 957, 956,  908,   0.400,   4.920,  345.569,  -89.4, 278.5,  31.3,  -1.1,  3.0, -0.1},
{ 4, 958, 957,  908,   0.400,   4.920,  350.490,  -90.6, 281.5,  31.2,  -1.1,  3.0, -0.1},
{ 4, 959, 958,  908,   0.400,   4.920,  355.410,  -91.7, 284.6,  31.1,  -1.1,  3.0, -0.1},
{ 4, 960, 959,  908,   0.400,   4.920,  360.330,  -92.8, 287.6,  31.0,  -1.1,  3.0, -0.1},
{ 4, 961, 960,  908,   0.400,   4.920,  365.251,  -94.0, 290.6,  30.9,  -1.1,  3.0, -0.1},
{ 4, 962, 961,  908,   0.400,   4.920,  370.171,  -95.1, 293.7,  30.8,  -1.1,  3.0, -0.1},
{ 4, 963, 962,  908,   0.400,   4.920,  375.092,  -96.3, 296.7,  30.7,  -1.1,  3.0, -0.1},
{ 4, 964, 963,  908,   0.400,   4.920,  380.012,  -97.4, 299.8,  30.6,  -1.1,  3.0, -0.1},
{ 4, 965, 964,  908,   0.400,   4.920,  384.933,  -98.5, 302.8,  30.5,  -1.1,  3.0, -0.1},
{ 4, 966, 965,  908,   0.400,   4.920,  389.853,  -99.7, 305.8,  30.4,  -1.1,  3.0, -0.1},
{ 4, 967, 966,  908,   0.400,   4.920,  394.773, -100.8, 308.9,  30.3,  -1.1,  3.0, -0.1},
{ 4, 968, 967,  908,   0.400,   4.920,  399.694, -101.9, 311.9,  30.2,  -1.1,  3.0, -0.1},
{ 4, 969, 968,  908,   0.400,   4.920,  404.614, -103.1, 315.0,  30.1,  -1.1,  3.0, -0.1},
{ 4, 970, 969,  908,   0.400,   4.920,  409.535, -104.2, 318.0,  30.0,  -1.1,  3.0, -0.1},
{ 4, 971, 970,  908,   0.400,   4.920,  414.455, -105.3, 321.1,  29.9,  -1.1,  3.0, -0.1},
{ 4, 972, 971,  908,   0.400,   4.920,  419.376, -106.5, 324.1,  29.8,  -1.1,  3.0, -0.1},
{ 4, 973, 972,  908,   0.400,   4.920,  424.296, -107.6, 327.1,  29.7,  -1.1,  3.0, -0.1},
{ 4, 974, 973,  908,   0.400,   4.920,  429.216, -108.8, 330.2,  29.6,  -1.1,  3.0, -0.1},
{ 4, 975, 974,  908,   0.400,   4.920,  434.137, -109.9, 333.2,  29.4,  -1.1,  3.0, -0.1},
{ 4, 976, 975,  908,   0.400,   4.920,  439.057, -111.0, 336.3,  29.3,  -1.1,  3.0, -0.1},
{ 4, 977, 976,  908,   0.400,   4.920,  443.978, -112.2, 339.3,  29.2,  -1.1,  3.0, -0.1},
{ 4, 978, 977,  908,   0.400,   4.920,  448.898, -113.3, 342.3,  29.1,  -1.1,  3.0, -0.1},
{ 4, 979, 978,  908,   0.400,   4.920,  453.819, -114.4, 345.4,  29.0,  -1.1,  3.0, -0.1},
{ 4, 980, 979,  908,   0.400,   4.920,  458.739, -115.6, 348.4,  28.9,  -1.1,  3.0, -0.1},
{ 4, 981, 980,  908,   0.400,   4.920,  463.659, -116.7, 351.5,  28.8,  -1.1,  3.0, -0.1},
{ 4, 982, 981,  908,   0.400,   4.920,  468.580, -117.9, 354.5,  28.7,  -1.1,  3.0, -0.1},
{ 4, 983, 982,  908,   0.400,   4.920,  473.500, -119.0, 357.5,  28.6,  -1.1,  3.0, -0.1},
{ 4, 984, 983,  908,   0.400,   4.920,  478.421, -120.1, 360.6,  28.5,  -1.1,  3.0, -0.1},
{ 4, 985, 984,  908,   0.400,   4.920,  483.341, -121.3, 363.6,  28.4,  -1.1,  3.0, -0.1},
{ 4, 986, 985,  908,   0.400,   4.920,  488.261, -122.4, 366.7,  28.3,  -1.1,  3.0, -0.1},
{ 4, 987, 986,  908,   0.400,   4.920,  493.182, -123.5, 369.7,  28.2,  -1.1,  3.0, -0.1},
{ 4, 988, 987,  908,   0.400,   4.920,  498.102, -124.7, 372.7,  28.1,  -1.1,  3.0, -0.1},
{ 4, 989, 988,  908,   0.400,   4.920,  503.023, -125.8, 375.8,  28.0,  -1.1,  3.0, -0.1},
{ 4, 990, 989,  908,   0.400,   4.920,  507.943, -126.9, 378.8,  27.9,  -1.1,  3.0, -0.1},
{ 4, 991, 990,  908,   0.400,   4.920,  512.864, -128.1, 381.9,  27.8,  -1.1,  3.0, -0.1},
{ 4, 992, 991,  908,   0.400,   4.920,  517.784, -129.2, 384.9,  27.7,  -1.1,  3.0, -0.1},
{ 4, 993, 992,  908,   0.400,   4.920,  522.704, -130.4, 388.0,  27.6,  -1.1,  3.0, -0.1},
{ 4, 994, 993,  908,   0.400,   4.920,  527.625, -131.5, 391.0,  27.5,  -1.1,  3.0, -0.1},
{ 4, 995, 994,  908,   0.400,   4.920,  532.545, -132.6, 394.0,  27.4,  -1.1,  3.0, -0.1},
{ 4, 996, 995,  908,   0.400,   4.920,  537.466, -133.8, 397.1,  27.3,  -1.1,  3.0, -0.1},
{ 4, 997, 996,  908,   0.400,   4.920,  542.386, -134.9, 400.1,  27.2,  -1.1,  3.0, -0.1},
{ 4, 998, 997,  908,   0.400,   4.920,  547.307, -136.0, 403.2,  27.1,  -1.1,  3.0, -0.1},
{ 4, 999, 998,  924,   0.240,   4.713,  552.123, -136.2, 406.4,  26.8,   0.9,  3.4, -0.5},
{ 4,1000, 999,  924,   0.240,   4.713,  556.836, -135.3, 409.7,  26.3,   0.9,  3.4, -0.5},
{ 4,1001,1000,  924,   0.240,   4.713,  561.548, -134.4, 413.1,  25.8,   0.9,  3.4, -0.5},
{ 4,1002,1001,  924,   0.240,   4.713,  566.261, -133.5, 416.5,  25.4,   0.9,  3.4, -0.5},
{ 4,1003,1002,  924,   0.240,   4.713,  570.973, -132.7, 419.8,  24.9,   0.9,  3.4, -0.5},
{ 4,1004,1003,  924,   0.240,   4.713,  575.686, -131.8, 423.2,  24.4,   0.9,  3.4, -0.5},
{ 4,1005,1004,  924,   0.240,   4.713,  580.398, -130.9, 426.6,  24.0,   0.9,  3.4, -0.5},
{ 4,1006,1005,  924,   0.240,   4.713,  585.111, -130.1, 429.9,  23.5,   0.9,  3.4, -0.5},
{ 4,1007, 998,  939,   0.240,   4.829,  552.181, -137.4, 406.8,  27.2,  -1.5,  4.1,  0.3},
{ 4,1008,1007,  939,   0.240,   4.829,  557.010, -138.9, 410.9,  27.5,  -1.5,  4.1,  0.3},
{ 4,1009,1008,  939,   0.240,   4.829,  561.839, -140.4, 415.0,  27.9,  -1.5,  4.1,  0.3},
{ 4,1010,1009,  939,   0.240,   4.829,  566.668, -142.0, 419.2,  28.2,  -1.5,  4.1,  0.3},
{ 4,1011,1010,  939,   0.240,   4.829,  571.497, -143.5, 423.3,  28.5,  -1.5,  4.1,  0.3},
{ 4,1012,1011,  939,   0.240,   4.829,  576.326, -145.0, 427.5,  28.9,  -1.5,  4.1,  0.3},
{ 4,1013,1012,  939,   0.240,   4.829,  581.155, -146.6, 431.6,  29.2,  -1.5,  4.1,  0.3},
{ 4,1014,1013,  939,   0.240,   4.829,  585.984, -148.1, 435.8,  29.6,  -1.5,  4.1,  0.3},
{ 4,1015,1014,  939,   0.240,   4.829,  590.812, -149.7, 439.9,  29.9,  -1.5,  4.1,  0.3},
{ 4,1016,1015,  939,   0.240,   4.829,  595.641, -151.2, 444.1,  30.3,  -1.5,  4.1,  0.3},
{ 4,1017,1016,  939,   0.240,   4.829,  600.470, -152.7, 448.2,  30.6,  -1.5,  4.1,  0.3},
{ 4,1018,1017,  939,   0.240,   4.829,  605.299, -154.3, 452.4,  30.9,  -1.5,  4.1,  0.3},
{ 4,1019,1018,  939,   0.240,   4.829,  610.128, -155.8, 456.5,  31.3,  -1.5,  4.1,  0.3},
{ 4,1020,1019,  939,   0.240,   4.829,  614.957, -157.3, 460.7,  31.6,  -1.5,  4.1,  0.3},
{ 4,1021,1020,  939,   0.240,   4.829,  619.786, -158.9, 464.8,  32.0,  -1.5,  4.1,  0.3},
{ 4,1022,1021,  939,   0.240,   4.829,  624.615, -160.4, 469.0,  32.3,  -1.5,  4.1,  0.3},
{ 4,1023,1022,  939,   0.240,   4.829,  629.444, -161.9, 473.1,  32.6,  -1.5,  4.1,  0.3},
{ 4,1024,1023,  939,   0.240,   4.829,  634.273, -163.5, 477.3,  33.0,  -1.5,  4.1,  0.3},
{ 4,1025,1024,  939,   0.240,   4.829,  639.101, -165.0, 481.4,  33.3,  -1.5,  4.1,  0.3},
{ 4,1026, 950,  967,   0.565,   4.868,  316.020,  -81.6, 260.7,  32.1,   1.0,  4.0,  0.3},
{ 4,1027,1026,  967,   0.565,   4.868,  320.888,  -80.6, 264.7,  32.4,   1.0,  4.0,  0.3},
{ 4,1028,1027,  967,   0.565,   4.868,  325.756,  -79.7, 268.7,  32.7,   1.0,  4.0,  0.3},
{ 4,1029,1028,  967,   0.565,   4.868,  330.624,  -78.7, 272.7,  33.0,   1.0,  4.0,  0.3},
{ 4,1030,1029,  967,   0.565,   4.868,  335.492,  -77.7, 276.7,  33.3,   1.0,  4.0,  0.3},
{ 4,1031,1030,  967,   0.565,   4.868,  340.360,  -76.8, 280.7,  33.5,   1.0,  4.0,  0.3},
{ 4,1032,1031,  967,   0.565,   4.868,  345.228,  -75.8, 284.7,  33.8,   1.0,  4.0,  0.3},
{ 4,1033,1032,  967,   0.565,   4.868,  350.096,  -74.9, 288.8,  34.1,   1.0,  4.0,  0.3},
{ 4,1034,1033,  967,   0.565,   4.868,  354.964,  -73.9, 292.8,  34.4,   1.0,  4.0,  0.3},
{ 4,1035,1034,  967,   0.565,   4.868,  359.832,  -73.0, 296.8,  34.6,   1.0,  4.0,  0.3},
{ 4,1036,1035,  967,   0.565,   4.868,  364.700,  -72.0, 300.8,  34.9,   1.0,  4.0,  0.3},
{ 4,1037,1036,  967,   0.565,   4.868,  369.568,  -71.0, 304.8,  35.2,   1.0,  4.0,  0.3},
{ 4,1038,1037,  967,   0.565,   4.868,  374.436,  -70.1, 308.8,  35.5,   1.0,  4.0,  0.3},
{ 4,1039,1038,  967,   0.565,   4.868,  379.304,  -69.1, 312.8,  35.8,   1.0,  4.0,  0.3},
{ 4,1040,1039,  967,   0.565,   4.868,  384.172,  -68.2, 316.8,  36.0,   1.0,  4.0,  0.3},
{ 4,1041,1040,  967,   0.565,   4.868,  389.040,  -67.2, 320.8,  36.3,   1.0,  4.0,  0.3},
{ 4,1042,1041,  967,   0.565,   4.868,  393.908,  -66.3, 324.8,  36.6,   1.0,  4.0,  0.3},
{ 4,1043,1042,  967,   0.565,   4.868,  398.776,  -65.3, 328.8,  36.9,   1.0,  4.0,  0.3},
{ 4,1044,1043,  967,   0.565,   4.868,  403.644,  -64.3, 332.8,  37.2,   1.0,  4.0,  0.3},
{ 4,1045,1044,  967,   0.565,   4.868,  408.512,  -63.4, 336.8,  37.4,   1.0,  4.0,  0.3},
{ 4,1046,1045,  967,   0.565,   4.868,  413.379,  -62.4, 340.8,  37.7,   1.0,  4.0,  0.3},
{ 4,1047,1046,  967,   0.565,   4.868,  418.247,  -61.5, 344.8,  38.0,   1.0,  4.0,  0.3},
{ 4,1048,1047,  967,   0.565,   4.868,  423.115,  -60.5, 348.8,  38.3,   1.0,  4.0,  0.3},
{ 4,1049,1048,  967,   0.565,   4.868,  427.983,  -59.6, 352.8,  38.6,   1.0,  4.0,  0.3},
{ 4,1050,1049,  967,   0.565,   4.868,  432.851,  -58.6, 356.8,  38.8,   1.0,  4.0,  0.3},
{ 4,1051,1050,  967,   0.565,   4.868,  437.719,  -57.6, 360.8,  39.1,   1.0,  4.0,  0.3},
{ 4,1052,1051, 1005,   0.400,   4.947,  442.627,  -56.8, 364.4,  39.2,   0.8,  3.2, -0.0},
{ 4,1053,1052, 1005,   0.400,   4.947,  447.574,  -55.9, 367.6,  39.2,   0.8,  3.2, -0.0},
{ 4,1054,1053, 1005,   0.400,   4.947,  452.521,  -55.1, 370.8,  39.1,   0.8,  3.2, -0.0},
{ 4,1055,1054, 1005,   0.400,   4.947,  457.468,  -54.3, 373.9,  39.1,   0.8,  3.2, -0.0},
{ 4,1056,1055, 1005,   0.400,   4.947,  462.416,  -53.5, 377.1,  39.0,   0.8,  3.2, -0.0},
{ 4,1057,1056, 1005,   0.400,   4.947,  467.363,  -52.7, 380.2,  39.0,   0.8,  3.2, -0.0},
{ 4,1058,1057, 1005,   0.400,   4.947,  472.310,  -51.8, 383.4,  38.9,   0.8,  3.2, -0.0},
{ 4,1059,1058, 1005,   0.400,   4.947,  477.257,  -51.0, 386.6,  38.9,   0.8,  3.2, -0.0},
{ 4,1060,1059, 1005,   0.400,   4.947,  482.204,  -50.2, 389.7,  38.8,   0.8,  3.2, -0.0},
{ 4,1061,1060, 1005,   0.400,   4.947,  487.152,  -49.4, 392.9,  38.8,   0.8,  3.2, -0.0},
{ 4,1062,1061, 1005,   0.400,   4.947,  492.099,  -48.6, 396.0,  38.7,   0.8,  3.2, -0.0},
{ 4,1063,1062, 1005,   0.400,   4.947,  497.046,  -47.8, 399.2,  38.7,   0.8,  3.2, -0.0},
{ 4,1064,1063, 1005,   0.400,   4.947,  501.993,  -46.9, 402.4,  38.6,   0.8,  3.2, -0.0},
{ 4,1065,1064, 1005,   0.400,   4.947,  506.940,  -46.1, 405.5,  38.6,   0.8,  3.2, -0.0},
{ 4,1066,1065, 1005,   0.400,   4.947,  511.887,  -45.3, 408.7,  38.5,   0.8,  3.2, -0.0},
{ 4,1067,1066, 1005,   0.400,   4.947,  516.835,  -44.5, 411.9,  38.5,   0.8,  3.2, -0.0},
{ 4,1068,1067, 1005,   0.400,   4.947,  521.782,  -43.7, 415.0,  38.4,   0.8,  3.2, -0.0},
{ 4,1069,1068, 1005,   0.400,   4.947,  526.729,  -42.8, 418.2,  38.4,   0.8,  3.2, -0.0},
{ 4,1070,1069, 1005,   0.400,   4.947,  531.676,  -42.0, 421.3,  38.3,   0.8,  3.2, -0.0},
{ 4,1071,1070, 1005,   0.400,   4.947,  536.623,  -41.2, 424.5,  38.3,   0.8,  3.2, -0.0},
{ 4,1072,1071, 1005,   0.400,   4.947,  541.571,  -40.4, 427.7,  38.2,   0.8,  3.2, -0.0},
{ 4,1073,1072, 1005,   0.400,   4.947,  546.518,  -39.6, 430.8,  38.2,   0.8,  3.2, -0.0},
{ 4,1074,1073, 1005,   0.400,   4.947,  551.465,  -38.8, 434.0,  38.2,   0.8,  3.2, -0.0},
{ 4,1075,1074, 1005,   0.400,   4.947,  556.412,  -37.9, 437.1,  38.1,   0.8,  3.2, -0.0},
{ 4,1076,1075, 1005,   0.400,   4.947,  561.359,  -37.1, 440.3,  38.1,   0.8,  3.2, -0.0},
{ 4,1077,1076, 1005,   0.400,   4.947,  566.307,  -36.3, 443.5,  38.0,   0.8,  3.2, -0.0},
{ 4,1078,1077, 1005,   0.400,   4.947,  571.254,  -35.5, 446.6,  38.0,   0.8,  3.2, -0.0},
{ 4,1079,1078, 1005,   0.400,   4.947,  576.201,  -34.7, 449.8,  37.9,   0.8,  3.2, -0.0},
{ 4,1080,1079, 1005,   0.400,   4.947,  581.148,  -33.8, 453.0,  37.9,   0.8,  3.2, -0.0},
{ 4,1081,1080, 1005,   0.400,   4.947,  586.095,  -33.0, 456.1,  37.8,   0.8,  3.2, -0.0},
{ 4,1082,1081, 1005,   0.400,   4.947,  591.042,  -32.2, 459.3,  37.8,   0.8,  3.2, -0.0},
{ 4,1083,1082, 1005,   0.400,   4.947,  595.990,  -31.4, 462.4,  37.7,   0.8,  3.2, -0.0},
{ 4,1084,1083, 1005,   0.400,   4.947,  600.937,  -30.6, 465.6,  37.7,   0.8,  3.2, -0.0},
{ 4,1085,1084, 1005,   0.400,   4.947,  605.884,  -29.8, 468.8,  37.6,   0.8,  3.2, -0.0},
{ 4,1086,1085, 1005,   0.400,   4.947,  610.831,  -28.9, 471.9,  37.6,   0.8,  3.2, -0.0},
{ 4,1087,1086, 1005,   0.400,   4.947,  615.778,  -28.1, 475.1,  37.5,   0.8,  3.2, -0.0},
{ 4,1088,1087, 1005,   0.400,   4.947,  620.726,  -27.3, 478.2,  37.5,   0.8,  3.2, -0.0},
{ 4,1089,1088, 1005,   0.400,   4.947,  625.673,  -26.5, 481.4,  37.4,   0.8,  3.2, -0.0},
{ 4,1090,1089, 1005,   0.400,   4.947,  630.620,  -25.7, 484.6,  37.4,   0.8,  3.2, -0.0},
{ 4,1091,1090, 1005,   0.400,   4.947,  635.567,  -24.8, 487.7,  37.3,   0.8,  3.2, -0.0},
{ 4,1092,1091, 1005,   0.400,   4.947,  640.514,  -24.0, 490.9,  37.3,   0.8,  3.2, -0.0},
{ 4,1093,1051, 1046,   0.560,   4.984,  442.645,  -57.2, 364.0,  39.0,  -0.2,  2.3, -0.6},
{ 4,1094,1093, 1046,   0.560,   4.984,  447.629,  -57.4, 366.2,  38.4,  -0.2,  2.3, -0.6},
{ 4,1095,1094, 1046,   0.560,   4.984,  452.613,  -57.6, 368.5,  37.8,  -0.2,  2.3, -0.6},
{ 4,1096,1095, 1046,   0.560,   4.984,  457.597,  -57.7, 370.8,  37.3,  -0.2,  2.3, -0.6},
{ 4,1097,1096, 1046,   0.560,   4.984,  462.581,  -57.9, 373.0,  36.7,  -0.2,  2.3, -0.6},
{ 4,1098,1097, 1046,   0.560,   4.984,  467.565,  -58.0, 375.3,  36.1,  -0.2,  2.3, -0.6},
{ 4,1099,1098, 1046,   0.560,   4.984,  472.549,  -58.2, 377.6,  35.6,  -0.2,  2.3, -0.6},
{ 4,1100,1099, 1046,   0.560,   4.984,  477.533,  -58.3, 379.8,  35.0,  -0.2,  2.3, -0.6},
{ 4,1101,1100, 1046,   0.560,   4.984,  482.517,  -58.5, 382.1,  34.4,  -0.2,  2.3, -0.6},
{ 4,1102,1101, 1046,   0.560,   4.984,  487.501,  -58.6, 384.4,  33.9,  -0.2,  2.3, -0.6},
{ 4,1103,1102, 1046,   0.560,   4.984,  492.485,  -58.8, 386.6,  33.3,  -0.2,  2.3, -0.6},
{ 4,1104,1103, 1046,   0.560,   4.984,  497.469,  -58.9, 388.9,  32.7,  -0.2,  2.3, -0.6},
{ 4,1105,1104, 1046,   0.560,   4.984,  502.453,  -59.1, 391.1,  32.2,  -0.2,  2.3, -0.6},
{ 4,1106,1105, 1046,   0.560,   4.984,  507.437,  -59.3, 393.4,  31.6,  -0.2,  2.3, -0.6},
{ 4,1107,1106, 1046,   0.560,   4.984,  512.421,  -59.4, 395.7,  31.0,  -0.2,  2.3, -0.6},
{ 4,1108,1107, 1046,   0.560,   4.984,  517.405,  -59.6, 397.9,  30.5,  -0.2,  2.3, -0.6},
{ 4,1109,1108, 1046,   0.560,   4.984,  522.390,  -59.7, 400.2,  29.9,  -0.2,  2.3, -0.6},
{ 4,1110,1109, 1046,   0.560,   4.984,  527.374,  -59.9, 402.5,  29.3,  -0.2,  2.3, -0.6},
{ 4,1111,1110, 1046,   0.560,   4.984,  532.358,  -60.0, 404.7,  28.8,  -0.2,  2.3, -0.6},
{ 4,1112,1111, 1046,   0.560,   4.984,  537.342,  -60.2, 407.0,  28.2,  -0.2,  2.3, -0.6},
{ 4,1113,1112, 1046,   0.560,   4.984,  542.326,  -60.3, 409.2,  27.6,  -0.2,  2.3, -0.6},
{ 4,1114,1113, 1046,   0.560,   4.984,  547.310,  -60.5, 411.5,  27.1,  -0.2,  2.3, -0.6},
{ 4,1115,1114, 1046,   0.560,   4.984,  552.294,  -60.6, 413.8,  26.5,  -0.2,  2.3, -0.6},
{ 4,1116,1115, 1046,   0.560,   4.984,  557.278,  -60.8, 416.0,  25.9,  -0.2,  2.3, -0.6},
{ 4,1117,1116, 1046,   0.560,   4.984,  562.262,  -61.0, 418.3,  25.4,  -0.2,  2.3, -0.6},
{ 4,1118,1117, 1046,   0.560,   4.984,  567.246,  -61.1, 420.6,  24.8,  -0.2,  2.3, -0.6},
{ 4,1119,1118, 1046,   0.560,   4.984,  572.230,  -61.3, 422.8,  24.2,  -0.2,  2.3, -0.6},
{ 4,1120,1119, 1046,   0.560,   4.984,  577.214,  -61.4, 425.1,  23.7,  -0.2,  2.3, -0.6},
{ 4,1121,1120, 1046,   0.560,   4.984,  582.198,  -61.6, 427.4,  23.1,  -0.2,  2.3, -0.6},
{ 4,1122,1121, 1046,   0.560,   4.984,  587.182,  -61.7, 429.6,  22.5,  -0.2,  2.3, -0.6},
{ 4,1123,1122, 1046,   0.560,   4.984,  592.166,  -61.9, 431.9,  22.0,  -0.2,  2.3, -0.6},
{ 4,1124,1123, 1046,   0.560,   4.984,  597.150,  -62.0, 434.1,  21.4,  -0.2,  2.3, -0.6},
{ 4,1125,1124, 1046,   0.560,   4.984,  602.134,  -62.2, 436.4,  20.8,  -0.2,  2.3, -0.6},
{ 4,1126,1125, 1046,   0.560,   4.984,  607.118,  -62.4, 438.7,  20.3,  -0.2,  2.3, -0.6},
{ 4,1127, 902, 1056,   0.400,   4.771,  262.867,  -88.0, 207.1,  32.5,  -1.7,  0.7,  0.9},
{ 4,1128,1127, 1056,   0.400,   4.771,  267.638,  -89.6, 207.8,  33.4,  -1.7,  0.7,  0.9},
{ 4,1129,1128, 1056,   0.400,   4.771,  272.408,  -91.3, 208.6,  34.3,  -1.7,  0.7,  0.9},
{ 4,1130,1129, 1056,   0.400,   4.771,  277.179,  -93.0, 209.3,  35.2,  -1.7,  0.7,  0.9},
{ 4,1131,1130, 1056,   0.400,   4.771,  281.950,  -94.7, 210.0,  36.1,  -1.7,  0.7,  0.9},
{ 4,1132,1131, 1056,   0.400,   4.771,  286.721,  -96.3, 210.7,  37.1,  -1.7,  0.7,  0.9},
{ 4,1133,1132, 1056,   0.400,   4.771,  291.492,  -98.0, 211.4,  38.0,  -1.7,  0.7,  0.9},
{ 4,1134,1133, 1056,   0.400,   4.771,  296.263,  -99.7, 212.1,  38.9,  -1.7,  0.7,  0.9},
{ 4,1135,1134, 1056,   0.400,   4.771,  301.034, -101.4, 212.8,  39.8,  -1.7,  0.7,  0.9},
{ 4,1136,1135, 1056,   0.400,   4.771,  305.805, -103.0, 213.5,  40.7,  -1.7,  0.7,  0.9},
{ 4,1137,1136, 1056,   0.400,   4.771,  310.576, -104.7, 214.2,  41.7,  -1.7,  0.7,  0.9},
{ 4,1138,1137, 1056,   0.400,   4.771,  315.347, -106.4, 214.9,  42.6,  -1.7,  0.7,  0.9},
{ 4,1139,1138, 1056,   0.400,   4.771,  320.117, -108.0, 215.6,  43.5,  -1.7,  0.7,  0.9},
{ 4,1140,1139, 1056,   0.400,   4.771,  324.888, -109.7, 216.3,  44.4,  -1.7,  0.7,  0.9},
{ 4,1141,1140, 1056,   0.400,   4.771,  329.659, -111.4, 217.0,  45.4,  -1.7,  0.7,  0.9},
{ 4,1142,1141, 1056,   0.400,   4.771,  334.430, -113.1, 217.7,  46.3,  -1.7,  0.7,  0.9},
{ 4,1143,1142, 1056,   0.400,   4.771,  339.201, -114.7, 218.4,  47.2,  -1.7,  0.7,  0.9},
{ 4,1144,1143, 1056,   0.400,   4.771,  343.972, -116.4, 219.1,  48.1,  -1.7,  0.7,  0.9},
{ 4,1145,1144, 1056,   0.400,   4.771,  348.743, -118.1, 219.8,  49.0,  -1.7,  0.7,  0.9},
{ 4,1146,1145, 1066,   0.240,   4.786,  353.521, -120.6, 221.7,  49.0,  -3.4,  3.0, -1.1},
{ 4,1147,1146, 1066,   0.240,   4.786,  358.308, -123.9, 224.6,  47.9,  -3.4,  3.0, -1.1},
{ 4,1148,1147, 1066,   0.240,   4.786,  363.094, -127.3, 227.6,  46.8,  -3.4,  3.0, -1.1},
{ 4,1149,1148, 1066,   0.240,   4.786,  367.880, -130.6, 230.5,  45.8,  -3.4,  3.0, -1.1},
{ 4,1150,1149, 1066,   0.240,   4.786,  372.666, -134.0, 233.5,  44.7,  -3.4,  3.0, -1.1},
{ 4,1151,1150, 1066,   0.240,   4.786,  377.452, -137.3, 236.4,  43.7,  -3.4,  3.0, -1.1},
{ 4,1152,1151, 1066,   0.240,   4.786,  382.238, -140.7, 239.4,  42.6,  -3.4,  3.0, -1.1},
{ 4,1153,1152, 1066,   0.240,   4.786,  387.024, -144.0, 242.3,  41.5,  -3.4,  3.0, -1.1},
{ 4,1154,1145, 1068,   0.323,   3.418,  352.838, -120.6, 220.2,  49.5,  -3.3,  0.1,  0.0},
{ 4,1155,1154, 1068,   0.323,   3.418,  356.256, -123.9, 220.4,  49.5,  -3.3,  0.1,  0.0},
{ 4,1156,1155, 1074,   0.240,   4.497,  360.214, -127.5, 221.0,  49.1,  -4.0,  1.2, -0.7},
{ 4,1157,1156, 1074,   0.240,   4.497,  364.710, -131.5, 222.2,  48.4,  -4.0,  1.2, -0.7},
{ 4,1158,1157, 1074,   0.240,   4.497,  369.207, -135.5, 223.4,  47.7,  -4.0,  1.2, -0.7},
{ 4,1159,1158, 1074,   0.240,   4.497,  373.704, -139.5, 224.6,  47.0,  -4.0,  1.2, -0.7},
{ 4,1160,1159, 1074,   0.240,   4.497,  378.200, -143.5, 225.8,  46.3,  -4.0,  1.2, -0.7},
{ 4,1161,1160, 1074,   0.240,   4.497,  382.697, -147.5, 226.9,  45.5,  -4.0,  1.2, -0.7},
{ 4,1162,1161, 1074,   0.240,   4.497,  387.193, -151.5, 228.1,  44.8,  -4.0,  1.2, -0.7},
{ 4,1163,1162, 1074,   0.240,   4.497,  391.690, -155.4, 229.3,  44.1,  -4.0,  1.2, -0.7},
{ 4,1164, 834, 1081,   0.560,   4.577,  147.899,  -51.4, 118.8,  43.1,  -2.0,  1.2, -2.3},
{ 4,1165,1164, 1081,   0.560,   4.577,  152.476,  -53.4, 120.0,  40.9,  -2.0,  1.2, -2.3},
{ 4,1166,1165, 1081,   0.560,   4.577,  157.053,  -55.4, 121.3,  38.6,  -2.0,  1.2, -2.3},
{ 4,1167,1166, 1081,   0.560,   4.577,  161.631,  -57.4, 122.5,  36.4,  -2.0,  1.2, -2.3},
{ 4,1168,1167, 1081,   0.560,   4.577,  166.208,  -59.5, 123.8,  34.1,  -2.0,  1.2, -2.3},
{ 4,1169,1168, 1081,   0.560,   4.577,  170.785,  -61.5, 125.0,  31.9,  -2.0,  1.2, -2.3},
{ 4,1170,1169, 1081,   0.560,   4.577,  175.363,  -63.5, 126.3,  29.6,  -2.0,  1.2, -2.3},
{ 4,1171,1170, 1116,   0.240,   5.003,  180.153,  -65.8, 127.3,  28.4,  -2.6,  0.8, -0.1},
{ 4,1172,1171, 1116,   0.240,   5.003,  185.156,  -68.4, 128.0,  28.3,  -2.6,  0.8, -0.1},
{ 4,1173,1172, 1116,   0.240,   5.003,  190.160,  -71.1, 128.8,  28.2,  -2.6,  0.8, -0.1},
{ 4,1174,1173, 1116,   0.240,   5.003,  195.163,  -73.7, 129.6,  28.0,  -2.6,  0.8, -0.1},
{ 4,1175,1174, 1116,   0.240,   5.003,  200.167,  -76.3, 130.4,  27.9,  -2.6,  0.8, -0.1},
{ 4,1176,1175, 1116,   0.240,   5.003,  205.170,  -78.9, 131.1,  27.8,  -2.6,  0.8, -0.1},
{ 4,1177,1176, 1116,   0.240,   5.003,  210.174,  -81.5, 131.9,  27.6,  -2.6,  0.8, -0.1},
{ 4,1178,1177, 1116,   0.240,   5.003,  215.177,  -84.1, 132.7,  27.5,  -2.6,  0.8, -0.1},
{ 4,1179,1178, 1116,   0.240,   5.003,  220.181,  -86.7, 133.4,  27.4,  -2.6,  0.8, -0.1},
{ 4,1180,1179, 1116,   0.240,   5.003,  225.184,  -89.4, 134.2,  27.3,  -2.6,  0.8, -0.1},
{ 4,1181,1180, 1116,   0.240,   5.003,  230.188,  -92.0, 135.0,  27.1,  -2.6,  0.8, -0.1},
{ 4,1182,1181, 1116,   0.240,   5.003,  235.191,  -94.6, 135.8,  27.0,  -2.6,  0.8, -0.1},
{ 4,1183,1182, 1116,   0.240,   5.003,  240.195,  -97.2, 136.5,  26.9,  -2.6,  0.8, -0.1},
{ 4,1184,1183, 1116,   0.240,   5.003,  245.198,  -99.8, 137.3,  26.7,  -2.6,  0.8, -0.1},
{ 4,1185,1184, 1116,   0.240,   5.003,  250.201, -102.4, 138.1,  26.6,  -2.6,  0.8, -0.1},
{ 4,1186,1185, 1116,   0.240,   5.003,  255.205, -105.1, 138.9,  26.5,  -2.6,  0.8, -0.1},
{ 4,1187,1186, 1116,   0.240,   5.003,  260.208, -107.7, 139.6,  26.3,  -2.6,  0.8, -0.1},
{ 4,1188,1187, 1116,   0.240,   5.003,  265.212, -110.3, 140.4,  26.2,  -2.6,  0.8, -0.1},
{ 4,1189,1188, 1116,   0.240,   5.003,  270.215, -112.9, 141.2,  26.1,  -2.6,  0.8, -0.1},
{ 4,1190,1189, 1116,   0.240,   5.003,  275.219, -115.5, 141.9,  25.9,  -2.6,  0.8, -0.1},
{ 4,1191,1190, 1116,   0.240,   5.003,  280.222, -118.1, 142.7,  25.8,  -2.6,  0.8, -0.1},
{ 4,1192,1191, 1116,   0.240,   5.003,  285.226, -120.7, 143.5,  25.7,  -2.6,  0.8, -0.1},
{ 4,1193,1192, 1116,   0.240,   5.003,  290.229, -123.4, 144.3,  25.6,  -2.6,  0.8, -0.1},
{ 4,1194,1193, 1116,   0.240,   5.003,  295.233, -126.0, 145.0,  25.4,  -2.6,  0.8, -0.1},
{ 4,1195,1194, 1116,   0.240,   5.003,  300.236, -128.6, 145.8,  25.3,  -2.6,  0.8, -0.1},
{ 4,1196,1195, 1116,   0.240,   5.003,  305.240, -131.2, 146.6,  25.2,  -2.6,  0.8, -0.1},
{ 4,1197,1196, 1116,   0.240,   5.003,  310.243, -133.8, 147.3,  25.0,  -2.6,  0.8, -0.1},
{ 4,1198,1197, 1116,   0.240,   5.003,  315.246, -136.4, 148.1,  24.9,  -2.6,  0.8, -0.1},
{ 4,1199,1198, 1116,   0.240,   5.003,  320.250, -139.1, 148.9,  24.8,  -2.6,  0.8, -0.1},
{ 4,1200,1199, 1116,   0.240,   5.003,  325.253, -141.7, 149.7,  24.6,  -2.6,  0.8, -0.1},
{ 4,1201,1200, 1116,   0.240,   5.003,  330.257, -144.3, 150.4,  24.5,  -2.6,  0.8, -0.1},
{ 4,1202,1201, 1116,   0.240,   5.003,  335.260, -146.9, 151.2,  24.4,  -2.6,  0.8, -0.1},
{ 4,1203,1202, 1116,   0.240,   5.003,  340.264, -149.5, 152.0,  24.2,  -2.6,  0.8, -0.1},
{ 4,1204,1203, 1116,   0.240,   5.003,  345.267, -152.1, 152.8,  24.1,  -2.6,  0.8, -0.1},
{ 4,1205,1204, 1116,   0.240,   5.003,  350.271, -154.7, 153.5,  24.0,  -2.6,  0.8, -0.1},
{ 4,1206,1205, 1116,   0.240,   5.003,  355.274, -157.4, 154.3,  23.9,  -2.6,  0.8, -0.1},
{ 4,1207,1206, 1116,   0.240,   5.003,  360.278, -160.0, 155.1,  23.7,  -2.6,  0.8, -0.1},
{ 4,1208,1207, 1116,   0.240,   5.003,  365.281, -162.6, 155.8,  23.6,  -2.6,  0.8, -0.1},
{ 4,1209,1208, 1116,   0.240,   5.003,  370.285, -165.2, 156.6,  23.5,  -2.6,  0.8, -0.1},
{ 4,1210,1209, 1116,   0.240,   5.003,  375.288, -167.8, 157.4,  23.3,  -2.6,  0.8, -0.1},
{ 4,1211,1210, 1116,   0.240,   5.003,  380.291, -170.4, 158.2,  23.2,  -2.6,  0.8, -0.1},
{ 4,1212,1211, 1116,   0.240,   5.003,  385.295, -173.1, 158.9,  23.1,  -2.6,  0.8, -0.1},
{ 4,1213,1170, 1118,   0.483,   4.539,  179.921,  -64.4, 127.2,  30.6,   0.1,  0.6,  4.1},
{ 4,1214,1213, 1118,   0.483,   4.539,  184.460,  -64.3, 127.7,  34.7,   0.1,  0.6,  4.1},
{ 4,1215,1214, 1118,   0.483,   4.539,  188.999,  -64.2, 128.3,  38.8,   0.1,  0.6,  4.1},
{ 4,1216,1215, 1118,   0.483,   4.539,  193.539,  -64.0, 128.9,  42.9,   0.1,  0.6,  4.1},
{ 4,1217,1216, 1141,   0.248,   4.971,  198.294,  -64.9, 129.7,  44.4,  -1.9,  1.1, -1.2},
{ 4,1218,1217, 1141,   0.248,   4.971,  203.264,  -66.8, 130.7,  43.2,  -1.9,  1.1, -1.2},
{ 4,1219,1218, 1141,   0.248,   4.971,  208.235,  -68.7, 131.8,  42.0,  -1.9,  1.1, -1.2},
{ 4,1220,1219, 1141,   0.248,   4.971,  213.206,  -70.6, 132.8,  40.8,  -1.9,  1.1, -1.2},
{ 4,1221,1220, 1141,   0.248,   4.971,  218.177,  -72.5, 133.9,  39.6,  -1.9,  1.1, -1.2},
{ 4,1222,1221, 1141,   0.248,   4.971,  223.147,  -74.4, 134.9,  38.4,  -1.9,  1.1, -1.2},
{ 4,1223,1222, 1141,   0.248,   4.971,  228.118,  -76.3, 136.0,  37.2,  -1.9,  1.1, -1.2},
{ 4,1224,1223, 1141,   0.248,   4.971,  233.089,  -78.2, 137.0,  36.0,  -1.9,  1.1, -1.2},
{ 4,1225,1224, 1141,   0.248,   4.971,  238.060,  -80.1, 138.1,  34.8,  -1.9,  1.1, -1.2},
{ 4,1226,1225, 1141,   0.248,   4.971,  243.030,  -82.0, 139.1,  33.6,  -1.9,  1.1, -1.2},
{ 4,1227,1226, 1141,   0.248,   4.971,  248.001,  -83.9, 140.2,  32.4,  -1.9,  1.1, -1.2},
{ 4,1228,1227, 1141,   0.248,   4.971,  252.972,  -85.8, 141.2,  31.2,  -1.9,  1.1, -1.2},
{ 4,1229,1228, 1141,   0.248,   4.971,  257.943,  -87.8, 142.3,  30.0,  -1.9,  1.1, -1.2},
{ 4,1230,1229, 1141,   0.248,   4.971,  262.914,  -89.7, 143.3,  28.8,  -1.9,  1.1, -1.2},
{ 4,1231,1230, 1141,   0.248,   4.971,  267.884,  -91.6, 144.4,  27.6,  -1.9,  1.1, -1.2},
{ 4,1232,1231, 1141,   0.248,   4.971,  272.855,  -93.5, 145.5,  26.4,  -1.9,  1.1, -1.2},
{ 4,1233, 577, 1158,   0.716,   4.827,   30.857,  -19.1,  59.1,   9.6,  -0.1,  3.7,  0.6},
{ 4,1234,1233, 1158,   0.716,   4.827,   35.683,  -19.2,  62.7,  10.2,  -0.1,  3.7,  0.6},
{ 4,1235,1234, 1158,   0.716,   4.827,   40.510,  -19.2,  66.4,  10.9,  -0.1,  3.7,  0.6},
{ 4,1236,1235, 1158,   0.716,   4.827,   45.336,  -19.3,  70.1,  11.5,  -0.1,  3.7,  0.6},
{ 4,1237,1236, 1158,   0.716,   4.827,   50.163,  -19.3,  73.8,  12.2,  -0.1,  3.7,  0.6},
{ 4,1238,1237, 1158,   0.716,   4.827,   54.989,  -19.4,  77.5,  12.8,  -0.1,  3.7,  0.6},
{ 4,1239,1238, 1158,   0.716,   4.827,   59.816,  -19.5,  81.2,  13.5,  -0.1,  3.7,  0.6},
{ 4,1240,1239, 1158,   0.716,   4.827,   64.643,  -19.5,  84.9,  14.1,  -0.1,  3.7,  0.6},
{ 4,1241,1240, 1158,   0.716,   4.827,   69.469,  -19.6,  88.6,  14.8,  -0.1,  3.7,  0.6},
{ 4,1242,1241, 1158,   0.716,   4.827,   74.296,  -19.7,  92.2,  15.4,  -0.1,  3.7,  0.6},
{ 4,1243,1242, 1158,   0.716,   4.827,   79.122,  -19.7,  95.9,  16.1,  -0.1,  3.7,  0.6},
{ 4,1244,1243, 1158,   0.716,   4.827,   83.949,  -19.8,  99.6,  16.7,  -0.1,  3.7,  0.6},
{ 4,1245,1244, 1158,   0.716,   4.827,   88.775,  -19.8, 103.3,  17.4,  -0.1,  3.7,  0.6},
{ 4,1246,1245, 1158,   0.716,   4.827,   93.602,  -19.9, 107.0,  18.0,  -0.1,  3.7,  0.6},
{ 4,1247,1246, 1158,   0.716,   4.827,   98.428,  -20.0, 110.7,  18.7,  -0.1,  3.7,  0.6},
{ 4,1248,1247, 1158,   0.716,   4.827,  103.255,  -20.0, 114.4,  19.3,  -0.1,  3.7,  0.6},
{ 4,1249,1248, 1158,   0.716,   4.827,  108.082,  -20.1, 118.1,  20.0,  -0.1,  3.7,  0.6},
{ 4,1250,1249, 1158,   0.716,   4.827,  112.908,  -20.2, 121.7,  20.6,  -0.1,  3.7,  0.6},
{ 4,1251,1250, 1158,   0.716,   4.827,  117.735,  -20.2, 125.4,  21.3,  -0.1,  3.7,  0.6},
{ 4,1252,1251, 1158,   0.716,   4.827,  122.561,  -20.3, 129.1,  21.9,  -0.1,  3.7,  0.6},
{ 4,1253,1252, 1174,   0.400,   4.842,  127.395,  -20.2, 132.3,  22.6,   0.3,  2.6,  0.7},
{ 4,1254,1253, 1174,   0.400,   4.842,  132.237,  -19.9, 134.9,  23.4,   0.3,  2.6,  0.7},
{ 4,1255,1254, 1174,   0.400,   4.842,  137.079,  -19.7, 137.5,  24.1,   0.3,  2.6,  0.7},
{ 4,1256,1255, 1174,   0.400,   4.842,  141.921,  -19.4, 140.1,  24.8,   0.3,  2.6,  0.7},
{ 4,1257,1256, 1174,   0.400,   4.842,  146.763,  -19.2, 142.7,  25.6,   0.3,  2.6,  0.7},
{ 4,1258,1257, 1174,   0.400,   4.842,  151.605,  -18.9, 145.3,  26.3,   0.3,  2.6,  0.7},
{ 4,1259,1258, 1174,   0.400,   4.842,  156.447,  -18.7, 147.9,  27.0,   0.3,  2.6,  0.7},
{ 4,1260,1259, 1174,   0.400,   4.842,  161.288,  -18.4, 150.5,  27.8,   0.3,  2.6,  0.7},
{ 4,1261,1260, 1174,   0.400,   4.842,  166.130,  -18.1, 153.1,  28.5,   0.3,  2.6,  0.7},
{ 4,1262,1261, 1174,   0.400,   4.842,  170.972,  -17.9, 155.7,  29.2,   0.3,  2.6,  0.7},
{ 4,1263,1262, 1174,   0.400,   4.842,  175.814,  -17.6, 158.3,  30.0,   0.3,  2.6,  0.7},
{ 4,1264,1263, 1174,   0.400,   4.842,  180.656,  -17.4, 160.9,  30.7,   0.3,  2.6,  0.7},
{ 4,1265,1264, 1174,   0.400,   4.842,  185.498,  -17.1, 163.5,  31.5,   0.3,  2.6,  0.7},
{ 4,1266,1265, 1174,   0.400,   4.842,  190.340,  -16.9, 166.1,  32.2,   0.3,  2.6,  0.7},
{ 4,1267,1266, 1174,   0.400,   4.842,  195.181,  -16.6, 168.7,  32.9,   0.3,  2.6,  0.7},
{ 4,1268,1267, 1174,   0.400,   4.842,  200.023,  -16.4, 171.3,  33.7,   0.3,  2.6,  0.7},
{ 4,1269,1268, 1174,   0.400,   4.842,  204.865,  -16.1, 173.9,  34.4,   0.3,  2.6,  0.7},
{ 4,1270,1269, 1174,   0.400,   4.842,  209.707,  -15.8, 176.4,  35.1,   0.3,  2.6,  0.7},
{ 4,1271,1270, 1174,   0.400,   4.842,  214.549,  -15.6, 179.0,  35.9,   0.3,  2.6,  0.7},
{ 4,1272,1271, 1183,   0.248,   4.477,  219.208,  -14.5, 181.5,  35.5,   1.8,  2.3, -1.5},
{ 4,1273,1272, 1183,   0.248,   4.477,  223.685,  -12.7, 183.8,  34.1,   1.8,  2.3, -1.5},
{ 4,1274,1273, 1183,   0.248,   4.477,  228.162,  -10.8, 186.1,  32.6,   1.8,  2.3, -1.5},
{ 4,1275,1274, 1183,   0.248,   4.477,  232.639,   -9.0, 188.4,  31.1,   1.8,  2.3, -1.5},
{ 4,1276,1275, 1183,   0.248,   4.477,  237.116,   -7.2, 190.7,  29.7,   1.8,  2.3, -1.5},
{ 4,1277,1276, 1183,   0.248,   4.477,  241.593,   -5.3, 193.0,  28.2,   1.8,  2.3, -1.5},
{ 4,1278,1277, 1183,   0.248,   4.477,  246.071,   -3.5, 195.3,  26.7,   1.8,  2.3, -1.5},
{ 4,1279,1278, 1198,   0.240,   4.798,  250.708,   -2.9, 197.2,  26.1,  -0.8,  1.6,  0.3},
{ 4,1280,1279, 1198,   0.240,   4.798,  255.506,   -3.7, 198.8,  26.4,  -0.8,  1.6,  0.3},
{ 4,1281,1280, 1198,   0.240,   4.798,  260.304,   -4.5, 200.3,  26.7,  -0.8,  1.6,  0.3},
{ 4,1282,1281, 1198,   0.240,   4.798,  265.103,   -5.3, 201.9,  26.9,  -0.8,  1.6,  0.3},
{ 4,1283,1282, 1198,   0.240,   4.798,  269.901,   -6.1, 203.5,  27.2,  -0.8,  1.6,  0.3},
{ 4,1284,1283, 1198,   0.240,   4.798,  274.699,   -6.8, 205.0,  27.5,  -0.8,  1.6,  0.3},
{ 4,1285,1284, 1198,   0.240,   4.798,  279.497,   -7.6, 206.6,  27.7,  -0.8,  1.6,  0.3},
{ 4,1286,1285, 1198,   0.240,   4.798,  284.295,   -8.4, 208.1,  28.0,  -0.8,  1.6,  0.3},
{ 4,1287,1286, 1198,   0.240,   4.798,  289.093,   -9.2, 209.7,  28.3,  -0.8,  1.6,  0.3},
{ 4,1288,1287, 1198,   0.240,   4.798,  293.891,  -10.0, 211.2,  28.6,  -0.8,  1.6,  0.3},
{ 4,1289,1288, 1198,   0.240,   4.798,  298.690,  -10.7, 212.8,  28.8,  -0.8,  1.6,  0.3},
{ 4,1290,1289, 1198,   0.240,   4.798,  303.488,  -11.5, 214.3,  29.1,  -0.8,  1.6,  0.3},
{ 4,1291,1290, 1198,   0.240,   4.798,  308.286,  -12.3, 215.9,  29.4,  -0.8,  1.6,  0.3},
{ 4,1292,1291, 1201,   0.240,   3.783,  312.577,  -13.2, 216.8,  27.7,  -1.0,  0.2, -3.6},
{ 4,1293,1292, 1201,   0.240,   3.783,  316.360,  -14.1, 217.0,  24.1,  -1.0,  0.2, -3.6},
{ 4,1294,1291, 1235,   0.240,   4.993,  313.181,  -12.6, 218.4,  29.3,   0.3,  3.4, -0.4},
{ 4,1295,1294, 1235,   0.240,   4.993,  318.174,  -12.3, 221.8,  28.9,   0.3,  3.4, -0.4},
{ 4,1296,1295, 1235,   0.240,   4.993,  323.167,  -12.0, 225.3,  28.5,   0.3,  3.4, -0.4},
{ 4,1297,1296, 1235,   0.240,   4.993,  328.159,  -11.7, 228.7,  28.1,   0.3,  3.4, -0.4},
{ 4,1298,1297, 1235,   0.240,   4.993,  333.152,  -11.4, 232.2,  27.7,   0.3,  3.4, -0.4},
{ 4,1299,1298, 1235,   0.240,   4.993,  338.144,  -11.1, 235.6,  27.4,   0.3,  3.4, -0.4},
{ 4,1300,1299, 1235,   0.240,   4.993,  343.137,  -10.9, 239.0,  27.0,   0.3,  3.4, -0.4},
{ 4,1301,1300, 1235,   0.240,   4.993,  348.130,  -10.6, 242.5,  26.6,   0.3,  3.4, -0.4},
{ 4,1302,1301, 1235,   0.240,   4.993,  353.122,  -10.3, 245.9,  26.2,   0.3,  3.4, -0.4},
{ 4,1303,1302, 1235,   0.240,   4.993,  358.115,  -10.0, 249.3,  25.8,   0.3,  3.4, -0.4},
{ 4,1304,1303, 1235,   0.240,   4.993,  363.108,   -9.7, 252.8,  25.4,   0.3,  3.4, -0.4},
{ 4,1305,1304, 1235,   0.240,   4.993,  368.100,   -9.5, 256.2,  25.0,   0.3,  3.4, -0.4},
{ 4,1306,1305, 1235,   0.240,   4.993,  373.093,   -9.2, 259.7,  24.6,   0.3,  3.4, -0.4},
{ 4,1307,1306, 1235,   0.240,   4.993,  378.086,   -8.9, 263.1,  24.2,   0.3,  3.4, -0.4},
{ 4,1308,1307, 1235,   0.240,   4.993,  383.078,   -8.6, 266.5,  23.9,   0.3,  3.4, -0.4},
{ 4,1309,1308, 1235,   0.240,   4.993,  388.071,   -8.3, 270.0,  23.5,   0.3,  3.4, -0.4},
{ 4,1310,1309, 1235,   0.240,   4.993,  393.064,   -8.0, 273.4,  23.1,   0.3,  3.4, -0.4},
{ 4,1311,1310, 1235,   0.240,   4.993,  398.056,   -7.8, 276.9,  22.7,   0.3,  3.4, -0.4},
{ 4,1312,1311, 1235,   0.240,   4.993,  403.049,   -7.5, 280.3,  22.3,   0.3,  3.4, -0.4},
{ 4,1313,1312, 1235,   0.240,   4.993,  408.041,   -7.2, 283.7,  21.9,   0.3,  3.4, -0.4},
{ 4,1314,1313, 1235,   0.240,   4.993,  413.034,   -6.9, 287.2,  21.5,   0.3,  3.4, -0.4},
{ 4,1315,1314, 1235,   0.240,   4.993,  418.027,   -6.6, 290.6,  21.1,   0.3,  3.4, -0.4},
{ 4,1316,1315, 1235,   0.240,   4.993,  423.019,   -6.4, 294.0,  20.7,   0.3,  3.4, -0.4},
{ 4,1317,1316, 1235,   0.240,   4.993,  428.012,   -6.1, 297.5,  20.4,   0.3,  3.4, -0.4},
{ 4,1318,1317, 1235,   0.240,   4.993,  433.005,   -5.8, 300.9,  20.0,   0.3,  3.4, -0.4},
{ 4,1319,1318, 1235,   0.240,   4.993,  437.997,   -5.5, 304.4,  19.6,   0.3,  3.4, -0.4},
{ 4,1320,1319, 1235,   0.240,   4.993,  442.990,   -5.2, 307.8,  19.2,   0.3,  3.4, -0.4},
{ 4,1321,1320, 1235,   0.240,   4.993,  447.983,   -4.9, 311.2,  18.8,   0.3,  3.4, -0.4},
{ 4,1322,1321, 1235,   0.240,   4.993,  452.975,   -4.7, 314.7,  18.4,   0.3,  3.4, -0.4},
{ 4,1323,1322, 1235,   0.240,   4.993,  457.968,   -4.4, 318.1,  18.0,   0.3,  3.4, -0.4},
{ 4,1324,1323, 1235,   0.240,   4.993,  462.961,   -4.1, 321.6,  17.6,   0.3,  3.4, -0.4},
{ 4,1325,1324, 1235,   0.240,   4.993,  467.953,   -3.8, 325.0,  17.2,   0.3,  3.4, -0.4},
{ 4,1326,1325, 1235,   0.240,   4.993,  472.946,   -3.5, 328.4,  16.9,   0.3,  3.4, -0.4},
{ 4,1327,1326, 1235,   0.240,   4.993,  477.938,   -3.2, 331.9,  16.5,   0.3,  3.4, -0.4},
{ 4,1328,1327, 1235,   0.240,   4.993,  482.931,   -3.0, 335.3,  16.1,   0.3,  3.4, -0.4},
{ 4,1329,1328, 1235,   0.240,   4.993,  487.924,   -2.7, 338.7,  15.7,   0.3,  3.4, -0.4},
{ 4,1330,1278, 1243,   0.240,   5.039,  250.828,   -1.9, 198.6,  26.3,   1.3,  4.3,  0.7},
{ 4,1331,1330, 1243,   0.240,   5.039,  255.867,   -0.6, 202.9,  27.0,   1.3,  4.3,  0.7},
{ 4,1332,1331, 1243,   0.240,   5.039,  260.906,    0.8, 207.2,  27.7,   1.3,  4.3,  0.7},
{ 4,1333,1332, 1243,   0.240,   5.039,  265.945,    2.1, 211.5,  28.4,   1.3,  4.3,  0.7},
{ 4,1334,1333, 1243,   0.240,   5.039,  270.984,    3.4, 215.8,  29.1,   1.3,  4.3,  0.7},
{ 4,1335,1334, 1243,   0.240,   5.039,  276.022,    4.7, 220.1,  29.8,   1.3,  4.3,  0.7},
{ 4,1336,1335, 1243,   0.240,   5.039,  281.061,    6.1, 224.4,  30.5,   1.3,  4.3,  0.7},
{ 4,1337,1336, 1243,   0.240,   5.039,  286.100,    7.4, 228.7,  31.2,   1.3,  4.3,  0.7},
{ 4,1338,1337, 1262,   0.240,   4.984,  291.111,    8.1, 233.0,  31.2,   0.0,  4.3, -0.5},
{ 4,1339,1338, 1262,   0.240,   4.984,  296.096,    8.1, 237.3,  30.7,   0.0,  4.3, -0.5},
{ 4,1340,1339, 1262,   0.240,   4.984,  301.080,    8.1, 241.7,  30.2,   0.0,  4.3, -0.5},
{ 4,1341,1340, 1262,   0.240,   4.984,  306.064,    8.1, 246.0,  29.7,   0.0,  4.3, -0.5},
{ 4,1342,1341, 1262,   0.240,   4.984,  311.048,    8.2, 250.3,  29.2,   0.0,  4.3, -0.5},
{ 4,1343,1342, 1262,   0.240,   4.984,  316.033,    8.2, 254.7,  28.7,   0.0,  4.3, -0.5},
{ 4,1344,1343, 1262,   0.240,   4.984,  321.017,    8.2, 259.0,  28.2,   0.0,  4.3, -0.5},
{ 4,1345,1344, 1262,   0.240,   4.984,  326.001,    8.2, 263.4,  27.6,   0.0,  4.3, -0.5},
{ 4,1346,1345, 1262,   0.240,   4.984,  330.986,    8.2, 267.7,  27.1,   0.0,  4.3, -0.5},
{ 4,1347,1346, 1262,   0.240,   4.984,  335.970,    8.3, 272.0,  26.6,   0.0,  4.3, -0.5},
{ 4,1348,1347, 1262,   0.240,   4.984,  340.954,    8.3, 276.4,  26.1,   0.0,  4.3, -0.5},
{ 4,1349,1348, 1262,   0.240,   4.984,  345.938,    8.3, 280.7,  25.6,   0.0,  4.3, -0.5},
{ 4,1350,1349, 1262,   0.240,   4.984,  350.923,    8.3, 285.1,  25.1,   0.0,  4.3, -0.5},
{ 4,1351,1350, 1262,   0.240,   4.984,  355.907,    8.4, 289.4,  24.6,   0.0,  4.3, -0.5},
{ 4,1352,1351, 1262,   0.240,   4.984,  360.891,    8.4, 293.7,  24.0,   0.0,  4.3, -0.5},
{ 4,1353,1352, 1262,   0.240,   4.984,  365.876,    8.4, 298.1,  23.5,   0.0,  4.3, -0.5},
{ 4,1354,1353, 1262,   0.240,   4.984,  370.860,    8.4, 302.4,  23.0,   0.0,  4.3, -0.5},
{ 4,1355,1354, 1262,   0.240,   4.984,  375.844,    8.5, 306.7,  22.5,   0.0,  4.3, -0.5},
{ 4,1356,1337, 1272,   0.240,   5.008,  291.123,    7.9, 232.0,  31.7,  -0.4,  2.3,  0.3},
{ 4,1357,1356, 1272,   0.240,   5.008,  296.131,    7.5, 234.3,  32.0,  -0.4,  2.3,  0.3},
{ 4,1358,1357, 1272,   0.240,   5.008,  301.139,    7.1, 236.7,  32.4,  -0.4,  2.3,  0.3},
{ 4,1359,1358, 1272,   0.240,   5.008,  306.147,    6.8, 239.0,  32.7,  -0.4,  2.3,  0.3},
{ 4,1360,1359, 1272,   0.240,   5.008,  311.155,    6.4, 241.3,  33.0,  -0.4,  2.3,  0.3},
{ 4,1361,1360, 1272,   0.240,   5.008,  316.163,    6.1, 243.7,  33.4,  -0.4,  2.3,  0.3},
{ 4,1362,1361, 1272,   0.240,   5.008,  321.171,    5.7, 246.0,  33.7,  -0.4,  2.3,  0.3},
{ 4,1363,1362, 1272,   0.240,   5.008,  326.180,    5.3, 248.3,  34.1,  -0.4,  2.3,  0.3},
{ 4,1364,1363, 1272,   0.240,   5.008,  331.188,    5.0, 250.7,  34.4,  -0.4,  2.3,  0.3},
{ 4,1365,1364, 1272,   0.240,   5.008,  336.196,    4.6, 253.0,  34.8,  -0.4,  2.3,  0.3},
{ 4,1366,1365, 1272,   0.240,   5.008,  341.204,    4.2, 255.3,  35.1,  -0.4,  2.3,  0.3},
{ 4,1367,1366, 1272,   0.240,   5.008,  346.212,    3.9, 257.7,  35.5,  -0.4,  2.3,  0.3},
{ 4,1368,1367, 1272,   0.240,   5.008,  351.220,    3.5, 260.0,  35.8,  -0.4,  2.3,  0.3},
{ 4,1369,1368, 1272,   0.240,   5.008,  356.228,    3.2, 262.4,  36.1,  -0.4,  2.3,  0.3},
{ 4,1370,1369, 1272,   0.240,   5.008,  361.236,    2.8, 264.7,  36.5,  -0.4,  2.3,  0.3},
{ 4,1371,1370, 1272,   0.240,   5.008,  366.244,    2.4, 267.0,  36.8,  -0.4,  2.3,  0.3},
{ 4,1372,1371, 1272,   0.240,   5.008,  371.252,    2.1, 269.4,  37.2,  -0.4,  2.3,  0.3},
{ 4,1373,1372, 1272,   0.240,   5.008,  376.260,    1.7, 271.7,  37.5,  -0.4,  2.3,  0.3},
{ 4,1374,1373, 1272,   0.240,   5.008,  381.268,    1.4, 274.0,  37.9,  -0.4,  2.3,  0.3},
{ 4,1375,1374, 1272,   0.240,   5.008,  386.276,    1.0, 276.4,  38.2,  -0.4,  2.3,  0.3},
{ 4,1376,1375, 1272,   0.240,   5.008,  391.284,    0.6, 278.7,  38.5,  -0.4,  2.3,  0.3},
{ 4,1377,1376, 1272,   0.240,   5.008,  396.292,    0.3, 281.0,  38.9,  -0.4,  2.3,  0.3},
{ 4,1378,1377, 1272,   0.240,   5.008,  401.300,   -0.1, 283.4,  39.2,  -0.4,  2.3,  0.3},
{ 4,1379,1378, 1272,   0.240,   5.008,  406.308,   -0.5, 285.7,  39.6,  -0.4,  2.3,  0.3},
{ 4,1380,1252, 1283,   0.400,   5.022,  127.485,  -20.0, 133.1,  22.0,   0.6,  4.3, -0.6},
{ 4,1381,1380, 1283,   0.400,   5.022,  132.507,  -19.4, 137.4,  21.4,   0.6,  4.3, -0.6},
{ 4,1382,1381, 1283,   0.400,   5.022,  137.529,  -18.8, 141.7,  20.8,   0.6,  4.3, -0.6},
{ 4,1383,1382, 1283,   0.400,   5.022,  142.550,  -18.2, 146.0,  20.3,   0.6,  4.3, -0.6},
{ 4,1384,1383, 1283,   0.400,   5.022,  147.572,  -17.6, 150.3,  19.7,   0.6,  4.3, -0.6},
{ 4,1385,1384, 1283,   0.400,   5.022,  152.594,  -17.0, 154.6,  19.2,   0.6,  4.3, -0.6},
{ 4,1386,1385, 1283,   0.400,   5.022,  157.616,  -16.5, 158.9,  18.6,   0.6,  4.3, -0.6},
{ 4,1387,1386, 1283,   0.400,   5.022,  162.637,  -15.9, 163.2,  18.0,   0.6,  4.3, -0.6},
{ 4,1388,1387, 1283,   0.400,   5.022,  167.659,  -15.3, 167.5,  17.5,   0.6,  4.3, -0.6},
{ 4,1389,1388, 1283,   0.400,   5.022,  172.681,  -14.7, 171.8,  16.9,   0.6,  4.3, -0.6},
{ 4,1390,1389, 1283,   0.400,   5.022,  177.702,  -14.1, 176.1,  16.3,   0.6,  4.3, -0.6},
{ 4,1391,1390, 1283,   0.400,   5.022,  182.724,  -13.5, 180.4,  15.8,   0.6,  4.3, -0.6},
{ 4,1392,1391, 1283,   0.400,   5.022,  187.746,  -12.9, 184.7,  15.2,   0.6,  4.3, -0.6},
{ 4,1393,1392, 1283,   0.400,   5.022,  192.767,  -12.3, 189.0,  14.7,   0.6,  4.3, -0.6},
{ 4,1394,1393, 1283,   0.400,   5.022,  197.789,  -11.7, 193.3,  14.1,   0.6,  4.3, -0.6},
{ 4,1395,1394, 1283,   0.400,   5.022,  202.811,  -11.1, 197.6,  13.5,   0.6,  4.3, -0.6},
{ 4,1396,1395, 1294,   0.400,   4.742,  207.693,  -11.2, 200.9,  13.9,  -0.8,  2.4,  1.3},
{ 4,1397,1396, 1294,   0.400,   4.742,  212.434,  -12.0, 203.4,  15.2,  -0.8,  2.4,  1.3},
{ 4,1398,1397, 1294,   0.400,   4.742,  217.176,  -12.7, 205.8,  16.5,  -0.8,  2.4,  1.3},
{ 4,1399,1398, 1294,   0.400,   4.742,  221.918,  -13.5, 208.3,  17.8,  -0.8,  2.4,  1.3},
{ 4,1400,1399, 1294,   0.400,   4.742,  226.660,  -14.3, 210.7,  19.1,  -0.8,  2.4,  1.3},
{ 4,1401,1400, 1294,   0.400,   4.742,  231.402,  -15.0, 213.1,  20.4,  -0.8,  2.4,  1.3},
{ 4,1402,1401, 1294,   0.400,   4.742,  236.143,  -15.8, 215.6,  21.7,  -0.8,  2.4,  1.3},
{ 4,1403,1402, 1294,   0.400,   4.742,  240.885,  -16.6, 218.0,  23.0,  -0.8,  2.4,  1.3},
{ 4,1404,1403, 1294,   0.400,   4.742,  245.627,  -17.4, 220.4,  24.3,  -0.8,  2.4,  1.3},
{ 4,1405,1404, 1294,   0.400,   4.742,  250.369,  -18.1, 222.9,  25.6,  -0.8,  2.4,  1.3},
{ 4,1406,1405, 1294,   0.400,   4.742,  255.111,  -18.9, 225.3,  26.9,  -0.8,  2.4,  1.3},
{ 4,1407,1406, 1294,   0.400,   4.742,  259.853,  -19.7, 227.8,  28.2,  -0.8,  2.4,  1.3},
{ 4,1408,1407, 1294,   0.400,   4.742,  264.594,  -20.4, 230.2,  29.5,  -0.8,  2.4,  1.3},
{ 4,1409,1408, 1294,   0.400,   4.742,  269.336,  -21.2, 232.6,  30.8,  -0.8,  2.4,  1.3},
{ 4,1410,1409, 1297,   0.533,   3.929,  273.672,  -19.7, 234.2,  31.4,   3.8,  0.6, -0.2},
{ 4,1411,1410, 1297,   0.533,   3.929,  277.600,  -15.9, 234.8,  31.2,   3.8,  0.6, -0.2},
{ 4,1412,1411, 1297,   0.533,   3.929,  281.529,  -12.1, 235.5,  31.0,   3.8,  0.6, -0.2},
{ 4,1413,1412, 1297,   0.533,   3.929,  285.458,   -8.2, 236.1,  30.8,   3.8,  0.6, -0.2},
{ 4,1414,1413, 1300,   0.526,   5.006,  289.926,   -6.1, 238.2,  31.8,   0.5,  3.6,  2.1},
{ 4,1415,1414, 1300,   0.526,   5.006,  294.932,   -5.6, 241.8,  33.9,   0.5,  3.6,  2.1},
{ 4,1416,1415, 1300,   0.526,   5.006,  299.938,   -5.2, 245.3,  36.1,   0.5,  3.6,  2.1},
{ 4,1417,1416, 1300,   0.526,   5.006,  304.944,   -4.7, 248.9,  38.2,   0.5,  3.6,  2.1},
{ 4,1418,1417, 1305,   0.560,   4.506,  309.699,   -4.5, 252.1,  38.7,  -0.1,  2.8, -1.2},
{ 4,1419,1418, 1305,   0.560,   4.506,  314.205,   -4.6, 254.9,  37.5,  -0.1,  2.8, -1.2},
{ 4,1420,1419, 1305,   0.560,   4.506,  318.711,   -4.7, 257.7,  36.3,  -0.1,  2.8, -1.2},
{ 4,1421,1420, 1305,   0.560,   4.506,  323.217,   -4.9, 260.5,  35.1,  -0.1,  2.8, -1.2},
{ 4,1422,1421, 1305,   0.560,   4.506,  327.722,   -5.0, 263.3,  33.9,  -0.1,  2.8, -1.2},
{ 4,1423,1422, 1305,   0.560,   4.506,  332.228,   -5.1, 266.1,  32.7,  -0.1,  2.8, -1.2},
{ 4,1424,1423, 1305,   0.560,   4.506,  336.734,   -5.2, 268.9,  31.5,  -0.1,  2.8, -1.2},
{ 4,1425,1424, 1305,   0.560,   4.506,  341.240,   -5.3, 271.8,  30.3,  -0.1,  2.8, -1.2},
{ 4,1426,1425, 1305,   0.560,   4.506,  345.745,   -5.4, 274.6,  29.1,  -0.1,  2.8, -1.2},
{ 4,1427,1426, 1313,   0.407,   4.677,  350.337,   -5.6, 277.9,  28.9,  -0.3,  3.8,  0.8},
{ 4,1428,1427, 1313,   0.407,   4.677,  355.014,   -5.8, 281.6,  29.7,  -0.3,  3.8,  0.8},
{ 4,1429,1428, 1313,   0.407,   4.677,  359.692,   -6.1, 285.4,  30.5,  -0.3,  3.8,  0.8},
{ 4,1430,1429, 1313,   0.407,   4.677,  364.369,   -6.4, 289.2,  31.3,  -0.3,  3.8,  0.8},
{ 4,1431,1430, 1313,   0.407,   4.677,  369.046,   -6.6, 293.0,  32.1,  -0.3,  3.8,  0.8},
{ 4,1432,1431, 1313,   0.407,   4.677,  373.724,   -6.9, 296.8,  32.9,  -0.3,  3.8,  0.8},
{ 4,1433,1432, 1313,   0.407,   4.677,  378.401,   -7.1, 300.5,  33.8,  -0.3,  3.8,  0.8},
{ 4,1434,1433, 1313,   0.407,   4.677,  383.079,   -7.4, 304.3,  34.6,  -0.3,  3.8,  0.8},
{ 4,1435,1434, 1313,   0.407,   4.677,  387.756,   -7.6, 308.1,  35.4,  -0.3,  3.8,  0.8},
{ 4,1436,1435, 1313,   0.407,   4.677,  392.434,   -7.9, 311.9,  36.2,  -0.3,  3.8,  0.8},
{ 4,1437,1436, 1313,   0.407,   4.677,  397.111,   -8.1, 315.7,  37.0,  -0.3,  3.8,  0.8},
{ 4,1438,1437, 1313,   0.407,   4.677,  401.788,   -8.4, 319.4,  37.8,  -0.3,  3.8,  0.8},
{ 4,1439,1438, 1313,   0.407,   4.677,  406.466,   -8.7, 323.2,  38.6,  -0.3,  3.8,  0.8},
{ 4,1440,1439, 1327,   0.400,   4.601,  411.105,   -8.3, 326.0,  39.5,   0.9,  1.8,  1.0},
{ 4,1441,1440, 1327,   0.400,   4.601,  415.706,   -7.4, 327.8,  40.5,   0.9,  1.8,  1.0},
{ 4,1442,1441, 1327,   0.400,   4.601,  420.308,   -6.5, 329.6,  41.4,   0.9,  1.8,  1.0},
{ 4,1443,1442, 1327,   0.400,   4.601,  424.909,   -5.6, 331.4,  42.4,   0.9,  1.8,  1.0},
{ 4,1444,1443, 1327,   0.400,   4.601,  429.510,   -4.7, 333.2,  43.4,   0.9,  1.8,  1.0},
{ 4,1445,1444, 1327,   0.400,   4.601,  434.112,   -3.8, 335.1,  44.4,   0.9,  1.8,  1.0},
{ 4,1446,1445, 1327,   0.400,   4.601,  438.713,   -2.9, 336.9,  45.3,   0.9,  1.8,  1.0},
{ 4,1447,1446, 1327,   0.400,   4.601,  443.314,   -2.0, 338.7,  46.3,   0.9,  1.8,  1.0},
{ 4,1448,1447, 1327,   0.400,   4.601,  447.916,   -1.0, 340.5,  47.3,   0.9,  1.8,  1.0},
{ 4,1449,1448, 1327,   0.400,   4.601,  452.517,   -0.1, 342.3,  48.3,   0.9,  1.8,  1.0},
{ 4,1450,1449, 1379,   0.244,   4.935,  457.285,    0.5, 344.7,  49.1,   0.4,  3.0,  0.7},
{ 4,1451,1450, 1379,   0.244,   4.935,  462.221,    1.0, 347.7,  49.8,   0.4,  3.0,  0.7},
{ 4,1452,1451, 1379,   0.244,   4.935,  467.156,    1.4, 350.7,  50.5,   0.4,  3.0,  0.7},
{ 4,1453,1452, 1379,   0.244,   4.935,  472.092,    1.9, 353.8,  51.3,   0.4,  3.0,  0.7},
{ 4,1454,1453, 1379,   0.244,   4.935,  477.027,    2.3, 356.8,  52.0,   0.4,  3.0,  0.7},
{ 4,1455,1454, 1379,   0.244,   4.935,  481.963,    2.8, 359.8,  52.7,   0.4,  3.0,  0.7},
{ 4,1456,1455, 1379,   0.244,   4.935,  486.898,    3.2, 362.8,  53.4,   0.4,  3.0,  0.7},
{ 4,1457,1456, 1379,   0.244,   4.935,  491.834,    3.7, 365.8,  54.1,   0.4,  3.0,  0.7},
{ 4,1458,1457, 1379,   0.244,   4.935,  496.769,    4.1, 368.9,  54.8,   0.4,  3.0,  0.7},
{ 4,1459,1458, 1379,   0.244,   4.935,  501.705,    4.6, 371.9,  55.6,   0.4,  3.0,  0.7},
{ 4,1460,1459, 1379,   0.244,   4.935,  506.640,    5.0, 374.9,  56.3,   0.4,  3.0,  0.7},
{ 4,1461,1460, 1379,   0.244,   4.935,  511.576,    5.4, 377.9,  57.0,   0.4,  3.0,  0.7},
{ 4,1462,1461, 1379,   0.244,   4.935,  516.511,    5.9, 380.9,  57.7,   0.4,  3.0,  0.7},
{ 4,1463,1462, 1379,   0.244,   4.935,  521.447,    6.3, 384.0,  58.4,   0.4,  3.0,  0.7},
{ 4,1464,1463, 1379,   0.244,   4.935,  526.382,    6.8, 387.0,  59.1,   0.4,  3.0,  0.7},
{ 4,1465,1464, 1379,   0.244,   4.935,  531.317,    7.2, 390.0,  59.8,   0.4,  3.0,  0.7},
{ 4,1466,1465, 1379,   0.244,   4.935,  536.253,    7.7, 393.0,  60.6,   0.4,  3.0,  0.7},
{ 4,1467,1466, 1379,   0.244,   4.935,  541.188,    8.1, 396.0,  61.3,   0.4,  3.0,  0.7},
{ 4,1468,1467, 1379,   0.244,   4.935,  546.124,    8.6, 399.1,  62.0,   0.4,  3.0,  0.7},
{ 4,1469,1468, 1379,   0.244,   4.935,  551.059,    9.0, 402.1,  62.7,   0.4,  3.0,  0.7},
{ 4,1470,1469, 1379,   0.244,   4.935,  555.995,    9.5, 405.1,  63.4,   0.4,  3.0,  0.7},
{ 4,1471,1470, 1379,   0.244,   4.935,  560.930,    9.9, 408.1,  64.1,   0.4,  3.0,  0.7},
{ 4,1472,1471, 1379,   0.244,   4.935,  565.866,   10.3, 411.1,  64.9,   0.4,  3.0,  0.7},
{ 4,1473,1472, 1379,   0.244,   4.935,  570.801,   10.8, 414.1,  65.6,   0.4,  3.0,  0.7},
{ 4,1474,1473, 1379,   0.244,   4.935,  575.737,   11.2, 417.2,  66.3,   0.4,  3.0,  0.7},
{ 4,1475,1474, 1379,   0.244,   4.935,  580.672,   11.7, 420.2,  67.0,   0.4,  3.0,  0.7},
{ 4,1476,1475, 1379,   0.244,   4.935,  585.608,   12.1, 423.2,  67.7,   0.4,  3.0,  0.7},
{ 4,1477,1476, 1379,   0.244,   4.935,  590.543,   12.6, 426.2,  68.4,   0.4,  3.0,  0.7},
{ 4,1478,1477, 1379,   0.244,   4.935,  595.479,   13.0, 429.2,  69.2,   0.4,  3.0,  0.7},
{ 4,1479,1478, 1379,   0.244,   4.935,  600.414,   13.5, 432.3,  69.9,   0.4,  3.0,  0.7},
{ 4,1480,1479, 1379,   0.244,   4.935,  605.350,   13.9, 435.3,  70.6,   0.4,  3.0,  0.7},
{ 4,1481,1480, 1379,   0.244,   4.935,  610.285,   14.4, 438.3,  71.3,   0.4,  3.0,  0.7},
{ 4,1482,1481, 1379,   0.244,   4.935,  615.221,   14.8, 441.3,  72.0,   0.4,  3.0,  0.7},
{ 4,1483,1482, 1379,   0.244,   4.935,  620.156,   15.2, 444.3,  72.7,   0.4,  3.0,  0.7},
{ 4,1484,1483, 1379,   0.244,   4.935,  625.092,   15.7, 447.4,  73.4,   0.4,  3.0,  0.7},
{ 4,1485,1484, 1379,   0.244,   4.935,  630.027,   16.1, 450.4,  74.2,   0.4,  3.0,  0.7},
{ 4,1486,1485, 1379,   0.244,   4.935,  634.962,   16.6, 453.4,  74.9,   0.4,  3.0,  0.7},
{ 4,1487,1486, 1379,   0.244,   4.935,  639.898,   17.0, 456.4,  75.6,   0.4,  3.0,  0.7},
{ 4,1488,1487, 1379,   0.244,   4.935,  644.833,   17.5, 459.4,  76.3,   0.4,  3.0,  0.7},
{ 4,1489,1488, 1379,   0.244,   4.935,  649.769,   17.9, 462.5,  77.0,   0.4,  3.0,  0.7},
{ 4,1490,1489, 1379,   0.244,   4.935,  654.704,   18.4, 465.5,  77.7,   0.4,  3.0,  0.7},
{ 4,1491,1490, 1379,   0.244,   4.935,  659.640,   18.8, 468.5,  78.5,   0.4,  3.0,  0.7},
{ 4,1492,1491, 1379,   0.244,   4.935,  664.575,   19.3, 471.5,  79.2,   0.4,  3.0,  0.7},
{ 4,1493,1492, 1379,   0.244,   4.935,  669.511,   19.7, 474.5,  79.9,   0.4,  3.0,  0.7},
{ 4,1494,1439, 1390,   0.400,   4.826,  411.217,   -8.9, 326.7,  40.0,  -0.3,  3.3,  2.1},
{ 4,1495,1494, 1390,   0.400,   4.826,  416.043,   -9.2, 330.0,  42.1,  -0.3,  3.3,  2.1},
{ 4,1496,1495, 1390,   0.400,   4.826,  420.869,   -9.5, 333.2,  44.2,  -0.3,  3.3,  2.1},
{ 4,1497,1496, 1390,   0.400,   4.826,  425.694,   -9.8, 336.5,  46.2,  -0.3,  3.3,  2.1},
{ 4,1498,1497, 1390,   0.400,   4.826,  430.520,  -10.0, 339.7,  48.3,  -0.3,  3.3,  2.1},
{ 4,1499,1498, 1390,   0.400,   4.826,  435.346,  -10.3, 343.0,  50.3,  -0.3,  3.3,  2.1},
{ 4,1500,1499, 1390,   0.400,   4.826,  440.172,  -10.6, 346.3,  52.4,  -0.3,  3.3,  2.1},
{ 4,1501,1500, 1390,   0.400,   4.826,  444.997,  -10.9, 349.5,  54.5,  -0.3,  3.3,  2.1},
{ 4,1502,1501, 1390,   0.400,   4.826,  449.823,  -11.1, 352.8,  56.5,  -0.3,  3.3,  2.1},
{ 4,1503,1502, 1390,   0.400,   4.826,  454.649,  -11.4, 356.0,  58.6,  -0.3,  3.3,  2.1},
{ 4,1504,1503, 1390,   0.400,   4.826,  459.474,  -11.7, 359.3,  60.7,  -0.3,  3.3,  2.1},
{ 4,1505,1504, 1390,   0.400,   4.826,  464.300,  -12.0, 362.5,  62.7,  -0.3,  3.3,  2.1},
{ 4,1506,1505, 1415,   0.244,   4.929,  469.177,  -12.8, 365.7,  64.4,  -1.3,  3.1,  1.3},
{ 4,1507,1506, 1415,   0.244,   4.929,  474.106,  -14.1, 368.7,  65.6,  -1.3,  3.1,  1.3},
{ 4,1508,1507, 1415,   0.244,   4.929,  479.035,  -15.5, 371.8,  66.9,  -1.3,  3.1,  1.3},
{ 4,1509,1508, 1415,   0.244,   4.929,  483.964,  -16.8, 374.8,  68.1,  -1.3,  3.1,  1.3},
{ 4,1510,1509, 1415,   0.244,   4.929,  488.893,  -18.2, 377.9,  69.4,  -1.3,  3.1,  1.3},
{ 4,1511,1510, 1415,   0.244,   4.929,  493.822,  -19.5, 381.0,  70.6,  -1.3,  3.1,  1.3},
{ 4,1512,1511, 1415,   0.244,   4.929,  498.751,  -20.8, 384.0,  71.9,  -1.3,  3.1,  1.3},
{ 4,1513,1512, 1415,   0.244,   4.929,  503.680,  -22.2, 387.1,  73.1,  -1.3,  3.1,  1.3},
{ 4,1514,1513, 1415,   0.244,   4.929,  508.609,  -23.5, 390.1,  74.4,  -1.3,  3.1,  1.3},
{ 4,1515,1514, 1415,   0.244,   4.929,  513.538,  -24.9, 393.2,  75.6,  -1.3,  3.1,  1.3},
{ 4,1516,1515, 1415,   0.244,   4.929,  518.467,  -26.2, 396.2,  76.9,  -1.3,  3.1,  1.3},
{ 4,1517,1516, 1415,   0.244,   4.929,  523.396,  -27.5, 399.3,  78.1,  -1.3,  3.1,  1.3},
{ 4,1518,1517, 1415,   0.244,   4.929,  528.325,  -28.9, 402.4,  79.4,  -1.3,  3.1,  1.3},
{ 4,1519,1518, 1415,   0.244,   4.929,  533.254,  -30.2, 405.4,  80.6,  -1.3,  3.1,  1.3},
{ 4,1520,1519, 1415,   0.244,   4.929,  538.183,  -31.6, 408.5,  81.9,  -1.3,  3.1,  1.3},
{ 4,1521,1520, 1415,   0.244,   4.929,  543.112,  -32.9, 411.5,  83.1,  -1.3,  3.1,  1.3},
{ 4,1522,1521, 1415,   0.244,   4.929,  548.041,  -34.2, 414.6,  84.4,  -1.3,  3.1,  1.3},
{ 4,1523,1522, 1415,   0.244,   4.929,  552.970,  -35.6, 417.6,  85.6,  -1.3,  3.1,  1.3},
{ 4,1524,1523, 1415,   0.244,   4.929,  557.899,  -36.9, 420.7,  86.9,  -1.3,  3.1,  1.3},
{ 4,1525,1524, 1415,   0.244,   4.929,  562.827,  -38.3, 423.7,  88.1,  -1.3,  3.1,  1.3},
{ 4,1526,1525, 1415,   0.244,   4.929,  567.756,  -39.6, 426.8,  89.4,  -1.3,  3.1,  1.3},
{ 4,1527,1526, 1415,   0.244,   4.929,  572.685,  -41.0, 429.9,  90.6,  -1.3,  3.1,  1.3},
{ 4,1528,1527, 1415,   0.244,   4.929,  577.614,  -42.3, 432.9,  91.9,  -1.3,  3.1,  1.3},
{ 4,1529,1528, 1415,   0.244,   4.929,  582.543,  -43.6, 436.0,  93.1,  -1.3,  3.1,  1.3},
{ 4,1530,1529, 1415,   0.244,   4.929,  587.472,  -45.0, 439.0,  94.4,  -1.3,  3.1,  1.3},
{ 4,1531,1530, 1415,   0.244,   4.929,  592.401,  -46.3, 442.1,  95.6,  -1.3,  3.1,  1.3},
{ 4,1532,1531, 1415,   0.244,   4.929,  597.330,  -47.7, 445.1,  96.9,  -1.3,  3.1,  1.3},
{ 4,1533,1532, 1415,   0.244,   4.929,  602.259,  -49.0, 448.2,  98.1,  -1.3,  3.1,  1.3},
{ 4,1534,1533, 1415,   0.244,   4.929,  607.188,  -50.3, 451.2,  99.4,  -1.3,  3.1,  1.3},
{ 4,1535,1534, 1415,   0.244,   4.929,  612.117,  -51.7, 454.3, 100.6,  -1.3,  3.1,  1.3},
{ 4,1536,1535, 1415,   0.244,   4.929,  617.046,  -53.0, 457.4, 101.9,  -1.3,  3.1,  1.3},
{ 4,1537,1536, 1415,   0.244,   4.929,  621.975,  -54.4, 460.4, 103.1,  -1.3,  3.1,  1.3},
{ 4,1538,1537, 1415,   0.244,   4.929,  626.904,  -55.7, 463.5, 104.4,  -1.3,  3.1,  1.3},
{ 4,1539,1395, 1451,   0.240,   4.998,  207.821,  -10.7, 201.6,  13.3,   0.2,  3.8,  0.2},
{ 4,1540,1539, 1451,   0.240,   4.998,  212.819,  -10.5, 205.3,  13.5,   0.2,  3.8,  0.2},
{ 4,1541,1540, 1451,   0.240,   4.998,  217.817,  -10.2, 209.1,  13.7,   0.2,  3.8,  0.2},
{ 4,1542,1541, 1451,   0.240,   4.998,  222.815,  -10.0, 212.9,  13.9,   0.2,  3.8,  0.2},
{ 4,1543,1542, 1451,   0.240,   4.998,  227.813,   -9.8, 216.6,  14.0,   0.2,  3.8,  0.2},
{ 4,1544,1543, 1451,   0.240,   4.998,  232.812,   -9.6, 220.4,  14.2,   0.2,  3.8,  0.2},
{ 4,1545,1544, 1451,   0.240,   4.998,  237.810,   -9.4, 224.1,  14.4,   0.2,  3.8,  0.2},
{ 4,1546,1545, 1451,   0.240,   4.998,  242.808,   -9.1, 227.9,  14.5,   0.2,  3.8,  0.2},
{ 4,1547,1546, 1451,   0.240,   4.998,  247.806,   -8.9, 231.6,  14.7,   0.2,  3.8,  0.2},
{ 4,1548,1547, 1451,   0.240,   4.998,  252.804,   -8.7, 235.4,  14.9,   0.2,  3.8,  0.2},
{ 4,1549,1548, 1451,   0.240,   4.998,  257.802,   -8.5, 239.1,  15.1,   0.2,  3.8,  0.2},
{ 4,1550,1549, 1451,   0.240,   4.998,  262.801,   -8.2, 242.9,  15.2,   0.2,  3.8,  0.2},
{ 4,1551,1550, 1451,   0.240,   4.998,  267.799,   -8.0, 246.6,  15.4,   0.2,  3.8,  0.2},
{ 4,1552,1551, 1451,   0.240,   4.998,  272.797,   -7.8, 250.4,  15.6,   0.2,  3.8,  0.2},
{ 4,1553,1552, 1451,   0.240,   4.998,  277.795,   -7.6, 254.1,  15.8,   0.2,  3.8,  0.2},
{ 4,1554,1553, 1451,   0.240,   4.998,  282.793,   -7.3, 257.9,  15.9,   0.2,  3.8,  0.2},
{ 4,1555,1554, 1451,   0.240,   4.998,  287.791,   -7.1, 261.7,  16.1,   0.2,  3.8,  0.2},
{ 4,1556,1555, 1451,   0.240,   4.998,  292.790,   -6.9, 265.4,  16.3,   0.2,  3.8,  0.2},
{ 4,1557,1556, 1451,   0.240,   4.998,  297.788,   -6.7, 269.2,  16.5,   0.2,  3.8,  0.2},
{ 4,1558,1557, 1451,   0.240,   4.998,  302.786,   -6.5, 272.9,  16.6,   0.2,  3.8,  0.2},
{ 4,1559,1558, 1451,   0.240,   4.998,  307.784,   -6.2, 276.7,  16.8,   0.2,  3.8,  0.2},
{ 4,1560,1559, 1451,   0.240,   4.998,  312.782,   -6.0, 280.4,  17.0,   0.2,  3.8,  0.2},
{ 4,1561,1560, 1451,   0.240,   4.998,  317.780,   -5.8, 284.2,  17.1,   0.2,  3.8,  0.2},
{ 4,1562,1561, 1451,   0.240,   4.998,  322.779,   -5.6, 287.9,  17.3,   0.2,  3.8,  0.2},
{ 4,1563,1562, 1451,   0.240,   4.998,  327.777,   -5.3, 291.7,  17.5,   0.2,  3.8,  0.2},
{ 4,1564,1563, 1451,   0.240,   4.998,  332.775,   -5.1, 295.4,  17.7,   0.2,  3.8,  0.2},
{ 4,1565,1564, 1451,   0.240,   4.998,  337.773,   -4.9, 299.2,  17.8,   0.2,  3.8,  0.2},
{ 4,1566,1565, 1451,   0.240,   4.998,  342.771,   -4.7, 302.9,  18.0,   0.2,  3.8,  0.2},
{ 4,1567,1566, 1451,   0.240,   4.998,  347.769,   -4.5, 306.7,  18.2,   0.2,  3.8,  0.2},
{ 4,1568,1567, 1451,   0.240,   4.998,  352.768,   -4.2, 310.5,  18.4,   0.2,  3.8,  0.2},
{ 4,1569,1568, 1451,   0.240,   4.998,  357.766,   -4.0, 314.2,  18.5,   0.2,  3.8,  0.2},
{ 4,1570,1569, 1451,   0.240,   4.998,  362.764,   -3.8, 318.0,  18.7,   0.2,  3.8,  0.2},
{ 4,1571,1570, 1451,   0.240,   4.998,  367.762,   -3.6, 321.7,  18.9,   0.2,  3.8,  0.2},
{ 4,1572,1571, 1451,   0.240,   4.998,  372.760,   -3.3, 325.5,  19.0,   0.2,  3.8,  0.2},
{ 4,1573,1572, 1451,   0.240,   4.998,  377.758,   -3.1, 329.2,  19.2,   0.2,  3.8,  0.2},
{ 4,1574,1573, 1451,   0.240,   4.998,  382.757,   -2.9, 333.0,  19.4,   0.2,  3.8,  0.2},
{ 4,1575,1574, 1451,   0.240,   4.998,  387.755,   -2.7, 336.7,  19.6,   0.2,  3.8,  0.2},
{ 4,1576,1575, 1451,   0.240,   4.998,  392.753,   -2.4, 340.5,  19.7,   0.2,  3.8,  0.2},
{ 4,1577,1576, 1451,   0.240,   4.998,  397.751,   -2.2, 344.2,  19.9,   0.2,  3.8,  0.2},
{ 4,1578, 573, 1455,   1.621,   4.229,   15.120,  -12.7,  48.5,  15.1,   0.5,  3.4,  0.8},
{ 4,1579,1578, 1455,   1.621,   4.229,   19.348,  -12.2,  51.9,  15.9,   0.5,  3.4,  0.8},
{ 4,1580,1579, 1455,   1.621,   4.229,   23.577,  -11.6,  55.4,  16.8,   0.5,  3.4,  0.8},
{ 4,1581,1580, 1455,   1.621,   4.229,   27.806,  -11.1,  58.8,  17.5,   0.5,  3.4,  0.8},
{ 4,1582,1581, 1455,   1.621,   4.229,   32.035,  -10.6,  62.2,  18.3,   0.5,  3.4,  0.8},
{ 4,1583,1582, 1467,   1.195,   4.900,   36.599,  -10.2,  66.0,  18.1,   0.4,  4.2, -1.3},
{ 4,1584,1583, 1467,   1.195,   4.900,   41.499,   -9.7,  70.2,  16.8,   0.4,  4.2, -1.3},
{ 4,1585,1584, 1467,   1.195,   4.900,   46.398,   -9.3,  74.4,  15.5,   0.4,  4.2, -1.3},
{ 4,1586,1585, 1467,   1.195,   4.900,   51.298,   -8.8,  78.5,  14.3,   0.4,  4.2, -1.3},
{ 4,1587,1586, 1467,   1.195,   4.900,   56.198,   -8.4,  82.7,  13.0,   0.4,  4.2, -1.3},
{ 4,1588,1587, 1467,   1.195,   4.900,   61.098,   -8.0,  86.9,  11.7,   0.4,  4.2, -1.3},
{ 4,1589,1588, 1467,   1.195,   4.900,   65.997,   -7.5,  91.0,  10.4,   0.4,  4.2, -1.3},
{ 4,1590,1589, 1467,   1.195,   4.900,   70.897,   -7.1,  95.2,   9.1,   0.4,  4.2, -1.3},
{ 4,1591,1590, 1467,   1.195,   4.900,   75.797,   -6.6,  99.4,   7.8,   0.4,  4.2, -1.3},
{ 4,1592,1591, 1467,   1.195,   4.900,   80.697,   -6.2, 103.5,   6.5,   0.4,  4.2, -1.3},
{ 4,1593,1592, 1467,   1.195,   4.900,   85.596,   -5.8, 107.7,   5.3,   0.4,  4.2, -1.3},
{ 4,1594,1593, 1467,   1.195,   4.900,   90.496,   -5.3, 111.9,   4.0,   0.4,  4.2, -1.3},
{ 4,1595,1594, 1467,   1.195,   4.900,   95.396,   -4.9, 116.0,   2.7,   0.4,  4.2, -1.3},
{ 4,1596,1595, 1467,   1.195,   4.900,  100.296,   -4.4, 120.2,   1.4,   0.4,  4.2, -1.3},
{ 4,1597,1596, 1470,   0.400,   3.806,  104.649,   -3.0, 122.9,   1.9,   2.5,  1.2,  2.2},
{ 4,1598,1597, 1470,   0.400,   3.806,  108.455,   -0.5, 124.1,   4.1,   2.5,  1.2,  2.2},
{ 4,1599,1598, 1470,   0.400,   3.806,  112.261,    1.9, 125.3,   6.3,   2.5,  1.2,  2.2},
{ 4,1600,1599, 1487,   0.400,   4.891,  116.610,    4.2, 126.2,   7.2,   2.0,  0.6, -0.5},
{ 4,1601,1600, 1487,   0.400,   4.891,  121.501,    6.2, 126.8,   6.8,   2.0,  0.6, -0.5},
{ 4,1602,1601, 1487,   0.400,   4.891,  126.392,    8.2, 127.3,   6.3,   2.0,  0.6, -0.5},
{ 4,1603,1602, 1487,   0.400,   4.891,  131.284,   10.2, 127.9,   5.8,   2.0,  0.6, -0.5},
{ 4,1604,1603, 1487,   0.400,   4.891,  136.175,   12.2, 128.5,   5.4,   2.0,  0.6, -0.5},
{ 4,1605,1604, 1487,   0.400,   4.891,  141.066,   14.2, 129.0,   4.9,   2.0,  0.6, -0.5},
{ 4,1606,1605, 1487,   0.400,   4.891,  145.957,   16.2, 129.6,   4.5,   2.0,  0.6, -0.5},
{ 4,1607,1606, 1487,   0.400,   4.891,  150.848,   18.2, 130.1,   4.0,   2.0,  0.6, -0.5},
{ 4,1608,1607, 1487,   0.400,   4.891,  155.739,   20.2, 130.7,   3.5,   2.0,  0.6, -0.5},
{ 4,1609,1608, 1487,   0.400,   4.891,  160.631,   22.2, 131.2,   3.1,   2.0,  0.6, -0.5},
{ 4,1610,1609, 1487,   0.400,   4.891,  165.522,   24.2, 131.8,   2.6,   2.0,  0.6, -0.5},
{ 4,1611,1610, 1487,   0.400,   4.891,  170.413,   26.2, 132.3,   2.1,   2.0,  0.6, -0.5},
{ 4,1612,1611, 1487,   0.400,   4.891,  175.304,   28.2, 132.9,   1.7,   2.0,  0.6, -0.5},
{ 4,1613,1612, 1502,   0.400,   4.964,  180.232,   30.6, 133.8,   0.6,   2.8,  1.4, -1.8},
{ 4,1614,1613, 1502,   0.400,   4.964,  185.196,   33.4, 135.2,  -1.2,   2.8,  1.4, -1.8},
{ 4,1615,1614, 1502,   0.400,   4.964,  190.160,   36.2, 136.5,  -3.0,   2.8,  1.4, -1.8},
{ 4,1616,1615, 1502,   0.400,   4.964,  195.124,   39.0, 137.9,  -4.7,   2.8,  1.4, -1.8},
{ 4,1617,1616, 1502,   0.400,   4.964,  200.089,   41.8, 139.2,  -6.5,   2.8,  1.4, -1.8},
{ 4,1618,1617, 1502,   0.400,   4.964,  205.053,   44.5, 140.6,  -8.3,   2.8,  1.4, -1.8},
{ 4,1619,1618, 1502,   0.400,   4.964,  210.017,   47.3, 142.0, -10.1,   2.8,  1.4, -1.8},
{ 4,1620,1619, 1502,   0.400,   4.964,  214.981,   50.1, 143.3, -11.8,   2.8,  1.4, -1.8},
{ 4,1621,1620, 1502,   0.400,   4.964,  219.946,   52.9, 144.7, -13.6,   2.8,  1.4, -1.8},
{ 4,1622,1621, 1502,   0.400,   4.964,  224.910,   55.7, 146.0, -15.4,   2.8,  1.4, -1.8},
{ 4,1623,1622, 1502,   0.400,   4.964,  229.874,   58.5, 147.4, -17.1,   2.8,  1.4, -1.8},
{ 4,1624,1623, 1502,   0.400,   4.964,  234.838,   61.3, 148.7, -18.9,   2.8,  1.4, -1.8},
{ 4,1625,1624, 1502,   0.400,   4.964,  239.803,   64.0, 150.1, -20.7,   2.8,  1.4, -1.8},
{ 4,1626,1625, 1515,   0.240,   4.998,  244.784,   67.6, 150.6, -21.7,   4.3, -0.4, -0.3},
{ 4,1627,1626, 1515,   0.240,   4.998,  249.781,   71.9, 150.2, -22.0,   4.3, -0.4, -0.3},
{ 4,1628,1627, 1515,   0.240,   4.998,  254.779,   76.2, 149.8, -22.3,   4.3, -0.4, -0.3},
{ 4,1629,1628, 1515,   0.240,   4.998,  259.776,   80.5, 149.4, -22.6,   4.3, -0.4, -0.3},
{ 4,1630,1629, 1515,   0.240,   4.998,  264.774,   84.8, 149.0, -22.9,   4.3, -0.4, -0.3},
{ 4,1631,1630, 1515,   0.240,   4.998,  269.771,   89.1, 148.6, -23.2,   4.3, -0.4, -0.3},
{ 4,1632,1631, 1515,   0.240,   4.998,  274.769,   93.4, 148.2, -23.5,   4.3, -0.4, -0.3},
{ 4,1633,1632, 1515,   0.240,   4.998,  279.766,   97.7, 147.8, -23.9,   4.3, -0.4, -0.3},
{ 4,1634,1633, 1515,   0.240,   4.998,  284.764,  102.0, 147.4, -24.2,   4.3, -0.4, -0.3},
{ 4,1635,1634, 1515,   0.240,   4.998,  289.761,  106.3, 146.9, -24.5,   4.3, -0.4, -0.3},
{ 4,1636,1635, 1515,   0.240,   4.998,  294.759,  110.6, 146.5, -24.8,   4.3, -0.4, -0.3},
{ 4,1637,1636, 1515,   0.240,   4.998,  299.756,  114.9, 146.1, -25.1,   4.3, -0.4, -0.3},
{ 4,1638,1637, 1515,   0.240,   4.998,  304.754,  119.2, 145.7, -25.4,   4.3, -0.4, -0.3},
{ 4,1639,1625, 1519,   0.400,   4.507,  244.538,   66.0, 151.5, -20.0,   1.1,  1.6,  3.1},
{ 4,1640,1639, 1519,   0.400,   4.507,  249.045,   67.0, 153.1, -16.9,   1.1,  1.6,  3.1},
{ 4,1641,1640, 1519,   0.400,   4.507,  253.552,   68.1, 154.7, -13.7,   1.1,  1.6,  3.1},
{ 4,1642,1641, 1519,   0.400,   4.507,  258.059,   69.1, 156.2, -10.6,   1.1,  1.6,  3.1},
{ 4,1643,1642, 1519,   0.400,   4.507,  262.566,   70.2, 157.8,  -7.5,   1.1,  1.6,  3.1},
{ 4,1644,1643, 1519,   0.400,   4.507,  267.073,   71.3, 159.3,  -4.4,   1.1,  1.6,  3.1},
{ 4,1645,1644, 1552,   0.250,   5.019,  271.836,   72.9, 161.3,  -3.5,   2.2,  2.4, -1.5},
{ 4,1646,1645, 1552,   0.250,   5.019,  276.856,   75.1, 163.7,  -5.0,   2.2,  2.4, -1.5},
{ 4,1647,1646, 1552,   0.250,   5.019,  281.875,   77.3, 166.1,  -6.5,   2.2,  2.4, -1.5},
{ 4,1648,1647, 1552,   0.250,   5.019,  286.894,   79.5, 168.4,  -8.0,   2.2,  2.4, -1.5},
{ 4,1649,1648, 1552,   0.250,   5.019,  291.913,   81.7, 170.8,  -9.5,   2.2,  2.4, -1.5},
{ 4,1650,1649, 1552,   0.250,   5.019,  296.932,   83.9, 173.2, -11.0,   2.2,  2.4, -1.5},
{ 4,1651,1650, 1552,   0.250,   5.019,  301.951,   86.1, 175.6, -12.5,   2.2,  2.4, -1.5},
{ 4,1652,1651, 1552,   0.250,   5.019,  306.970,   88.4, 177.9, -14.0,   2.2,  2.4, -1.5},
{ 4,1653,1652, 1552,   0.250,   5.019,  311.989,   90.6, 180.3, -15.5,   2.2,  2.4, -1.5},
{ 4,1654,1653, 1552,   0.250,   5.019,  317.008,   92.8, 182.7, -16.9,   2.2,  2.4, -1.5},
{ 4,1655,1654, 1552,   0.250,   5.019,  322.028,   95.0, 185.0, -18.4,   2.2,  2.4, -1.5},
{ 4,1656,1655, 1552,   0.250,   5.019,  327.047,   97.2, 187.4, -19.9,   2.2,  2.4, -1.5},
{ 4,1657,1656, 1552,   0.250,   5.019,  332.066,   99.4, 189.8, -21.4,   2.2,  2.4, -1.5},
{ 4,1658,1657, 1552,   0.250,   5.019,  337.085,  101.6, 192.2, -22.9,   2.2,  2.4, -1.5},
{ 4,1659,1658, 1552,   0.250,   5.019,  342.104,  103.8, 194.5, -24.4,   2.2,  2.4, -1.5},
{ 4,1660,1659, 1552,   0.250,   5.019,  347.123,  106.0, 196.9, -25.9,   2.2,  2.4, -1.5},
{ 4,1661,1660, 1552,   0.250,   5.019,  352.142,  108.2, 199.3, -27.4,   2.2,  2.4, -1.5},
{ 4,1662,1661, 1552,   0.250,   5.019,  357.161,  110.4, 201.7, -28.9,   2.2,  2.4, -1.5},
{ 4,1663,1662, 1552,   0.250,   5.019,  362.181,  112.7, 204.0, -30.3,   2.2,  2.4, -1.5},
{ 4,1664,1663, 1552,   0.250,   5.019,  367.200,  114.9, 206.4, -31.8,   2.2,  2.4, -1.5},
{ 4,1665,1664, 1552,   0.250,   5.019,  372.219,  117.1, 208.8, -33.3,   2.2,  2.4, -1.5},
{ 4,1666,1665, 1552,   0.250,   5.019,  377.238,  119.3, 211.2, -34.8,   2.2,  2.4, -1.5},
{ 4,1667,1666, 1552,   0.250,   5.019,  382.257,  121.5, 213.5, -36.3,   2.2,  2.4, -1.5},
{ 4,1668,1612, 1558,   0.400,   4.296,  179.898,   30.5, 134.4,   1.2,   2.6,  2.5, -0.4},
{ 4,1669,1668, 1558,   0.400,   4.296,  184.194,   33.1, 136.9,   0.8,   2.6,  2.5, -0.4},
{ 4,1670,1669, 1558,   0.400,   4.296,  188.490,   35.7, 139.5,   0.3,   2.6,  2.5, -0.4},
{ 4,1671,1670, 1558,   0.400,   4.296,  192.786,   38.3, 142.0,  -0.1,   2.6,  2.5, -0.4},
{ 4,1672,1671, 1558,   0.400,   4.296,  197.082,   40.8, 144.5,  -0.6,   2.6,  2.5, -0.4},
{ 4,1673,1672, 1600,   0.243,   4.941,  201.700,   43.7, 146.2,  -1.0,   3.2,  0.8, -0.4},
{ 4,1674,1673, 1600,   0.243,   4.941,  206.642,   47.0, 146.9,  -1.4,   3.2,  0.8, -0.4},
{ 4,1675,1674, 1600,   0.243,   4.941,  211.583,   50.2, 147.7,  -1.7,   3.2,  0.8, -0.4},
{ 4,1676,1675, 1600,   0.243,   4.941,  216.525,   53.4, 148.5,  -2.1,   3.2,  0.8, -0.4},
{ 4,1677,1676, 1600,   0.243,   4.941,  221.466,   56.6, 149.3,  -2.5,   3.2,  0.8, -0.4},
{ 4,1678,1677, 1600,   0.243,   4.941,  226.407,   59.8, 150.1,  -2.9,   3.2,  0.8, -0.4},
{ 4,1679,1678, 1600,   0.243,   4.941,  231.349,   63.1, 150.9,  -3.2,   3.2,  0.8, -0.4},
{ 4,1680,1679, 1600,   0.243,   4.941,  236.290,   66.3, 151.7,  -3.6,   3.2,  0.8, -0.4},
{ 4,1681,1680, 1600,   0.243,   4.941,  241.232,   69.5, 152.5,  -4.0,   3.2,  0.8, -0.4},
{ 4,1682,1681, 1600,   0.243,   4.941,  246.173,   72.7, 153.3,  -4.4,   3.2,  0.8, -0.4},
{ 4,1683,1682, 1600,   0.243,   4.941,  251.115,   75.9, 154.1,  -4.7,   3.2,  0.8, -0.4},
{ 4,1684,1683, 1600,   0.243,   4.941,  256.056,   79.2, 154.9,  -5.1,   3.2,  0.8, -0.4},
{ 4,1685,1684, 1600,   0.243,   4.941,  260.997,   82.4, 155.7,  -5.5,   3.2,  0.8, -0.4},
{ 4,1686,1685, 1600,   0.243,   4.941,  265.939,   85.6, 156.5,  -5.9,   3.2,  0.8, -0.4},
{ 4,1687,1686, 1600,   0.243,   4.941,  270.880,   88.8, 157.3,  -6.2,   3.2,  0.8, -0.4},
{ 4,1688,1687, 1600,   0.243,   4.941,  275.822,   92.0, 158.1,  -6.6,   3.2,  0.8, -0.4},
{ 4,1689,1688, 1600,   0.243,   4.941,  280.763,   95.3, 158.9,  -7.0,   3.2,  0.8, -0.4},
{ 4,1690,1689, 1600,   0.243,   4.941,  285.704,   98.5, 159.6,  -7.4,   3.2,  0.8, -0.4},
{ 4,1691,1690, 1600,   0.243,   4.941,  290.646,  101.7, 160.4,  -7.7,   3.2,  0.8, -0.4},
{ 4,1692,1691, 1600,   0.243,   4.941,  295.587,  104.9, 161.2,  -8.1,   3.2,  0.8, -0.4},
{ 4,1693,1692, 1600,   0.243,   4.941,  300.529,  108.1, 162.0,  -8.5,   3.2,  0.8, -0.4},
{ 4,1694,1693, 1600,   0.243,   4.941,  305.470,  111.4, 162.8,  -8.9,   3.2,  0.8, -0.4},
{ 4,1695,1694, 1600,   0.243,   4.941,  310.411,  114.6, 163.6,  -9.2,   3.2,  0.8, -0.4},
{ 4,1696,1695, 1600,   0.243,   4.941,  315.353,  117.8, 164.4,  -9.6,   3.2,  0.8, -0.4},
{ 4,1697,1696, 1600,   0.243,   4.941,  320.294,  121.0, 165.2, -10.0,   3.2,  0.8, -0.4},
{ 4,1698,1697, 1600,   0.243,   4.941,  325.236,  124.2, 166.0, -10.4,   3.2,  0.8, -0.4},
{ 4,1699,1698, 1600,   0.243,   4.941,  330.177,  127.5, 166.8, -10.7,   3.2,  0.8, -0.4},
{ 4,1700,1699, 1600,   0.243,   4.941,  335.118,  130.7, 167.6, -11.1,   3.2,  0.8, -0.4},
{ 4,1701,1700, 1600,   0.243,   4.941,  340.060,  133.9, 168.4, -11.5,   3.2,  0.8, -0.4},
{ 4,1702,1701, 1600,   0.243,   4.941,  345.001,  137.1, 169.2, -11.9,   3.2,  0.8, -0.4},
{ 4,1703,1702, 1600,   0.243,   4.941,  349.943,  140.3, 170.0, -12.2,   3.2,  0.8, -0.4},
{ 4,1704,1703, 1600,   0.243,   4.941,  354.884,  143.6, 170.8, -12.6,   3.2,  0.8, -0.4},
{ 4,1705,1704, 1600,   0.243,   4.941,  359.825,  146.8, 171.5, -13.0,   3.2,  0.8, -0.4},
{ 4,1706,1705, 1600,   0.243,   4.941,  364.767,  150.0, 172.3, -13.4,   3.2,  0.8, -0.4},
{ 4,1707,1599, 1602,   0.400,   3.152,  115.740,    2.3, 127.0,   6.8,  -1.7,  2.2, -1.3},
{ 4,1708,1707, 1620,   0.240,   4.845,  119.739,    2.9, 127.9,   5.6,   2.8, -0.5, -1.2},
{ 4,1709,1708, 1620,   0.240,   4.845,  124.584,    5.6, 127.4,   4.4,   2.8, -0.5, -1.2},
{ 4,1710,1709, 1620,   0.240,   4.845,  129.428,    8.4, 126.9,   3.2,   2.8, -0.5, -1.2},
{ 4,1711,1710, 1620,   0.240,   4.845,  134.273,   11.1, 126.3,   2.0,   2.8, -0.5, -1.2},
{ 4,1712,1711, 1620,   0.240,   4.845,  139.118,   13.9, 125.8,   0.8,   2.8, -0.5, -1.2},
{ 4,1713,1712, 1620,   0.240,   4.845,  143.963,   16.7, 125.3,  -0.4,   2.8, -0.5, -1.2},
{ 4,1714,1713, 1620,   0.240,   4.845,  148.808,   19.4, 124.8,  -1.6,   2.8, -0.5, -1.2},
{ 4,1715,1714, 1620,   0.240,   4.845,  153.652,   22.2, 124.3,  -2.8,   2.8, -0.5, -1.2},
{ 4,1716,1715, 1620,   0.240,   4.845,  158.497,   25.0, 123.8,  -4.0,   2.8, -0.5, -1.2},
{ 4,1717,1716, 1620,   0.240,   4.845,  163.342,   27.7, 123.3,  -5.2,   2.8, -0.5, -1.2},
{ 4,1718,1717, 1620,   0.240,   4.845,  168.187,   30.5, 122.8,  -6.4,   2.8, -0.5, -1.2},
{ 4,1719,1718, 1620,   0.240,   4.845,  173.031,   33.3, 122.3,  -7.6,   2.8, -0.5, -1.2},
{ 4,1720,1719, 1620,   0.240,   4.845,  177.876,   36.0, 121.7,  -8.8,   2.8, -0.5, -1.2},
{ 4,1721,1720, 1620,   0.240,   4.845,  182.721,   38.8, 121.2, -10.0,   2.8, -0.5, -1.2},
{ 4,1722,1721, 1620,   0.240,   4.845,  187.566,   41.6, 120.7, -11.2,   2.8, -0.5, -1.2},
{ 4,1723,1722, 1620,   0.240,   4.845,  192.411,   44.3, 120.2, -12.4,   2.8, -0.5, -1.2},
{ 4,1724,1723, 1620,   0.240,   4.845,  197.255,   47.1, 119.7, -13.6,   2.8, -0.5, -1.2},
{ 4,1725,1724, 1620,   0.240,   4.845,  202.100,   49.8, 119.2, -14.8,   2.8, -0.5, -1.2},
{ 4,1726,1725, 1620,   0.240,   4.845,  206.945,   52.6, 118.7, -16.0,   2.8, -0.5, -1.2},
{ 4,1727,1726, 1620,   0.240,   4.845,  211.790,   55.4, 118.2, -17.2,   2.8, -0.5, -1.2},
{ 4,1728,1707, 1622,   0.400,   2.965,  118.799,    0.5, 127.9,   6.9,  -1.9, -0.4,  1.5},
{ 4,1729,1728, 1631,   0.240,   4.735,  122.649,    0.5, 128.7,   6.5,   1.9,  1.9, -2.3},
{ 4,1730,1729, 1631,   0.240,   4.735,  127.384,    2.5, 130.6,   4.2,   1.9,  1.9, -2.3},
{ 4,1731,1730, 1631,   0.240,   4.735,  132.119,    4.4, 132.5,   1.9,   1.9,  1.9, -2.3},
{ 4,1732,1731, 1631,   0.240,   4.735,  136.854,    6.4, 134.4,  -0.5,   1.9,  1.9, -2.3},
{ 4,1733,1732, 1631,   0.240,   4.735,  141.589,    8.3, 136.3,  -2.8,   1.9,  1.9, -2.3},
{ 4,1734,1733, 1631,   0.240,   4.735,  146.324,   10.2, 138.2,  -5.1,   1.9,  1.9, -2.3},
{ 4,1735,1734, 1631,   0.240,   4.735,  151.059,   12.2, 140.1,  -7.5,   1.9,  1.9, -2.3},
{ 4,1736,1735, 1631,   0.240,   4.735,  155.794,   14.1, 142.0,  -9.8,   1.9,  1.9, -2.3},
{ 4,1737,1736, 1631,   0.240,   4.735,  160.530,   16.1, 143.9, -12.1,   1.9,  1.9, -2.3},
{ 4,1738,1737, 1631,   0.240,   4.735,  165.265,   18.0, 145.8, -14.5,   1.9,  1.9, -2.3},
{ 4,1739,1738, 1631,   0.240,   4.735,  170.000,   20.0, 147.7, -16.8,   1.9,  1.9, -2.3},
{ 4,1740,1739, 1631,   0.240,   4.735,  174.735,   21.9, 149.6, -19.1,   1.9,  1.9, -2.3},
{ 4,1741,1740, 1649,   0.392,   5.020,  179.612,   24.1, 151.5, -20.2,   2.5,  1.9,  0.2},
{ 4,1742,1741, 1649,   0.392,   5.020,  184.633,   26.6, 153.4, -20.0,   2.5,  1.9,  0.2},
{ 4,1743,1742, 1649,   0.392,   5.020,  189.653,   29.0, 155.2, -19.8,   2.5,  1.9,  0.2},
{ 4,1744,1743, 1649,   0.392,   5.020,  194.673,   31.5, 157.1, -19.6,   2.5,  1.9,  0.2},
{ 4,1745,1744, 1649,   0.392,   5.020,  199.693,   33.9, 159.0, -19.4,   2.5,  1.9,  0.2},
{ 4,1746,1745, 1649,   0.392,   5.020,  204.713,   36.4, 160.9, -19.2,   2.5,  1.9,  0.2},
{ 4,1747,1746, 1649,   0.392,   5.020,  209.733,   38.9, 162.8, -19.0,   2.5,  1.9,  0.2},
{ 4,1748,1747, 1649,   0.392,   5.020,  214.753,   41.3, 164.6, -18.8,   2.5,  1.9,  0.2},
{ 4,1749,1748, 1649,   0.392,   5.020,  219.773,   43.8, 166.5, -18.6,   2.5,  1.9,  0.2},
{ 4,1750,1749, 1649,   0.392,   5.020,  224.794,   46.3, 168.4, -18.4,   2.5,  1.9,  0.2},
{ 4,1751,1750, 1649,   0.392,   5.020,  229.814,   48.7, 170.3, -18.2,   2.5,  1.9,  0.2},
{ 4,1752,1751, 1649,   0.392,   5.020,  234.834,   51.2, 172.2, -18.0,   2.5,  1.9,  0.2},
{ 4,1753,1752, 1649,   0.392,   5.020,  239.854,   53.7, 174.1, -17.8,   2.5,  1.9,  0.2},
{ 4,1754,1753, 1649,   0.392,   5.020,  244.874,   56.1, 175.9, -17.6,   2.5,  1.9,  0.2},
{ 4,1755,1754, 1649,   0.392,   5.020,  249.894,   58.6, 177.8, -17.4,   2.5,  1.9,  0.2},
{ 4,1756,1755, 1649,   0.392,   5.020,  254.914,   61.0, 179.7, -17.2,   2.5,  1.9,  0.2},
{ 4,1757,1756, 1649,   0.392,   5.020,  259.934,   63.5, 181.6, -17.0,   2.5,  1.9,  0.2},
{ 4,1758,1757, 1649,   0.392,   5.020,  264.954,   66.0, 183.5, -16.8,   2.5,  1.9,  0.2},
{ 4,1759,1758, 1649,   0.392,   5.020,  269.975,   68.4, 185.3, -16.6,   2.5,  1.9,  0.2},
{ 4,1760,1759, 1649,   0.392,   5.020,  274.995,   70.9, 187.2, -16.4,   2.5,  1.9,  0.2},
{ 4,1761,1760, 1649,   0.392,   5.020,  280.015,   73.4, 189.1, -16.2,   2.5,  1.9,  0.2},
{ 4,1762,1761, 1649,   0.392,   5.020,  285.035,   75.8, 191.0, -16.0,   2.5,  1.9,  0.2},
{ 4,1763,1762, 1649,   0.392,   5.020,  290.055,   78.3, 192.9, -15.8,   2.5,  1.9,  0.2},
{ 4,1764,1763, 1649,   0.392,   5.020,  295.075,   80.7, 194.7, -15.6,   2.5,  1.9,  0.2},
{ 4,1765,1728, 1652,   0.368,   4.271,  122.417,    0.2, 129.1,   8.7,   1.3,  2.8,  2.0},
{ 4,1766,1765, 1652,   0.368,   4.271,  126.687,    1.6, 131.9,  10.7,   1.3,  2.8,  2.0},
{ 4,1767,1766, 1652,   0.368,   4.271,  130.958,    2.9, 134.8,  12.7,   1.3,  2.8,  2.0},
{ 4,1768,1767, 1677,   0.240,   4.928,  135.558,    4.1, 137.4,  12.7,   1.1,  2.5, -2.0},
{ 4,1769,1768, 1677,   0.240,   4.928,  140.486,    5.2, 140.0,  10.8,   1.1,  2.5, -2.0},
{ 4,1770,1769, 1677,   0.240,   4.928,  145.414,    6.3, 142.5,   8.8,   1.1,  2.5, -2.0},
{ 4,1771,1770, 1677,   0.240,   4.928,  150.342,    7.4, 145.0,   6.8,   1.1,  2.5, -2.0},
{ 4,1772,1771, 1677,   0.240,   4.928,  155.271,    8.5, 147.5,   4.9,   1.1,  2.5, -2.0},
{ 4,1773,1772, 1677,   0.240,   4.928,  160.199,    9.6, 150.0,   2.9,   1.1,  2.5, -2.0},
{ 4,1774,1773, 1677,   0.240,   4.928,  165.127,   10.7, 152.5,   1.0,   1.1,  2.5, -2.0},
{ 4,1775,1774, 1677,   0.240,   4.928,  170.055,   11.7, 155.0,  -1.0,   1.1,  2.5, -2.0},
{ 4,1776,1775, 1677,   0.240,   4.928,  174.983,   12.8, 157.5,  -3.0,   1.1,  2.5, -2.0},
{ 4,1777,1776, 1677,   0.240,   4.928,  179.912,   13.9, 160.0,  -4.9,   1.1,  2.5, -2.0},
{ 4,1778,1777, 1677,   0.240,   4.928,  184.840,   15.0, 162.5,  -6.9,   1.1,  2.5, -2.0},
{ 4,1779,1778, 1677,   0.240,   4.928,  189.768,   16.1, 165.0,  -8.8,   1.1,  2.5, -2.0},
{ 4,1780,1779, 1677,   0.240,   4.928,  194.696,   17.2, 167.5, -10.8,   1.1,  2.5, -2.0},
{ 4,1781,1780, 1677,   0.240,   4.928,  199.624,   18.2, 170.0, -12.8,   1.1,  2.5, -2.0},
{ 4,1782,1781, 1677,   0.240,   4.928,  204.553,   19.3, 172.5, -14.7,   1.1,  2.5, -2.0},
{ 4,1783,1782, 1677,   0.240,   4.928,  209.481,   20.4, 175.0, -16.7,   1.1,  2.5, -2.0},
{ 4,1784,1783, 1677,   0.240,   4.928,  214.409,   21.5, 177.5, -18.6,   1.1,  2.5, -2.0},
{ 4,1785,1784, 1677,   0.240,   4.928,  219.337,   22.6, 180.0, -20.6,   1.1,  2.5, -2.0},
{ 4,1786,1785, 1677,   0.240,   4.928,  224.265,   23.7, 182.5, -22.6,   1.1,  2.5, -2.0},
{ 4,1787,1786, 1677,   0.240,   4.928,  229.193,   24.7, 185.1, -24.5,   1.1,  2.5, -2.0},
{ 4,1788,1787, 1677,   0.240,   4.928,  234.122,   25.8, 187.6, -26.5,   1.1,  2.5, -2.0},
{ 4,1789,1788, 1677,   0.240,   4.928,  239.050,   26.9, 190.1, -28.4,   1.1,  2.5, -2.0},
{ 4,1790,1789, 1677,   0.240,   4.928,  243.978,   28.0, 192.6, -30.4,   1.1,  2.5, -2.0},
{ 4,1791,1790, 1677,   0.240,   4.928,  248.906,   29.1, 195.1, -32.4,   1.1,  2.5, -2.0},
{ 4,1792,1791, 1677,   0.240,   4.928,  253.834,   30.2, 197.6, -34.3,   1.1,  2.5, -2.0},
{ 4,1793,1792, 1680,   0.321,   4.343,  258.470,   29.7, 199.8, -34.2,  -1.9,  1.9,  2.1},
{ 4,1794,1793, 1680,   0.321,   4.343,  262.813,   27.8, 201.7, -32.1,  -1.9,  1.9,  2.1},
{ 4,1795,1794, 1680,   0.321,   4.343,  267.156,   25.8, 203.6, -30.0,  -1.9,  1.9,  2.1},
{ 4,1796,1795, 1680,   0.321,   4.343,  271.499,   23.9, 205.5, -27.9,  -1.9,  1.9,  2.1},
{ 4,1797,1796, 1686,   0.240,   4.981,  276.161,   22.9, 208.2, -28.0,  -0.1,  3.3, -2.5},
{ 4,1798,1797, 1686,   0.240,   4.981,  281.142,   22.7, 211.5, -30.5,  -0.1,  3.3, -2.5},
{ 4,1799,1798, 1686,   0.240,   4.981,  286.122,   22.6, 214.8, -33.0,  -0.1,  3.3, -2.5},
{ 4,1800,1799, 1686,   0.240,   4.981,  291.103,   22.5, 218.1, -35.5,  -0.1,  3.3, -2.5},
{ 4,1801,1800, 1686,   0.240,   4.981,  296.084,   22.3, 221.4, -38.0,  -0.1,  3.3, -2.5},
{ 4,1802,1801, 1686,   0.240,   4.981,  301.065,   22.2, 224.7, -40.5,  -0.1,  3.3, -2.5},
{ 4,1803,1802, 1686,   0.240,   4.981,  306.045,   22.1, 228.0, -43.0,  -0.1,  3.3, -2.5},
{ 4,1804,1803, 1686,   0.240,   4.981,  311.026,   21.9, 231.3, -45.5,  -0.1,  3.3, -2.5},
{ 4,1805,1804, 1686,   0.240,   4.981,  316.007,   21.8, 234.6, -48.0,  -0.1,  3.3, -2.5},
{ 4,1806,1596, 1693,   0.880,   4.889,  105.190,   -3.9, 124.6,   0.7,   0.6,  4.6, -0.0},
{ 4,1807,1806, 1693,   0.880,   4.889,  110.079,   -3.3, 129.1,   0.7,   0.6,  4.6, -0.0},
{ 4,1808,1807, 1693,   0.880,   4.889,  114.968,   -2.7, 133.7,   0.7,   0.6,  4.6, -0.0},
{ 4,1809,1808, 1693,   0.880,   4.889,  119.857,   -2.1, 138.3,   0.7,   0.6,  4.6, -0.0},
{ 4,1810,1809, 1693,   0.880,   4.889,  124.745,   -1.5, 142.8,   0.7,   0.6,  4.6, -0.0},
{ 4,1811,1810, 1693,   0.880,   4.889,  129.634,   -0.9, 147.4,   0.7,   0.6,  4.6, -0.0},
{ 4,1812,1811, 1703,   0.400,   5.004,  134.581,    0.8, 150.9,   1.0,   2.8,  2.5,  0.7},
{ 4,1813,1812, 1703,   0.400,   5.004,  139.585,    3.6, 153.5,   1.7,   2.8,  2.5,  0.7},
{ 4,1814,1813, 1703,   0.400,   5.004,  144.588,    6.4, 156.0,   2.4,   2.8,  2.5,  0.7},
{ 4,1815,1814, 1703,   0.400,   5.004,  149.592,    9.2, 158.6,   3.0,   2.8,  2.5,  0.7},
{ 4,1816,1815, 1703,   0.400,   5.004,  154.596,   12.1, 161.1,   3.7,   2.8,  2.5,  0.7},
{ 4,1817,1816, 1703,   0.400,   5.004,  159.600,   14.9, 163.6,   4.4,   2.8,  2.5,  0.7},
{ 4,1818,1817, 1725,   0.240,   4.943,  164.573,   17.6, 165.2,   4.3,   2.6,  0.7, -0.8},
{ 4,1819,1818, 1725,   0.240,   4.943,  169.516,   20.3, 165.9,   3.5,   2.6,  0.7, -0.8},
{ 4,1820,1819, 1725,   0.240,   4.943,  174.458,   22.9, 166.6,   2.7,   2.6,  0.7, -0.8},
{ 4,1821,1820, 1725,   0.240,   4.943,  179.401,   25.5, 167.2,   1.9,   2.6,  0.7, -0.8},
{ 4,1822,1821, 1725,   0.240,   4.943,  184.343,   28.2, 167.9,   1.1,   2.6,  0.7, -0.8},
{ 4,1823,1822, 1725,   0.240,   4.943,  189.286,   30.8, 168.6,   0.3,   2.6,  0.7, -0.8},
{ 4,1824,1823, 1725,   0.240,   4.943,  194.229,   33.4, 169.2,  -0.5,   2.6,  0.7, -0.8},
{ 4,1825,1824, 1725,   0.240,   4.943,  199.171,   36.1, 169.9,  -1.3,   2.6,  0.7, -0.8},
{ 4,1826,1825, 1725,   0.240,   4.943,  204.114,   38.7, 170.5,  -2.1,   2.6,  0.7, -0.8},
{ 4,1827,1826, 1725,   0.240,   4.943,  209.056,   41.4, 171.2,  -2.9,   2.6,  0.7, -0.8},
{ 4,1828,1827, 1725,   0.240,   4.943,  213.999,   44.0, 171.9,  -3.7,   2.6,  0.7, -0.8},
{ 4,1829,1828, 1725,   0.240,   4.943,  218.941,   46.6, 172.5,  -4.4,   2.6,  0.7, -0.8},
{ 4,1830,1829, 1725,   0.240,   4.943,  223.884,   49.3, 173.2,  -5.2,   2.6,  0.7, -0.8},
{ 4,1831,1830, 1725,   0.240,   4.943,  228.827,   51.9, 173.9,  -6.0,   2.6,  0.7, -0.8},
{ 4,1832,1831, 1725,   0.240,   4.943,  233.769,   54.5, 174.5,  -6.8,   2.6,  0.7, -0.8},
{ 4,1833,1832, 1725,   0.240,   4.943,  238.712,   57.2, 175.2,  -7.6,   2.6,  0.7, -0.8},
{ 4,1834,1833, 1725,   0.240,   4.943,  243.654,   59.8, 175.8,  -8.4,   2.6,  0.7, -0.8},
{ 4,1835,1834, 1725,   0.240,   4.943,  248.597,   62.5, 176.5,  -9.2,   2.6,  0.7, -0.8},
{ 4,1836,1835, 1725,   0.240,   4.943,  253.539,   65.1, 177.2, -10.0,   2.6,  0.7, -0.8},
{ 4,1837,1836, 1725,   0.240,   4.943,  258.482,   67.7, 177.8, -10.8,   2.6,  0.7, -0.8},
{ 4,1838,1837, 1725,   0.240,   4.943,  263.425,   70.4, 178.5, -11.6,   2.6,  0.7, -0.8},
{ 4,1839,1838, 1725,   0.240,   4.943,  268.367,   73.0, 179.2, -12.4,   2.6,  0.7, -0.8},
{ 4,1840,1817, 1748,   0.400,   4.849,  164.527,   17.1, 166.2,   4.7,   1.6,  2.6,  0.0},
{ 4,1841,1840, 1748,   0.400,   4.849,  169.376,   18.7, 168.9,   4.7,   1.6,  2.6,  0.0},
{ 4,1842,1841, 1748,   0.400,   4.849,  174.226,   20.3, 171.5,   4.7,   1.6,  2.6,  0.0},
{ 4,1843,1842, 1748,   0.400,   4.849,  179.075,   21.9, 174.1,   4.8,   1.6,  2.6,  0.0},
{ 4,1844,1843, 1748,   0.400,   4.849,  183.924,   23.5, 176.8,   4.8,   1.6,  2.6,  0.0},
{ 4,1845,1844, 1748,   0.400,   4.849,  188.774,   25.1, 179.4,   4.8,   1.6,  2.6,  0.0},
{ 4,1846,1845, 1748,   0.400,   4.849,  193.623,   26.7, 182.1,   4.8,   1.6,  2.6,  0.0},
{ 4,1847,1846, 1748,   0.400,   4.849,  198.473,   28.3, 184.7,   4.8,   1.6,  2.6,  0.0},
{ 4,1848,1847, 1748,   0.400,   4.849,  203.322,   29.9, 187.3,   4.8,   1.6,  2.6,  0.0},
{ 4,1849,1848, 1748,   0.400,   4.849,  208.172,   31.5, 190.0,   4.9,   1.6,  2.6,  0.0},
{ 4,1850,1849, 1748,   0.400,   4.849,  213.021,   33.1, 192.6,   4.9,   1.6,  2.6,  0.0},
{ 4,1851,1850, 1748,   0.400,   4.849,  217.871,   34.7, 195.3,   4.9,   1.6,  2.6,  0.0},
{ 4,1852,1851, 1748,   0.400,   4.849,  222.720,   36.2, 197.9,   4.9,   1.6,  2.6,  0.0},
{ 4,1853,1852, 1748,   0.400,   4.849,  227.570,   37.8, 200.5,   4.9,   1.6,  2.6,  0.0},
{ 4,1854,1853, 1748,   0.400,   4.849,  232.419,   39.4, 203.2,   4.9,   1.6,  2.6,  0.0},
{ 4,1855,1854, 1781,   0.240,   4.839,  237.263,   41.0, 205.9,   4.2,   1.6,  2.9, -1.4},
{ 4,1856,1855, 1781,   0.240,   4.839,  242.102,   42.6, 208.8,   2.8,   1.6,  2.9, -1.4},
{ 4,1857,1856, 1781,   0.240,   4.839,  246.940,   44.2, 211.7,   1.4,   1.6,  2.9, -1.4},
{ 4,1858,1857, 1781,   0.240,   4.839,  251.779,   45.8, 214.6,   0.0,   1.6,  2.9, -1.4},
{ 4,1859,1858, 1781,   0.240,   4.839,  256.617,   47.3, 217.4,  -1.4,   1.6,  2.9, -1.4},
{ 4,1860,1859, 1781,   0.240,   4.839,  261.456,   48.9, 220.3,  -2.8,   1.6,  2.9, -1.4},
{ 4,1861,1860, 1781,   0.240,   4.839,  266.294,   50.5, 223.2,  -4.2,   1.6,  2.9, -1.4},
{ 4,1862,1861, 1781,   0.240,   4.839,  271.133,   52.1, 226.1,  -5.6,   1.6,  2.9, -1.4},
{ 4,1863,1862, 1781,   0.240,   4.839,  275.971,   53.7, 228.9,  -7.0,   1.6,  2.9, -1.4},
{ 4,1864,1863, 1781,   0.240,   4.839,  280.810,   55.2, 231.8,  -8.4,   1.6,  2.9, -1.4},
{ 4,1865,1864, 1781,   0.240,   4.839,  285.648,   56.8, 234.7,  -9.8,   1.6,  2.9, -1.4},
{ 4,1866,1865, 1781,   0.240,   4.839,  290.487,   58.4, 237.5, -11.2,   1.6,  2.9, -1.4},
{ 4,1867,1866, 1781,   0.240,   4.839,  295.326,   60.0, 240.4, -12.6,   1.6,  2.9, -1.4},
{ 4,1868,1867, 1781,   0.240,   4.839,  300.164,   61.6, 243.3, -14.0,   1.6,  2.9, -1.4},
{ 4,1869,1868, 1781,   0.240,   4.839,  305.003,   63.1, 246.2, -15.4,   1.6,  2.9, -1.4},
{ 4,1870,1869, 1781,   0.240,   4.839,  309.841,   64.7, 249.0, -16.8,   1.6,  2.9, -1.4},
{ 4,1871,1870, 1781,   0.240,   4.839,  314.680,   66.3, 251.9, -18.3,   1.6,  2.9, -1.4},
{ 4,1872,1871, 1781,   0.240,   4.839,  319.518,   67.9, 254.8, -19.7,   1.6,  2.9, -1.4},
{ 4,1873,1872, 1781,   0.240,   4.839,  324.357,   69.5, 257.7, -21.1,   1.6,  2.9, -1.4},
{ 4,1874,1873, 1781,   0.240,   4.839,  329.195,   71.0, 260.5, -22.5,   1.6,  2.9, -1.4},
{ 4,1875,1874, 1781,   0.240,   4.839,  334.034,   72.6, 263.4, -23.9,   1.6,  2.9, -1.4},
{ 4,1876,1875, 1781,   0.240,   4.839,  338.872,   74.2, 266.3, -25.3,   1.6,  2.9, -1.4},
{ 4,1877,1876, 1781,   0.240,   4.839,  343.711,   75.8, 269.2, -26.7,   1.6,  2.9, -1.4},
{ 4,1878,1877, 1781,   0.240,   4.839,  348.549,   77.3, 272.0, -28.1,   1.6,  2.9, -1.4},
{ 4,1879,1854, 1802,   0.400,   4.942,  237.315,   40.6, 206.3,   5.4,   0.7,  3.5,  0.9},
{ 4,1880,1879, 1802,   0.400,   4.942,  242.256,   41.3, 209.8,   6.3,   0.7,  3.5,  0.9},
{ 4,1881,1880, 1802,   0.400,   4.942,  247.198,   42.0, 213.3,   7.2,   0.7,  3.5,  0.9},
{ 4,1882,1881, 1802,   0.400,   4.942,  252.139,   42.7, 216.8,   8.1,   0.7,  3.5,  0.9},
{ 4,1883,1882, 1802,   0.400,   4.942,  257.081,   43.4, 220.3,   9.0,   0.7,  3.5,  0.9},
{ 4,1884,1883, 1802,   0.400,   4.942,  262.023,   44.1, 223.8,   9.9,   0.7,  3.5,  0.9},
{ 4,1885,1884, 1802,   0.400,   4.942,  266.964,   44.8, 227.3,  10.8,   0.7,  3.5,  0.9},
{ 4,1886,1885, 1802,   0.400,   4.942,  271.906,   45.4, 230.8,  11.6,   0.7,  3.5,  0.9},
{ 4,1887,1886, 1802,   0.400,   4.942,  276.847,   46.1, 234.3,  12.5,   0.7,  3.5,  0.9},
{ 4,1888,1887, 1802,   0.400,   4.942,  281.789,   46.8, 237.8,  13.4,   0.7,  3.5,  0.9},
{ 4,1889,1888, 1802,   0.400,   4.942,  286.731,   47.5, 241.3,  14.3,   0.7,  3.5,  0.9},
{ 4,1890,1889, 1802,   0.400,   4.942,  291.672,   48.2, 244.8,  15.2,   0.7,  3.5,  0.9},
{ 4,1891,1890, 1802,   0.400,   4.942,  296.614,   48.9, 248.3,  16.1,   0.7,  3.5,  0.9},
{ 4,1892,1891, 1802,   0.400,   4.942,  301.555,   49.6, 251.8,  17.0,   0.7,  3.5,  0.9},
{ 4,1893,1892, 1802,   0.400,   4.942,  306.497,   50.3, 255.3,  17.9,   0.7,  3.5,  0.9},
{ 4,1894,1893, 1802,   0.400,   4.942,  311.438,   51.0, 258.8,  18.8,   0.7,  3.5,  0.9},
{ 4,1895,1894, 1802,   0.400,   4.942,  316.380,   51.7, 262.3,  19.7,   0.7,  3.5,  0.9},
{ 4,1896,1895, 1802,   0.400,   4.942,  321.322,   52.4, 265.8,  20.6,   0.7,  3.5,  0.9},
{ 4,1897,1896, 1802,   0.400,   4.942,  326.263,   53.1, 269.3,  21.5,   0.7,  3.5,  0.9},
{ 4,1898,1897, 1802,   0.400,   4.942,  331.205,   53.8, 272.8,  22.4,   0.7,  3.5,  0.9},
{ 4,1899,1898, 1802,   0.400,   4.942,  336.146,   54.5, 276.3,  23.3,   0.7,  3.5,  0.9},
{ 4,1900,1811, 1808,   0.720,   4.936,  134.547,   -0.5, 152.0,   0.7,   0.3,  4.7,  0.0},
{ 4,1901,1900, 1808,   0.720,   4.936,  139.483,   -0.2, 156.7,   0.7,   0.3,  4.7,  0.0},
{ 4,1902,1901, 1808,   0.720,   4.936,  144.419,    0.2, 161.4,   0.7,   0.3,  4.7,  0.0},
{ 4,1903,1902, 1808,   0.720,   4.936,  149.355,    0.5, 166.0,   0.7,   0.3,  4.7,  0.0},
{ 4,1904,1903, 1808,   0.720,   4.936,  154.291,    0.8, 170.7,   0.7,   0.3,  4.7,  0.0},
{ 4,1905,1904, 1808,   0.720,   4.936,  159.227,    1.1, 175.4,   0.7,   0.3,  4.7,  0.0},
{ 4,1906,1905, 1810,   0.720,   0.890,  162.140,    1.3, 178.2,   0.7,   0.0,  0.9,  0.0},
{ 4,1907,1906, 1839,   0.240,   4.952,  165.061,    1.0, 180.0,   0.9,  -0.5,  2.7,  0.3},
{ 4,1908,1907, 1839,   0.240,   4.952,  170.013,    0.5, 182.7,   1.2,  -0.5,  2.7,  0.3},
{ 4,1909,1908, 1839,   0.240,   4.952,  174.966,   -0.1, 185.5,   1.5,  -0.5,  2.7,  0.3},
{ 4,1910,1909, 1839,   0.240,   4.952,  179.918,   -0.6, 188.2,   1.8,  -0.5,  2.7,  0.3},
{ 4,1911,1910, 1839,   0.240,   4.952,  184.870,   -1.1, 190.9,   2.1,  -0.5,  2.7,  0.3},
{ 4,1912,1911, 1839,   0.240,   4.952,  189.822,   -1.6, 193.7,   2.4,  -0.5,  2.7,  0.3},
{ 4,1913,1912, 1839,   0.240,   4.952,  194.774,   -2.2, 196.4,   2.7,  -0.5,  2.7,  0.3},
{ 4,1914,1913, 1839,   0.240,   4.952,  199.726,   -2.7, 199.2,   3.0,  -0.5,  2.7,  0.3},
{ 4,1915,1914, 1839,   0.240,   4.952,  204.678,   -3.2, 201.9,   3.3,  -0.5,  2.7,  0.3},
{ 4,1916,1915, 1839,   0.240,   4.952,  209.630,   -3.8, 204.7,   3.6,  -0.5,  2.7,  0.3},
{ 4,1917,1916, 1839,   0.240,   4.952,  214.582,   -4.3, 207.4,   3.9,  -0.5,  2.7,  0.3},
{ 4,1918,1917, 1839,   0.240,   4.952,  219.535,   -4.8, 210.2,   4.2,  -0.5,  2.7,  0.3},
{ 4,1919,1918, 1839,   0.240,   4.952,  224.487,   -5.4, 212.9,   4.5,  -0.5,  2.7,  0.3},
{ 4,1920,1919, 1839,   0.240,   4.952,  229.439,   -5.9, 215.6,   4.8,  -0.5,  2.7,  0.3},
{ 4,1921,1920, 1839,   0.240,   4.952,  234.391,   -6.4, 218.4,   5.1,  -0.5,  2.7,  0.3},
{ 4,1922,1921, 1839,   0.240,   4.952,  239.343,   -7.0, 221.1,   5.4,  -0.5,  2.7,  0.3},
{ 4,1923,1922, 1839,   0.240,   4.952,  244.295,   -7.5, 223.9,   5.7,  -0.5,  2.7,  0.3},
{ 4,1924,1923, 1839,   0.240,   4.952,  249.247,   -8.0, 226.6,   6.0,  -0.5,  2.7,  0.3},
{ 4,1925,1924, 1839,   0.240,   4.952,  254.199,   -8.6, 229.4,   6.3,  -0.5,  2.7,  0.3},
{ 4,1926,1925, 1839,   0.240,   4.952,  259.151,   -9.1, 232.1,   6.6,  -0.5,  2.7,  0.3},
{ 4,1927,1926, 1839,   0.240,   4.952,  264.104,   -9.6, 234.8,   6.9,  -0.5,  2.7,  0.3},
{ 4,1928,1927, 1839,   0.240,   4.952,  269.056,  -10.2, 237.6,   7.2,  -0.5,  2.7,  0.3},
{ 4,1929,1928, 1839,   0.240,   4.952,  274.008,  -10.7, 240.3,   7.5,  -0.5,  2.7,  0.3},
{ 4,1930,1929, 1839,   0.240,   4.952,  278.960,  -11.2, 243.1,   7.8,  -0.5,  2.7,  0.3},
{ 4,1931,1930, 1839,   0.240,   4.952,  283.912,  -11.8, 245.8,   8.1,  -0.5,  2.7,  0.3},
{ 4,1932,1931, 1839,   0.240,   4.952,  288.864,  -12.3, 248.6,   8.4,  -0.5,  2.7,  0.3},
{ 4,1933,1932, 1839,   0.240,   4.952,  293.816,  -12.8, 251.3,   8.7,  -0.5,  2.7,  0.3},
{ 4,1934,1933, 1839,   0.240,   4.952,  298.768,  -13.4, 254.0,   9.0,  -0.5,  2.7,  0.3},
{ 4,1935,1934, 1839,   0.240,   4.952,  303.720,  -13.9, 256.8,   9.3,  -0.5,  2.7,  0.3},
{ 4,1936,1935, 1839,   0.240,   4.952,  308.673,  -14.4, 259.5,   9.6,  -0.5,  2.7,  0.3},
{ 4,1937,1936, 1839,   0.240,   4.952,  313.625,  -14.9, 262.3,   9.9,  -0.5,  2.7,  0.3},
{ 4,1938,1937, 1839,   0.240,   4.952,  318.577,  -15.5, 265.0,  10.2,  -0.5,  2.7,  0.3},
{ 4,1939,1938, 1839,   0.240,   4.952,  323.529,  -16.0, 267.8,  10.5,  -0.5,  2.7,  0.3},
{ 4,1940,1939, 1839,   0.240,   4.952,  328.481,  -16.5, 270.5,  10.8,  -0.5,  2.7,  0.3},
{ 4,1941,1940, 1839,   0.240,   4.952,  333.433,  -17.1, 273.2,  11.1,  -0.5,  2.7,  0.3},
{ 4,1942,1941, 1839,   0.240,   4.952,  338.385,  -17.6, 276.0,  11.4,  -0.5,  2.7,  0.3},
{ 4,1943,1942, 1839,   0.240,   4.952,  343.337,  -18.1, 278.7,  11.7,  -0.5,  2.7,  0.3},
{ 4,1944,1943, 1839,   0.240,   4.952,  348.289,  -18.7, 281.5,  12.0,  -0.5,  2.7,  0.3},
{ 4,1945,1944, 1839,   0.240,   4.952,  353.241,  -19.2, 284.2,  12.3,  -0.5,  2.7,  0.3},
{ 4,1946,1945, 1839,   0.240,   4.952,  358.194,  -19.7, 287.0,  12.6,  -0.5,  2.7,  0.3},
{ 4,1947,1946, 1839,   0.240,   4.952,  363.146,  -20.3, 289.7,  12.9,  -0.5,  2.7,  0.3},
{ 4,1948,1906, 1842,   0.720,   4.082,  164.626,    1.9, 180.5,   0.9,   1.2,  3.8,  0.4},
{ 4,1949,1948, 1842,   0.720,   4.082,  168.708,    3.0, 184.3,   1.3,   1.2,  3.8,  0.4},
{ 4,1950,1949, 1849,   0.240,   4.318,  172.907,    3.0, 187.2,   2.7,  -1.2,  2.0,  2.4},
{ 4,1951,1950, 1849,   0.240,   4.318,  177.225,    1.8, 189.2,   5.1,  -1.2,  2.0,  2.4},
{ 4,1952,1951, 1849,   0.240,   4.318,  181.543,    0.6, 191.1,   7.5,  -1.2,  2.0,  2.4},
{ 4,1953,1952, 1849,   0.240,   4.318,  185.861,   -0.5, 193.1,   9.9,  -1.2,  2.0,  2.4},
{ 4,1954,1953, 1849,   0.240,   4.318,  190.179,   -1.7, 195.0,  12.3,  -1.2,  2.0,  2.4},
{ 4,1955,1949, 1854,   0.720,   4.578,  173.038,    4.1, 188.2,   1.0,   0.9,  3.9, -0.9},
{ 4,1956,1955, 1854,   0.720,   4.578,  177.616,    5.0, 192.0,   0.1,   0.9,  3.9, -0.9},
{ 4,1957,1956, 1854,   0.720,   4.578,  182.195,    5.9, 195.9,  -0.8,   0.9,  3.9, -0.9},
{ 4,1958,1957, 1854,   0.720,   4.578,  186.773,    6.9, 199.8,  -1.7,   0.9,  3.9, -0.9},
{ 4,1959,1958, 1854,   0.720,   4.578,  191.351,    7.8, 203.7,  -2.6,   0.9,  3.9, -0.9},
{ 4,1960,1959, 1895,   0.240,   4.925,  196.103,    8.4, 207.1,  -2.6,   0.3,  3.1,  0.8},
{ 4,1961,1960, 1895,   0.240,   4.925,  201.028,    8.7, 210.2,  -1.8,   0.3,  3.1,  0.8},
{ 4,1962,1961, 1895,   0.240,   4.925,  205.954,    9.1, 213.3,  -1.0,   0.3,  3.1,  0.8},
{ 4,1963,1962, 1895,   0.240,   4.925,  210.879,    9.4, 216.4,  -0.2,   0.3,  3.1,  0.8},
{ 4,1964,1963, 1895,   0.240,   4.925,  215.804,    9.7, 219.5,   0.7,   0.3,  3.1,  0.8},
{ 4,1965,1964, 1895,   0.240,   4.925,  220.729,   10.0, 222.6,   1.5,   0.3,  3.1,  0.8},
{ 4,1966,1965, 1895,   0.240,   4.925,  225.654,   10.4, 225.7,   2.3,   0.3,  3.1,  0.8},
{ 4,1967,1966, 1895,   0.240,   4.925,  230.579,   10.7, 228.8,   3.1,   0.3,  3.1,  0.8},
{ 4,1968,1967, 1895,   0.240,   4.925,  235.505,   11.0, 231.9,   4.0,   0.3,  3.1,  0.8},
{ 4,1969,1968, 1895,   0.240,   4.925,  240.430,   11.3, 235.0,   4.8,   0.3,  3.1,  0.8},
{ 4,1970,1969, 1895,   0.240,   4.925,  245.355,   11.6, 238.1,   5.6,   0.3,  3.1,  0.8},
{ 4,1971,1970, 1895,   0.240,   4.925,  250.280,   12.0, 241.2,   6.4,   0.3,  3.1,  0.8},
{ 4,1972,1971, 1895,   0.240,   4.925,  255.205,   12.3, 244.3,   7.3,   0.3,  3.1,  0.8},
{ 4,1973,1972, 1895,   0.240,   4.925,  260.131,   12.6, 247.4,   8.1,   0.3,  3.1,  0.8},
{ 4,1974,1973, 1895,   0.240,   4.925,  265.056,   12.9, 250.5,   8.9,   0.3,  3.1,  0.8},
{ 4,1975,1974, 1895,   0.240,   4.925,  269.981,   13.3, 253.6,   9.7,   0.3,  3.1,  0.8},
{ 4,1976,1975, 1895,   0.240,   4.925,  274.906,   13.6, 256.7,  10.6,   0.3,  3.1,  0.8},
{ 4,1977,1976, 1895,   0.240,   4.925,  279.831,   13.9, 259.8,  11.4,   0.3,  3.1,  0.8},
{ 4,1978,1977, 1895,   0.240,   4.925,  284.756,   14.2, 262.9,  12.2,   0.3,  3.1,  0.8},
{ 4,1979,1978, 1895,   0.240,   4.925,  289.682,   14.6, 266.0,  13.0,   0.3,  3.1,  0.8},
{ 4,1980,1979, 1895,   0.240,   4.925,  294.607,   14.9, 269.1,  13.9,   0.3,  3.1,  0.8},
{ 4,1981,1980, 1895,   0.240,   4.925,  299.532,   15.2, 272.2,  14.7,   0.3,  3.1,  0.8},
{ 4,1982,1981, 1895,   0.240,   4.925,  304.457,   15.5, 275.3,  15.5,   0.3,  3.1,  0.8},
{ 4,1983,1982, 1895,   0.240,   4.925,  309.382,   15.8, 278.4,  16.3,   0.3,  3.1,  0.8},
{ 4,1984,1983, 1895,   0.240,   4.925,  314.308,   16.2, 281.5,  17.2,   0.3,  3.1,  0.8},
{ 4,1985,1984, 1895,   0.240,   4.925,  319.233,   16.5, 284.6,  18.0,   0.3,  3.1,  0.8},
{ 4,1986,1985, 1895,   0.240,   4.925,  324.158,   16.8, 287.7,  18.8,   0.3,  3.1,  0.8},
{ 4,1987,1986, 1895,   0.240,   4.925,  329.083,   17.1, 290.8,  19.6,   0.3,  3.1,  0.8},
{ 4,1988,1987, 1895,   0.240,   4.925,  334.008,   17.5, 293.9,  20.5,   0.3,  3.1,  0.8},
{ 4,1989,1988, 1895,   0.240,   4.925,  338.933,   17.8, 297.0,  21.3,   0.3,  3.1,  0.8},
{ 4,1990,1989, 1895,   0.240,   4.925,  343.859,   18.1, 300.1,  22.1,   0.3,  3.1,  0.8},
{ 4,1991,1990, 1895,   0.240,   4.925,  348.784,   18.4, 303.2,  22.9,   0.3,  3.1,  0.8},
{ 4,1992,1991, 1895,   0.240,   4.925,  353.709,   18.8, 306.3,  23.8,   0.3,  3.1,  0.8},
{ 4,1993,1992, 1895,   0.240,   4.925,  358.634,   19.1, 309.4,  24.6,   0.3,  3.1,  0.8},
{ 4,1994,1993, 1895,   0.240,   4.925,  363.559,   19.4, 312.5,  25.4,   0.3,  3.1,  0.8},
{ 4,1995,1994, 1895,   0.240,   4.925,  368.484,   19.7, 315.6,  26.2,   0.3,  3.1,  0.8},
{ 4,1996,1995, 1895,   0.240,   4.925,  373.410,   20.0, 318.7,  27.1,   0.3,  3.1,  0.8},
{ 4,1997,1996, 1895,   0.240,   4.925,  378.335,   20.4, 321.8,  27.9,   0.3,  3.1,  0.8},
{ 4,1998,1997, 1895,   0.240,   4.925,  383.260,   20.7, 324.9,  28.7,   0.3,  3.1,  0.8},
{ 4,1999,1998, 1895,   0.240,   4.925,  388.185,   21.0, 328.0,  29.5,   0.3,  3.1,  0.8},
{ 4,2000,1959, 1897,   0.720,   4.760,  196.021,    9.0, 206.8,  -4.7,   1.5,  2.4, -3.3},
{ 4,2001,2000, 1925,   0.240,   5.008,  200.904,    8.8, 209.5,  -5.8,  -1.8,  3.1,  1.0},
{ 4,2002,2001, 1925,   0.240,   5.008,  205.912,    7.0, 212.6,  -4.9,  -1.8,  3.1,  1.0},
{ 4,2003,2002, 1925,   0.240,   5.008,  210.920,    5.2, 215.7,  -3.9,  -1.8,  3.1,  1.0},
{ 4,2004,2003, 1925,   0.240,   5.008,  215.928,    3.3, 218.8,  -3.0,  -1.8,  3.1,  1.0},
{ 4,2005,2004, 1925,   0.240,   5.008,  220.936,    1.5, 221.9,  -2.0,  -1.8,  3.1,  1.0},
{ 4,2006,2005, 1925,   0.240,   5.008,  225.944,   -0.3, 225.0,  -1.1,  -1.8,  3.1,  1.0},
{ 4,2007,2006, 1925,   0.240,   5.008,  230.952,   -2.2, 228.1,  -0.1,  -1.8,  3.1,  1.0},
{ 4,2008,2007, 1925,   0.240,   5.008,  235.960,   -4.0, 231.2,   0.8,  -1.8,  3.1,  1.0},
{ 4,2009,2008, 1925,   0.240,   5.008,  240.968,   -5.8, 234.3,   1.8,  -1.8,  3.1,  1.0},
{ 4,2010,2009, 1925,   0.240,   5.008,  245.976,   -7.7, 237.4,   2.8,  -1.8,  3.1,  1.0},
{ 4,2011,2010, 1925,   0.240,   5.008,  250.984,   -9.5, 240.5,   3.7,  -1.8,  3.1,  1.0},
{ 4,2012,2011, 1925,   0.240,   5.008,  255.992,  -11.3, 243.6,   4.7,  -1.8,  3.1,  1.0},
{ 4,2013,2012, 1925,   0.240,   5.008,  261.000,  -13.2, 246.7,   5.6,  -1.8,  3.1,  1.0},
{ 4,2014,2013, 1925,   0.240,   5.008,  266.008,  -15.0, 249.8,   6.6,  -1.8,  3.1,  1.0},
{ 4,2015,2014, 1925,   0.240,   5.008,  271.016,  -16.8, 252.9,   7.5,  -1.8,  3.1,  1.0},
{ 4,2016,2015, 1925,   0.240,   5.008,  276.023,  -18.7, 256.0,   8.5,  -1.8,  3.1,  1.0},
{ 4,2017,2016, 1925,   0.240,   5.008,  281.031,  -20.5, 259.1,   9.4,  -1.8,  3.1,  1.0},
{ 4,2018,2017, 1925,   0.240,   5.008,  286.039,  -22.3, 262.2,  10.4,  -1.8,  3.1,  1.0},
{ 4,2019,2018, 1925,   0.240,   5.008,  291.047,  -24.2, 265.3,  11.3,  -1.8,  3.1,  1.0},
{ 4,2020,2019, 1925,   0.240,   5.008,  296.055,  -26.0, 268.4,  12.3,  -1.8,  3.1,  1.0},
{ 4,2021,2020, 1925,   0.240,   5.008,  301.063,  -27.8, 271.5,  13.2,  -1.8,  3.1,  1.0},
{ 4,2022,2021, 1925,   0.240,   5.008,  306.071,  -29.7, 274.6,  14.2,  -1.8,  3.1,  1.0},
{ 4,2023,2022, 1925,   0.240,   5.008,  311.079,  -31.5, 277.7,  15.1,  -1.8,  3.1,  1.0},
{ 4,2024,2023, 1925,   0.240,   5.008,  316.087,  -33.3, 280.8,  16.1,  -1.8,  3.1,  1.0},
{ 4,2025,2024, 1925,   0.240,   5.008,  321.095,  -35.2, 283.9,  17.1,  -1.8,  3.1,  1.0},
{ 4,2026,2025, 1925,   0.240,   5.008,  326.103,  -37.0, 287.0,  18.0,  -1.8,  3.1,  1.0},
{ 4,2027,2026, 1925,   0.240,   5.008,  331.111,  -38.8, 290.1,  19.0,  -1.8,  3.1,  1.0},
{ 4,2028,2027, 1925,   0.240,   5.008,  336.119,  -40.7, 293.2,  19.9,  -1.8,  3.1,  1.0},
{ 4,2029,2028, 1925,   0.240,   5.008,  341.127,  -42.5, 296.3,  20.9,  -1.8,  3.1,  1.0},
{ 4,2030,2029, 1925,   0.240,   5.008,  346.135,  -44.3, 299.4,  21.8,  -1.8,  3.1,  1.0},
{ 4,2031,2030, 1925,   0.240,   5.008,  351.143,  -46.2, 302.5,  22.8,  -1.8,  3.1,  1.0},
{ 4,2032,2031, 1925,   0.240,   5.008,  356.150,  -48.0, 305.6,  23.7,  -1.8,  3.1,  1.0},
{ 4,2033,2000, 1936,   0.560,   4.479,  200.640,   10.1, 209.7,  -6.2,   0.8,  3.4,  0.2},
{ 4,2034,2033, 1936,   0.560,   4.479,  205.118,   11.0, 213.1,  -6.0,   0.8,  3.4,  0.2},
{ 4,2035,2034, 1936,   0.560,   4.479,  209.597,   11.8, 216.5,  -5.8,   0.8,  3.4,  0.2},
{ 4,2036,2035, 1936,   0.560,   4.479,  214.076,   12.6, 219.9,  -5.5,   0.8,  3.4,  0.2},
{ 4,2037,2036, 1936,   0.560,   4.479,  218.554,   13.4, 223.4,  -5.3,   0.8,  3.4,  0.2},
{ 4,2038,2037, 1936,   0.560,   4.479,  223.033,   14.2, 226.8,  -5.1,   0.8,  3.4,  0.2},
{ 4,2039,2038, 1936,   0.560,   4.479,  227.511,   15.0, 230.2,  -4.9,   0.8,  3.4,  0.2},
{ 4,2040,2039, 1956,   0.405,   4.946,  232.224,   16.0, 233.9,  -5.2,   1.2,  4.0, -0.7},
{ 4,2041,2040, 1956,   0.405,   4.946,  237.170,   17.2, 237.9,  -5.9,   1.2,  4.0, -0.7},
{ 4,2042,2041, 1956,   0.405,   4.946,  242.116,   18.4, 241.9,  -6.6,   1.2,  4.0, -0.7},
{ 4,2043,2042, 1956,   0.405,   4.946,  247.062,   19.6, 245.9,  -7.4,   1.2,  4.0, -0.7},
{ 4,2044,2043, 1956,   0.405,   4.946,  252.008,   20.7, 249.9,  -8.1,   1.2,  4.0, -0.7},
{ 4,2045,2044, 1956,   0.405,   4.946,  256.954,   21.9, 253.9,  -8.8,   1.2,  4.0, -0.7},
{ 4,2046,2045, 1956,   0.405,   4.946,  261.900,   23.1, 257.9,  -9.6,   1.2,  4.0, -0.7},
{ 4,2047,2046, 1956,   0.405,   4.946,  266.846,   24.2, 261.9, -10.3,   1.2,  4.0, -0.7},
{ 4,2048,2047, 1956,   0.405,   4.946,  271.792,   25.4, 265.8, -11.1,   1.2,  4.0, -0.7},
{ 4,2049,2048, 1956,   0.405,   4.946,  276.738,   26.6, 269.8, -11.8,   1.2,  4.0, -0.7},
{ 4,2050,2049, 1956,   0.405,   4.946,  281.684,   27.8, 273.8, -12.5,   1.2,  4.0, -0.7},
{ 4,2051,2050, 1956,   0.405,   4.946,  286.630,   28.9, 277.8, -13.3,   1.2,  4.0, -0.7},
{ 4,2052,2051, 1956,   0.405,   4.946,  291.576,   30.1, 281.8, -14.0,   1.2,  4.0, -0.7},
{ 4,2053,2052, 1956,   0.405,   4.946,  296.522,   31.3, 285.8, -14.7,   1.2,  4.0, -0.7},
{ 4,2054,2053, 1956,   0.405,   4.946,  301.468,   32.4, 289.8, -15.5,   1.2,  4.0, -0.7},
{ 4,2055,2054, 1956,   0.405,   4.946,  306.414,   33.6, 293.8, -16.2,   1.2,  4.0, -0.7},
{ 4,2056,2055, 1956,   0.405,   4.946,  311.360,   34.8, 297.8, -16.9,   1.2,  4.0, -0.7},
{ 4,2057,2056, 1964,   0.400,   4.299,  315.982,   36.5, 300.7, -18.1,   2.3,  1.8, -1.7},
{ 4,2058,2057, 1964,   0.400,   4.299,  320.281,   38.9, 302.5, -19.8,   2.3,  1.8, -1.7},
{ 4,2059,2058, 1964,   0.400,   4.299,  324.579,   41.2, 304.3, -21.5,   2.3,  1.8, -1.7},
{ 4,2060,2059, 1964,   0.400,   4.299,  328.878,   43.5, 306.1, -23.1,   2.3,  1.8, -1.7},
{ 4,2061,2060, 1964,   0.400,   4.299,  333.177,   45.8, 307.9, -24.8,   2.3,  1.8, -1.7},
{ 4,2062,2061, 1964,   0.400,   4.299,  337.475,   48.2, 309.8, -26.5,   2.3,  1.8, -1.7},
{ 4,2063,2062, 2002,   0.242,   4.912,  342.081,   50.1, 312.0, -27.1,   1.5,  2.8,  0.4},
{ 4,2064,2063, 2002,   0.242,   4.912,  346.993,   51.7, 314.8, -26.6,   1.5,  2.8,  0.4},
{ 4,2065,2064, 2002,   0.242,   4.912,  351.905,   53.2, 317.6, -26.2,   1.5,  2.8,  0.4},
{ 4,2066,2065, 2002,   0.242,   4.912,  356.817,   54.7, 320.3, -25.8,   1.5,  2.8,  0.4},
{ 4,2067,2066, 2002,   0.242,   4.912,  361.729,   56.3, 323.1, -25.3,   1.5,  2.8,  0.4},
{ 4,2068,2067, 2002,   0.242,   4.912,  366.641,   57.8, 325.9, -24.9,   1.5,  2.8,  0.4},
{ 4,2069,2068, 2002,   0.242,   4.912,  371.553,   59.3, 328.6, -24.4,   1.5,  2.8,  0.4},
{ 4,2070,2069, 2002,   0.242,   4.912,  376.465,   60.9, 331.4, -24.0,   1.5,  2.8,  0.4},
{ 4,2071,2070, 2002,   0.242,   4.912,  381.378,   62.4, 334.1, -23.5,   1.5,  2.8,  0.4},
{ 4,2072,2071, 2002,   0.242,   4.912,  386.290,   64.0, 336.9, -23.1,   1.5,  2.8,  0.4},
{ 4,2073,2072, 2002,   0.242,   4.912,  391.202,   65.5, 339.7, -22.7,   1.5,  2.8,  0.4},
{ 4,2074,2073, 2002,   0.242,   4.912,  396.114,   67.0, 342.4, -22.2,   1.5,  2.8,  0.4},
{ 4,2075,2074, 2002,   0.242,   4.912,  401.026,   68.6, 345.2, -21.8,   1.5,  2.8,  0.4},
{ 4,2076,2075, 2002,   0.242,   4.912,  405.938,   70.1, 348.0, -21.3,   1.5,  2.8,  0.4},
{ 4,2077,2076, 2002,   0.242,   4.912,  410.850,   71.6, 350.7, -20.9,   1.5,  2.8,  0.4},
{ 4,2078,2077, 2002,   0.242,   4.912,  415.763,   73.2, 353.5, -20.5,   1.5,  2.8,  0.4},
{ 4,2079,2078, 2002,   0.242,   4.912,  420.675,   74.7, 356.2, -20.0,   1.5,  2.8,  0.4},
{ 4,2080,2079, 2002,   0.242,   4.912,  425.587,   76.3, 359.0, -19.6,   1.5,  2.8,  0.4},
{ 4,2081,2080, 2002,   0.242,   4.912,  430.499,   77.8, 361.8, -19.1,   1.5,  2.8,  0.4},
{ 4,2082,2081, 2002,   0.242,   4.912,  435.411,   79.3, 364.5, -18.7,   1.5,  2.8,  0.4},
{ 4,2083,2082, 2002,   0.242,   4.912,  440.323,   80.9, 367.3, -18.2,   1.5,  2.8,  0.4},
{ 4,2084,2083, 2002,   0.242,   4.912,  445.235,   82.4, 370.1, -17.8,   1.5,  2.8,  0.4},
{ 4,2085,2084, 2002,   0.242,   4.912,  450.147,   84.0, 372.8, -17.4,   1.5,  2.8,  0.4},
{ 4,2086,2085, 2002,   0.242,   4.912,  455.060,   85.5, 375.6, -16.9,   1.5,  2.8,  0.4},
{ 4,2087,2086, 2002,   0.242,   4.912,  459.972,   87.0, 378.4, -16.5,   1.5,  2.8,  0.4},
{ 4,2088,2087, 2002,   0.242,   4.912,  464.884,   88.6, 381.1, -16.0,   1.5,  2.8,  0.4},
{ 4,2089,2088, 2002,   0.242,   4.912,  469.796,   90.1, 383.9, -15.6,   1.5,  2.8,  0.4},
{ 4,2090,2089, 2002,   0.242,   4.912,  474.708,   91.6, 386.6, -15.2,   1.5,  2.8,  0.4},
{ 4,2091,2090, 2002,   0.242,   4.912,  479.620,   93.2, 389.4, -14.7,   1.5,  2.8,  0.4},
{ 4,2092,2091, 2002,   0.242,   4.912,  484.532,   94.7, 392.2, -14.3,   1.5,  2.8,  0.4},
{ 4,2093,2092, 2019,   0.240,   5.009,  489.493,   96.0, 394.7, -14.7,   1.0,  2.2, -1.3},
{ 4,2094,2093, 2019,   0.240,   5.009,  494.502,   97.0, 396.9, -16.0,   1.0,  2.2, -1.3},
{ 4,2095,2094, 2019,   0.240,   5.009,  499.510,   98.0, 399.1, -17.3,   1.0,  2.2, -1.3},
{ 4,2096,2095, 2019,   0.240,   5.009,  504.519,   98.9, 401.3, -18.6,   1.0,  2.2, -1.3},
{ 4,2097,2096, 2019,   0.240,   5.009,  509.528,   99.9, 403.5, -19.8,   1.0,  2.2, -1.3},
{ 4,2098,2097, 2019,   0.240,   5.009,  514.537,  100.9, 405.8, -21.1,   1.0,  2.2, -1.3},
{ 4,2099,2098, 2019,   0.240,   5.009,  519.546,  101.9, 408.0, -22.4,   1.0,  2.2, -1.3},
{ 4,2100,2099, 2019,   0.240,   5.009,  524.554,  102.9, 410.2, -23.7,   1.0,  2.2, -1.3},
{ 4,2101,2100, 2019,   0.240,   5.009,  529.563,  103.9, 412.4, -25.0,   1.0,  2.2, -1.3},
{ 4,2102,2101, 2019,   0.240,   5.009,  534.572,  104.8, 414.7, -26.3,   1.0,  2.2, -1.3},
{ 4,2103,2102, 2019,   0.240,   5.009,  539.581,  105.8, 416.9, -27.6,   1.0,  2.2, -1.3},
{ 4,2104,2103, 2019,   0.240,   5.009,  544.590,  106.8, 419.1, -28.8,   1.0,  2.2, -1.3},
{ 4,2105,2104, 2019,   0.240,   5.009,  549.598,  107.8, 421.3, -30.1,   1.0,  2.2, -1.3},
{ 4,2106,2105, 2019,   0.240,   5.009,  554.607,  108.8, 423.5, -31.4,   1.0,  2.2, -1.3},
{ 4,2107,2092, 2048,   0.240,   4.965,  489.471,   95.6, 395.4, -14.2,   0.3,  3.7, -0.3},
{ 4,2108,2107, 2048,   0.240,   4.965,  494.436,   96.0, 399.1, -14.4,   0.3,  3.7, -0.3},
{ 4,2109,2108, 2048,   0.240,   4.965,  499.402,   96.3, 402.9, -14.7,   0.3,  3.7, -0.3},
{ 4,2110,2109, 2048,   0.240,   4.965,  504.367,   96.6, 406.6, -15.0,   0.3,  3.7, -0.3},
{ 4,2111,2110, 2048,   0.240,   4.965,  509.332,   96.9, 410.3, -15.2,   0.3,  3.7, -0.3},
{ 4,2112,2111, 2048,   0.240,   4.965,  514.298,   97.2, 414.0, -15.5,   0.3,  3.7, -0.3},
{ 4,2113,2112, 2048,   0.240,   4.965,  519.263,   97.5, 417.8, -15.7,   0.3,  3.7, -0.3},
{ 4,2114,2113, 2048,   0.240,   4.965,  524.228,   97.8, 421.5, -16.0,   0.3,  3.7, -0.3},
{ 4,2115,2114, 2048,   0.240,   4.965,  529.194,   98.1, 425.2, -16.3,   0.3,  3.7, -0.3},
{ 4,2116,2115, 2048,   0.240,   4.965,  534.159,   98.4, 429.0, -16.5,   0.3,  3.7, -0.3},
{ 4,2117,2116, 2048,   0.240,   4.965,  539.124,   98.7, 432.7, -16.8,   0.3,  3.7, -0.3},
{ 4,2118,2117, 2048,   0.240,   4.965,  544.090,   99.0, 436.4, -17.1,   0.3,  3.7, -0.3},
{ 4,2119,2118, 2048,   0.240,   4.965,  549.055,   99.3, 440.1, -17.3,   0.3,  3.7, -0.3},
{ 4,2120,2119, 2048,   0.240,   4.965,  554.020,   99.6, 443.9, -17.6,   0.3,  3.7, -0.3},
{ 4,2121,2120, 2048,   0.240,   4.965,  558.986,   99.9, 447.6, -17.8,   0.3,  3.7, -0.3},
{ 4,2122,2121, 2048,   0.240,   4.965,  563.951,  100.2, 451.3, -18.1,   0.3,  3.7, -0.3},
{ 4,2123,2122, 2048,   0.240,   4.965,  568.916,  100.5, 455.0, -18.4,   0.3,  3.7, -0.3},
{ 4,2124,2123, 2048,   0.240,   4.965,  573.882,  100.8, 458.8, -18.6,   0.3,  3.7, -0.3},
{ 4,2125,2124, 2048,   0.240,   4.965,  578.847,  101.1, 462.5, -18.9,   0.3,  3.7, -0.3},
{ 4,2126,2125, 2048,   0.240,   4.965,  583.812,  101.4, 466.2, -19.1,   0.3,  3.7, -0.3},
{ 4,2127,2126, 2048,   0.240,   4.965,  588.778,  101.7, 469.9, -19.4,   0.3,  3.7, -0.3},
{ 4,2128,2127, 2048,   0.240,   4.965,  593.743,  102.0, 473.7, -19.7,   0.3,  3.7, -0.3},
{ 4,2129,2128, 2048,   0.240,   4.965,  598.708,  102.3, 477.4, -19.9,   0.3,  3.7, -0.3},
{ 4,2130,2056, 2062,   0.400,   4.750,  316.208,   35.1, 301.7, -17.9,  -0.6,  3.9, -1.2},
{ 4,2131,2130, 2062,   0.400,   4.750,  320.958,   34.5, 305.6, -19.1,  -0.6,  3.9, -1.2},
{ 4,2132,2131, 2062,   0.400,   4.750,  325.708,   34.0, 309.5, -20.3,  -0.6,  3.9, -1.2},
{ 4,2133,2132, 2062,   0.400,   4.750,  330.458,   33.4, 313.4, -21.5,  -0.6,  3.9, -1.2},
{ 4,2134,2133, 2062,   0.400,   4.750,  335.208,   32.8, 317.2, -22.7,  -0.6,  3.9, -1.2},
{ 4,2135,2134, 2062,   0.400,   4.750,  339.958,   32.3, 321.1, -23.9,  -0.6,  3.9, -1.2},
{ 4,2136,2135, 2062,   0.400,   4.750,  344.708,   31.7, 325.0, -25.1,  -0.6,  3.9, -1.2},
{ 4,2137,2136, 2062,   0.400,   4.750,  349.458,   31.2, 328.9, -26.3,  -0.6,  3.9, -1.2},
{ 4,2138,2137, 2062,   0.400,   4.750,  354.208,   30.6, 332.7, -27.6,  -0.6,  3.9, -1.2},
{ 4,2139,2138, 2062,   0.400,   4.750,  358.958,   30.0, 336.6, -28.8,  -0.6,  3.9, -1.2},
{ 4,2140,2139, 2062,   0.400,   4.750,  363.708,   29.5, 340.5, -30.0,  -0.6,  3.9, -1.2},
{ 4,2141,2140, 2062,   0.400,   4.750,  368.458,   28.9, 344.4, -31.2,  -0.6,  3.9, -1.2},
{ 4,2142,2141, 2062,   0.400,   4.750,  373.208,   28.4, 348.2, -32.4,  -0.6,  3.9, -1.2},
{ 4,2143,2142, 2062,   0.400,   4.750,  377.958,   27.8, 352.1, -33.6,  -0.6,  3.9, -1.2},
{ 4,2144,2143, 2062,   0.400,   4.750,  382.708,   27.2, 356.0, -34.8,  -0.6,  3.9, -1.2},
{ 4,2145,2144, 2062,   0.400,   4.750,  387.458,   26.7, 359.9, -36.0,  -0.6,  3.9, -1.2},
{ 4,2146,2145, 2062,   0.400,   4.750,  392.208,   26.1, 363.8, -37.2,  -0.6,  3.9, -1.2},
{ 4,2147,2146, 2102,   0.240,   4.936,  397.051,   25.6, 367.5, -38.0,  -0.4,  3.7, -0.3},
{ 4,2148,2147, 2102,   0.240,   4.936,  401.987,   25.2, 371.2, -38.3,  -0.4,  3.7, -0.3},
{ 4,2149,2148, 2102,   0.240,   4.936,  406.923,   24.7, 374.9, -38.6,  -0.4,  3.7, -0.3},
{ 4,2150,2149, 2102,   0.240,   4.936,  411.859,   24.3, 378.6, -38.9,  -0.4,  3.7, -0.3},
{ 4,2151,2150, 2102,   0.240,   4.936,  416.795,   23.8, 382.3, -39.2,  -0.4,  3.7, -0.3},
{ 4,2152,2151, 2102,   0.240,   4.936,  421.731,   23.4, 385.9, -39.5,  -0.4,  3.7, -0.3},
{ 4,2153,2152, 2102,   0.240,   4.936,  426.667,   23.0, 389.6, -39.8,  -0.4,  3.7, -0.3},
{ 4,2154,2153, 2102,   0.240,   4.936,  431.603,   22.5, 393.3, -40.1,  -0.4,  3.7, -0.3},
{ 4,2155,2154, 2102,   0.240,   4.936,  436.539,   22.1, 397.0, -40.5,  -0.4,  3.7, -0.3},
{ 4,2156,2155, 2102,   0.240,   4.936,  441.474,   21.6, 400.7, -40.8,  -0.4,  3.7, -0.3},
{ 4,2157,2156, 2102,   0.240,   4.936,  446.410,   21.2, 404.3, -41.1,  -0.4,  3.7, -0.3},
{ 4,2158,2157, 2102,   0.240,   4.936,  451.346,   20.8, 408.0, -41.4,  -0.4,  3.7, -0.3},
{ 4,2159,2158, 2102,   0.240,   4.936,  456.282,   20.3, 411.7, -41.7,  -0.4,  3.7, -0.3},
{ 4,2160,2159, 2102,   0.240,   4.936,  461.218,   19.9, 415.4, -42.0,  -0.4,  3.7, -0.3},
{ 4,2161,2160, 2102,   0.240,   4.936,  466.154,   19.4, 419.1, -42.3,  -0.4,  3.7, -0.3},
{ 4,2162,2161, 2102,   0.240,   4.936,  471.090,   19.0, 422.7, -42.6,  -0.4,  3.7, -0.3},
{ 4,2163,2162, 2102,   0.240,   4.936,  476.026,   18.6, 426.4, -43.0,  -0.4,  3.7, -0.3},
{ 4,2164,2163, 2102,   0.240,   4.936,  480.962,   18.1, 430.1, -43.3,  -0.4,  3.7, -0.3},
{ 4,2165,2164, 2102,   0.240,   4.936,  485.898,   17.7, 433.8, -43.6,  -0.4,  3.7, -0.3},
{ 4,2166,2165, 2102,   0.240,   4.936,  490.834,   17.2, 437.5, -43.9,  -0.4,  3.7, -0.3},
{ 4,2167,2166, 2102,   0.240,   4.936,  495.770,   16.8, 441.1, -44.2,  -0.4,  3.7, -0.3},
{ 4,2168,2167, 2102,   0.240,   4.936,  500.706,   16.3, 444.8, -44.5,  -0.4,  3.7, -0.3},
{ 4,2169,2168, 2102,   0.240,   4.936,  505.642,   15.9, 448.5, -44.8,  -0.4,  3.7, -0.3},
{ 4,2170,2169, 2102,   0.240,   4.936,  510.578,   15.5, 452.2, -45.1,  -0.4,  3.7, -0.3},
{ 4,2171,2170, 2102,   0.240,   4.936,  515.514,   15.0, 455.9, -45.5,  -0.4,  3.7, -0.3},
{ 4,2172,2171, 2102,   0.240,   4.936,  520.450,   14.6, 459.5, -45.8,  -0.4,  3.7, -0.3},
{ 4,2173,2172, 2102,   0.240,   4.936,  525.386,   14.1, 463.2, -46.1,  -0.4,  3.7, -0.3},
{ 4,2174,2173, 2102,   0.240,   4.936,  530.321,   13.7, 466.9, -46.4,  -0.4,  3.7, -0.3},
{ 4,2175,2174, 2102,   0.240,   4.936,  535.257,   13.3, 470.6, -46.7,  -0.4,  3.7, -0.3},
{ 4,2176,2175, 2102,   0.240,   4.936,  540.193,   12.8, 474.3, -47.0,  -0.4,  3.7, -0.3},
{ 4,2177,2176, 2102,   0.240,   4.936,  545.129,   12.4, 477.9, -47.3,  -0.4,  3.7, -0.3},
{ 4,2178,2177, 2102,   0.240,   4.936,  550.065,   11.9, 481.6, -47.6,  -0.4,  3.7, -0.3},
{ 4,2179,2178, 2102,   0.240,   4.936,  555.001,   11.5, 485.3, -48.0,  -0.4,  3.7, -0.3},
{ 4,2180,2179, 2102,   0.240,   4.936,  559.937,   11.1, 489.0, -48.3,  -0.4,  3.7, -0.3},
{ 4,2181,2180, 2102,   0.240,   4.936,  564.873,   10.6, 492.7, -48.6,  -0.4,  3.7, -0.3},
{ 4,2182,2181, 2102,   0.240,   4.936,  569.809,   10.2, 496.4, -48.9,  -0.4,  3.7, -0.3},
{ 4,2183,2146, 2146,   0.240,   4.995,  397.080,   25.0, 367.4, -37.0,  -1.6,  3.4,  1.5},
{ 4,2184,2183, 2146,   0.240,   4.995,  402.075,   23.4, 370.8, -35.5,  -1.6,  3.4,  1.5},
{ 4,2185,2184, 2146,   0.240,   4.995,  407.070,   21.8, 374.2, -33.9,  -1.6,  3.4,  1.5},
{ 4,2186,2185, 2146,   0.240,   4.995,  412.065,   20.3, 377.5, -32.4,  -1.6,  3.4,  1.5},
{ 4,2187,2186, 2146,   0.240,   4.995,  417.060,   18.7, 380.9, -30.9,  -1.6,  3.4,  1.5},
{ 4,2188,2187, 2146,   0.240,   4.995,  422.055,   17.1, 384.3, -29.3,  -1.6,  3.4,  1.5},
{ 4,2189,2188, 2146,   0.240,   4.995,  427.050,   15.5, 387.7, -27.8,  -1.6,  3.4,  1.5},
{ 4,2190,2189, 2146,   0.240,   4.995,  432.045,   13.9, 391.1, -26.2,  -1.6,  3.4,  1.5},
{ 4,2191,2190, 2146,   0.240,   4.995,  437.040,   12.3, 394.5, -24.7,  -1.6,  3.4,  1.5},
{ 4,2192,2191, 2146,   0.240,   4.995,  442.034,   10.7, 397.8, -23.2,  -1.6,  3.4,  1.5},
{ 4,2193,2192, 2146,   0.240,   4.995,  447.029,    9.1, 401.2, -21.6,  -1.6,  3.4,  1.5},
{ 4,2194,2193, 2146,   0.240,   4.995,  452.024,    7.5, 404.6, -20.1,  -1.6,  3.4,  1.5},
{ 4,2195,2194, 2146,   0.240,   4.995,  457.019,    5.9, 408.0, -18.5,  -1.6,  3.4,  1.5},
{ 4,2196,2195, 2146,   0.240,   4.995,  462.014,    4.3, 411.4, -17.0,  -1.6,  3.4,  1.5},
{ 4,2197,2196, 2146,   0.240,   4.995,  467.009,    2.7, 414.8, -15.4,  -1.6,  3.4,  1.5},
{ 4,2198,2197, 2146,   0.240,   4.995,  472.004,    1.1, 418.2, -13.9,  -1.6,  3.4,  1.5},
{ 4,2199,2198, 2146,   0.240,   4.995,  476.999,   -0.5, 421.5, -12.4,  -1.6,  3.4,  1.5},
{ 4,2200,2199, 2146,   0.240,   4.995,  481.994,   -2.1, 424.9, -10.8,  -1.6,  3.4,  1.5},
{ 4,2201,2200, 2146,   0.240,   4.995,  486.988,   -3.7, 428.3,  -9.3,  -1.6,  3.4,  1.5},
{ 4,2202,2201, 2146,   0.240,   4.995,  491.983,   -5.3, 431.7,  -7.7,  -1.6,  3.4,  1.5},
{ 4,2203,2202, 2146,   0.240,   4.995,  496.978,   -6.8, 435.1,  -6.2,  -1.6,  3.4,  1.5},
{ 4,2204,2203, 2146,   0.240,   4.995,  501.973,   -8.4, 438.5,  -4.7,  -1.6,  3.4,  1.5},
{ 4,2205,2204, 2146,   0.240,   4.995,  506.968,  -10.0, 441.8,  -3.1,  -1.6,  3.4,  1.5},
{ 4,2206,2205, 2146,   0.240,   4.995,  511.963,  -11.6, 445.2,  -1.6,  -1.6,  3.4,  1.5},
{ 4,2207,2206, 2146,   0.240,   4.995,  516.958,  -13.2, 448.6,  -0.0,  -1.6,  3.4,  1.5},
{ 4,2208,2207, 2146,   0.240,   4.995,  521.953,  -14.8, 452.0,   1.5,  -1.6,  3.4,  1.5},
{ 4,2209,2208, 2146,   0.240,   4.995,  526.947,  -16.4, 455.4,   3.1,  -1.6,  3.4,  1.5},
{ 4,2210,2209, 2146,   0.240,   4.995,  531.942,  -18.0, 458.8,   4.6,  -1.6,  3.4,  1.5},
{ 4,2211,2210, 2146,   0.240,   4.995,  536.937,  -19.6, 462.1,   6.1,  -1.6,  3.4,  1.5},
{ 4,2212,2211, 2146,   0.240,   4.995,  541.932,  -21.2, 465.5,   7.7,  -1.6,  3.4,  1.5},
{ 4,2213,2212, 2146,   0.240,   4.995,  546.927,  -22.8, 468.9,   9.2,  -1.6,  3.4,  1.5},
{ 4,2214,2213, 2146,   0.240,   4.995,  551.922,  -24.4, 472.3,  10.8,  -1.6,  3.4,  1.5},
{ 4,2215,2214, 2146,   0.240,   4.995,  556.917,  -26.0, 475.7,  12.3,  -1.6,  3.4,  1.5},
{ 4,2216,2215, 2146,   0.240,   4.995,  561.912,  -27.6, 479.1,  13.8,  -1.6,  3.4,  1.5},
{ 4,2217,2216, 2146,   0.240,   4.995,  566.907,  -29.2, 482.4,  15.4,  -1.6,  3.4,  1.5},
{ 4,2218,2217, 2146,   0.240,   4.995,  571.901,  -30.8, 485.8,  16.9,  -1.6,  3.4,  1.5},
{ 4,2219,1905, 2149,   0.345,   3.380,  163.385,    0.3, 178.5,   9.0,  -2.0,  1.4, 16.6},
{ 4,2220,2219, 2149,   0.345,   3.380,  166.765,   -1.7, 179.9,  25.7,  -2.0,  1.4, 16.6},
{ 4,2221,2220, 2157,   0.400,   5.004,  170.957,   -2.3, 182.2,  34.5,   0.7,  3.3,  0.9},
{ 4,2222,2221, 2157,   0.400,   5.004,  175.961,   -1.5, 185.5,  35.4,   0.7,  3.3,  0.9},
{ 4,2223,2222, 2157,   0.400,   5.004,  180.965,   -0.8, 188.8,  36.3,   0.7,  3.3,  0.9},
{ 4,2224,2223, 2157,   0.400,   5.004,  185.969,   -0.1, 192.1,  37.2,   0.7,  3.3,  0.9},
{ 4,2225,2224, 2157,   0.400,   5.004,  190.973,    0.7, 195.3,  38.1,   0.7,  3.3,  0.9},
{ 4,2226,2225, 2157,   0.400,   5.004,  195.977,    1.4, 198.6,  39.0,   0.7,  3.3,  0.9},
{ 4,2227,2226, 2177,   0.249,   4.974,  200.966,    2.2, 201.8,  40.2,   1.0,  3.0,  1.4},
{ 4,2228,2227, 2177,   0.249,   4.974,  205.940,    3.2, 204.8,  41.5,   1.0,  3.0,  1.4},
{ 4,2229,2228, 2177,   0.249,   4.974,  210.914,    4.1, 207.9,  42.9,   1.0,  3.0,  1.4},
{ 4,2230,2229, 2177,   0.249,   4.974,  215.888,    5.1, 210.9,  44.2,   1.0,  3.0,  1.4},
{ 4,2231,2230, 2177,   0.249,   4.974,  220.862,    6.0, 214.0,  45.6,   1.0,  3.0,  1.4},
{ 4,2232,2231, 2177,   0.249,   4.974,  225.836,    7.0, 217.0,  46.9,   1.0,  3.0,  1.4},
{ 4,2233,2232, 2177,   0.249,   4.974,  230.810,    7.9, 220.0,  48.3,   1.0,  3.0,  1.4},
{ 4,2234,2233, 2177,   0.249,   4.974,  235.784,    8.9, 223.1,  49.6,   1.0,  3.0,  1.4},
{ 4,2235,2234, 2177,   0.249,   4.974,  240.758,    9.8, 226.1,  51.0,   1.0,  3.0,  1.4},
{ 4,2236,2235, 2177,   0.249,   4.974,  245.732,   10.8, 229.2,  52.3,   1.0,  3.0,  1.4},
{ 4,2237,2236, 2177,   0.249,   4.974,  250.706,   11.7, 232.2,  53.7,   1.0,  3.0,  1.4},
{ 4,2238,2237, 2177,   0.249,   4.974,  255.680,   12.7, 235.3,  55.0,   1.0,  3.0,  1.4},
{ 4,2239,2238, 2177,   0.249,   4.974,  260.654,   13.6, 238.3,  56.4,   1.0,  3.0,  1.4},
{ 4,2240,2239, 2177,   0.249,   4.974,  265.628,   14.6, 241.4,  57.8,   1.0,  3.0,  1.4},
{ 4,2241,2240, 2177,   0.249,   4.974,  270.602,   15.5, 244.4,  59.1,   1.0,  3.0,  1.4},
{ 4,2242,2241, 2177,   0.249,   4.974,  275.576,   16.5, 247.5,  60.5,   1.0,  3.0,  1.4},
{ 4,2243,2242, 2177,   0.249,   4.974,  280.550,   17.4, 250.5,  61.8,   1.0,  3.0,  1.4},
{ 4,2244,2243, 2177,   0.249,   4.974,  285.524,   18.4, 253.6,  63.2,   1.0,  3.0,  1.4},
{ 4,2245,2244, 2177,   0.249,   4.974,  290.498,   19.3, 256.6,  64.5,   1.0,  3.0,  1.4},
{ 4,2246,2245, 2177,   0.249,   4.974,  295.472,   20.3, 259.7,  65.9,   1.0,  3.0,  1.4},
{ 4,2247,2246, 2177,   0.249,   4.974,  300.446,   21.2, 262.7,  67.2,   1.0,  3.0,  1.4},
{ 4,2248,2247, 2177,   0.249,   4.974,  305.420,   22.2, 265.8,  68.6,   1.0,  3.0,  1.4},
{ 4,2249,2248, 2177,   0.249,   4.974,  310.394,   23.1, 268.8,  69.9,   1.0,  3.0,  1.4},
{ 4,2250,2249, 2177,   0.249,   4.974,  315.368,   24.1, 271.9,  71.3,   1.0,  3.0,  1.4},
{ 4,2251,2250, 2177,   0.249,   4.974,  320.342,   25.0, 274.9,  72.6,   1.0,  3.0,  1.4},
{ 4,2252,2251, 2177,   0.249,   4.974,  325.316,   26.0, 278.0,  74.0,   1.0,  3.0,  1.4},
{ 4,2253,2252, 2177,   0.249,   4.974,  330.290,   26.9, 281.0,  75.3,   1.0,  3.0,  1.4},
{ 3,2254,   0, 2183,   1.040,   4.727,    2.363,   -1.7,  -5.1,  -1.1,  -1.2, -1.1, -4.3},
{ 3,2255,2254, 2183,   1.040,   4.727,    7.090,   -2.9,  -6.2,  -5.4,  -1.2, -1.1, -4.3},
{ 3,2256,2255, 2183,   1.040,   4.727,   11.817,   -4.1,  -7.2,  -9.7,  -1.2, -1.1, -4.3},
{ 3,2257,2256, 2183,   1.040,   4.727,   16.544,   -5.3,  -8.3, -14.0,  -1.2, -1.1, -4.3},
{ 3,2258,2257, 2183,   1.040,   4.727,   21.271,   -6.5,  -9.4, -18.3,  -1.2, -1.1, -4.3},
{ 3,2259,2258, 2183,   1.040,   4.727,   25.998,   -7.7, -10.5, -22.6,  -1.2, -1.1, -4.3},
{ 3,2260,2259, 2183,   1.040,   4.727,   30.725,   -9.0, -11.5, -26.9,  -1.2, -1.1, -4.3},
{ 3,2261,2260, 2200,   0.599,   4.967,   35.572,  -10.4, -13.2, -28.2,  -1.7, -2.3,  1.5},
{ 3,2262,2261, 2200,   0.599,   4.967,   40.539,  -12.1, -15.5, -26.7,  -1.7, -2.3,  1.5},
{ 3,2263,2262, 2200,   0.599,   4.967,   45.506,  -13.9, -17.7, -25.2,  -1.7, -2.3,  1.5},
{ 3,2264,2263, 2200,   0.599,   4.967,   50.473,  -15.6, -20.0, -23.7,  -1.7, -2.3,  1.5},
{ 3,2265,2264, 2200,   0.599,   4.967,   55.440,  -17.3, -22.3, -22.2,  -1.7, -2.3,  1.5},
{ 3,2266,2265, 2200,   0.599,   4.967,   60.407,  -19.0, -24.5, -20.7,  -1.7, -2.3,  1.5},
{ 3,2267,2266, 2200,   0.599,   4.967,   65.374,  -20.7, -26.8, -19.2,  -1.7, -2.3,  1.5},
{ 3,2268,2267, 2200,   0.599,   4.967,   70.341,  -22.4, -29.1, -17.7,  -1.7, -2.3,  1.5},
{ 3,2269,2268, 2200,   0.599,   4.967,   75.308,  -24.2, -31.3, -16.1,  -1.7, -2.3,  1.5},
{ 3,2270,2269, 2200,   0.599,   4.967,   80.275,  -25.9, -33.6, -14.6,  -1.7, -2.3,  1.5},
{ 3,2271,2270, 2200,   0.599,   4.967,   85.242,  -27.6, -35.9, -13.1,  -1.7, -2.3,  1.5},
{ 3,2272,2271, 2200,   0.599,   4.967,   90.209,  -29.3, -38.1, -11.6,  -1.7, -2.3,  1.5},
{ 3,2273,2272, 2200,   0.599,   4.967,   95.176,  -31.0, -40.4, -10.1,  -1.7, -2.3,  1.5},
{ 3,2274,2273, 2200,   0.599,   4.967,  100.143,  -32.7, -42.7,  -8.6,  -1.7, -2.3,  1.5},
{ 3,2275,2274, 2200,   0.599,   4.967,  105.110,  -34.5, -44.9,  -7.1,  -1.7, -2.3,  1.5},
{ 3,2276,2275, 2200,   0.599,   4.967,  110.077,  -36.2, -47.2,  -5.5,  -1.7, -2.3,  1.5},
{ 3,2277,2276, 2200,   0.599,   4.967,  115.044,  -37.9, -49.5,  -4.0,  -1.7, -2.3,  1.5},
{ 3,2278,2277, 2200,   0.599,   4.967,  120.011,  -39.6, -51.7,  -2.5,  -1.7, -2.3,  1.5},
{ 3,2279,2278, 2200,   0.599,   4.967,  124.978,  -41.3, -54.0,  -1.0,  -1.7, -2.3,  1.5},
{ 3,2280,2279, 2204,   0.432,   5.060,  129.991,  -42.8, -55.7,   0.0,  -1.3, -1.1,  0.5},
{ 3,2281,2280, 2204,   0.432,   5.060,  135.052,  -44.2, -56.9,   0.5,  -1.3, -1.1,  0.5},
{ 3,2282,2281, 2204,   0.432,   5.060,  140.112,  -45.5, -58.0,   1.0,  -1.3, -1.1,  0.5},
{ 3,2283,2282, 2204,   0.432,   5.060,  145.172,  -46.8, -59.1,   1.5,  -1.3, -1.1,  0.5},
{ 3,2284,2283, 2204,   0.432,   5.060,  150.232,  -48.1, -60.3,   2.0,  -1.3, -1.1,  0.5},
{ 3,2285,2284, 2204,   0.432,   5.060,  155.293,  -49.5, -61.4,   2.5,  -1.3, -1.1,  0.5},
{ 3,2286,2285, 2204,   0.432,   5.060,  160.353,  -50.8, -62.6,   3.0,  -1.3, -1.1,  0.5},
{ 3,2287,2286, 2204,   0.432,   5.060,  165.413,  -52.1, -63.7,   3.5,  -1.3, -1.1,  0.5},
{ 3,2288,2287, 2213,   0.240,   4.731,  170.309,  -53.2, -65.4,   3.4,  -0.9, -2.2, -0.7},
{ 3,2289,2288, 2213,   0.240,   4.731,  175.041,  -54.1, -67.6,   2.7,  -0.9, -2.2, -0.7},
{ 3,2290,2289, 2213,   0.240,   4.731,  179.772,  -54.9, -69.8,   2.1,  -0.9, -2.2, -0.7},
{ 3,2291,2290, 2213,   0.240,   4.731,  184.504,  -55.8, -72.0,   1.4,  -0.9, -2.2, -0.7},
{ 3,2292,2291, 2213,   0.240,   4.731,  189.235,  -56.6, -74.2,   0.7,  -0.9, -2.2, -0.7},
{ 3,2293,2292, 2213,   0.240,   4.731,  193.967,  -57.5, -76.4,   0.0,  -0.9, -2.2, -0.7},
{ 3,2294,2293, 2213,   0.240,   4.731,  198.698,  -58.4, -78.6,  -0.7,  -0.9, -2.2, -0.7},
{ 3,2295,2294, 2213,   0.240,   4.731,  203.429,  -59.2, -80.8,  -1.3,  -0.9, -2.2, -0.7},
{ 3,2296,2295, 2213,   0.240,   4.731,  208.161,  -60.1, -83.0,  -2.0,  -0.9, -2.2, -0.7},
{ 3,2297,2296, 2213,   0.240,   4.731,  212.892,  -61.0, -85.2,  -2.7,  -0.9, -2.2, -0.7},
{ 3,2298,2297, 2213,   0.240,   4.731,  217.624,  -61.8, -87.5,  -3.4,  -0.9, -2.2, -0.7},
{ 3,2299,2298, 2213,   0.240,   4.731,  222.355,  -62.7, -89.7,  -4.1,  -0.9, -2.2, -0.7},
{ 3,2300,2299, 2213,   0.240,   4.731,  227.087,  -63.5, -91.9,  -4.7,  -0.9, -2.2, -0.7},
{ 3,2301,2300, 2213,   0.240,   4.731,  231.818,  -64.4, -94.1,  -5.4,  -0.9, -2.2, -0.7},
{ 3,2302,2301, 2245,   0.399,   4.900,  236.634,  -64.9, -97.3,  -5.5,  -0.1, -4.3,  0.6},
{ 3,2303,2302, 2245,   0.399,   4.900,  241.535,  -65.0,-101.7,  -4.9,  -0.1, -4.3,  0.6},
{ 3,2304,2303, 2245,   0.399,   4.900,  246.435,  -65.0,-106.0,  -4.3,  -0.1, -4.3,  0.6},
{ 3,2305,2304, 2245,   0.399,   4.900,  251.335,  -65.1,-110.3,  -3.8,  -0.1, -4.3,  0.6},
{ 3,2306,2305, 2245,   0.399,   4.900,  256.235,  -65.2,-114.6,  -3.2,  -0.1, -4.3,  0.6},
{ 3,2307,2306, 2245,   0.399,   4.900,  261.135,  -65.3,-118.9,  -2.6,  -0.1, -4.3,  0.6},
{ 3,2308,2307, 2245,   0.399,   4.900,  266.036,  -65.4,-123.2,  -2.1,  -0.1, -4.3,  0.6},
{ 3,2309,2308, 2245,   0.399,   4.900,  270.936,  -65.4,-127.6,  -1.5,  -0.1, -4.3,  0.6},
{ 3,2310,2309, 2245,   0.399,   4.900,  275.836,  -65.5,-131.9,  -0.9,  -0.1, -4.3,  0.6},
{ 3,2311,2310, 2245,   0.399,   4.900,  280.736,  -65.6,-136.2,  -0.3,  -0.1, -4.3,  0.6},
{ 3,2312,2311, 2245,   0.399,   4.900,  285.636,  -65.7,-140.5,   0.2,  -0.1, -4.3,  0.6},
{ 3,2313,2312, 2245,   0.399,   4.900,  290.537,  -65.8,-144.8,   0.8,  -0.1, -4.3,  0.6},
{ 3,2314,2313, 2245,   0.399,   4.900,  295.437,  -65.8,-149.1,   1.4,  -0.1, -4.3,  0.6},
{ 3,2315,2314, 2245,   0.399,   4.900,  300.337,  -65.9,-153.5,   1.9,  -0.1, -4.3,  0.6},
{ 3,2316,2315, 2245,   0.399,   4.900,  305.237,  -66.0,-157.8,   2.5,  -0.1, -4.3,  0.6},
{ 3,2317,2316, 2245,   0.399,   4.900,  310.137,  -66.1,-162.1,   3.1,  -0.1, -4.3,  0.6},
{ 3,2318,2317, 2245,   0.399,   4.900,  315.038,  -66.2,-166.4,   3.6,  -0.1, -4.3,  0.6},
{ 3,2319,2318, 2245,   0.399,   4.900,  319.938,  -66.2,-170.7,   4.2,  -0.1, -4.3,  0.6},
{ 3,2320,2319, 2245,   0.399,   4.900,  324.838,  -66.3,-175.1,   4.8,  -0.1, -4.3,  0.6},
{ 3,2321,2320, 2245,   0.399,   4.900,  329.738,  -66.4,-179.4,   5.3,  -0.1, -4.3,  0.6},
{ 3,2322,2321, 2245,   0.399,   4.900,  334.639,  -66.5,-183.7,   5.9,  -0.1, -4.3,  0.6},
{ 3,2323,2322, 2245,   0.399,   4.900,  339.539,  -66.6,-188.0,   6.5,  -0.1, -4.3,  0.6},
{ 3,2324,2323, 2245,   0.399,   4.900,  344.439,  -66.6,-192.3,   7.1,  -0.1, -4.3,  0.6},
{ 3,2325,2324, 2245,   0.399,   4.900,  349.339,  -66.7,-196.6,   7.6,  -0.1, -4.3,  0.6},
{ 3,2326,2325, 2245,   0.399,   4.900,  354.239,  -66.8,-201.0,   8.2,  -0.1, -4.3,  0.6},
{ 3,2327,2326, 2245,   0.399,   4.900,  359.140,  -66.9,-205.3,   8.8,  -0.1, -4.3,  0.6},
{ 3,2328,2327, 2245,   0.399,   4.900,  364.040,  -67.0,-209.6,   9.3,  -0.1, -4.3,  0.6},
{ 3,2329,2328, 2245,   0.399,   4.900,  368.940,  -67.0,-213.9,   9.9,  -0.1, -4.3,  0.6},
{ 3,2330,2329, 2245,   0.399,   4.900,  373.840,  -67.1,-218.2,  10.5,  -0.1, -4.3,  0.6},
{ 3,2331,2287, 2256,   0.240,   4.658,  170.272,  -54.3, -65.5,   3.5,  -3.1, -2.4, -0.5},
{ 3,2332,2331, 2256,   0.240,   4.658,  174.930,  -57.4, -67.8,   2.9,  -3.1, -2.4, -0.5},
{ 3,2333,2332, 2256,   0.240,   4.658,  179.588,  -60.4, -70.2,   2.4,  -3.1, -2.4, -0.5},
{ 3,2334,2333, 2256,   0.240,   4.658,  184.246,  -63.5, -72.5,   1.8,  -3.1, -2.4, -0.5},
{ 3,2335,2334, 2256,   0.240,   4.658,  188.904,  -66.5, -74.9,   1.3,  -3.1, -2.4, -0.5},
{ 3,2336,2335, 2256,   0.240,   4.658,  193.562,  -69.6, -77.2,   0.8,  -3.1, -2.4, -0.5},
{ 3,2337,2336, 2256,   0.240,   4.658,  198.220,  -72.7, -79.6,   0.2,  -3.1, -2.4, -0.5},
{ 3,2338,2337, 2256,   0.240,   4.658,  202.878,  -75.7, -81.9,  -0.3,  -3.1, -2.4, -0.5},
{ 3,2339,2338, 2256,   0.240,   4.658,  207.536,  -78.8, -84.3,  -0.9,  -3.1, -2.4, -0.5},
{ 3,2340,2339, 2256,   0.240,   4.658,  212.194,  -81.8, -86.6,  -1.4,  -3.1, -2.4, -0.5},
{ 3,2341,2340, 2256,   0.240,   4.658,  216.852,  -84.9, -89.0,  -2.0,  -3.1, -2.4, -0.5},
{ 3,2342,2341, 2276,   0.399,   4.938,  221.650,  -87.7, -91.7,  -2.2,  -2.6, -3.0,  0.1},
{ 3,2343,2342, 2276,   0.399,   4.938,  226.588,  -90.3, -94.6,  -2.1,  -2.6, -3.0,  0.1},
{ 3,2344,2343, 2276,   0.399,   4.938,  231.526,  -93.0, -97.6,  -2.0,  -2.6, -3.0,  0.1},
{ 3,2345,2344, 2276,   0.399,   4.938,  236.464,  -95.6,-100.6,  -1.9,  -2.6, -3.0,  0.1},
{ 3,2346,2345, 2276,   0.399,   4.938,  241.402,  -98.2,-103.5,  -1.8,  -2.6, -3.0,  0.1},
{ 3,2347,2346, 2276,   0.399,   4.938,  246.340, -100.8,-106.5,  -1.7,  -2.6, -3.0,  0.1},
{ 3,2348,2347, 2276,   0.399,   4.938,  251.278, -103.4,-109.5,  -1.6,  -2.6, -3.0,  0.1},
{ 3,2349,2348, 2276,   0.399,   4.938,  256.216, -106.0,-112.4,  -1.5,  -2.6, -3.0,  0.1},
{ 3,2350,2349, 2276,   0.399,   4.938,  261.154, -108.6,-115.4,  -1.4,  -2.6, -3.0,  0.1},
{ 3,2351,2350, 2276,   0.399,   4.938,  266.092, -111.2,-118.4,  -1.3,  -2.6, -3.0,  0.1},
{ 3,2352,2351, 2276,   0.399,   4.938,  271.030, -113.8,-121.3,  -1.2,  -2.6, -3.0,  0.1},
{ 3,2353,2352, 2276,   0.399,   4.938,  275.969, -116.4,-124.3,  -1.1,  -2.6, -3.0,  0.1},
{ 3,2354,2353, 2276,   0.399,   4.938,  280.907, -119.0,-127.3,  -1.0,  -2.6, -3.0,  0.1},
{ 3,2355,2354, 2276,   0.399,   4.938,  285.845, -121.6,-130.2,  -1.0,  -2.6, -3.0,  0.1},
{ 3,2356,2355, 2276,   0.399,   4.938,  290.783, -124.2,-133.2,  -0.9,  -2.6, -3.0,  0.1},
{ 3,2357,2356, 2276,   0.399,   4.938,  295.721, -126.8,-136.2,  -0.8,  -2.6, -3.0,  0.1},
{ 3,2358,2357, 2276,   0.399,   4.938,  300.659, -129.4,-139.1,  -0.7,  -2.6, -3.0,  0.1},
{ 3,2359,2358, 2276,   0.399,   4.938,  305.597, -132.0,-142.1,  -0.6,  -2.6, -3.0,  0.1},
{ 3,2360,2359, 2276,   0.399,   4.938,  310.535, -134.6,-145.1,  -0.5,  -2.6, -3.0,  0.1},
{ 3,2361,2360, 2276,   0.399,   4.938,  315.473, -137.2,-148.0,  -0.4,  -2.6, -3.0,  0.1},
{ 3,2362,2361, 2276,   0.399,   4.938,  320.411, -139.8,-151.0,  -0.3,  -2.6, -3.0,  0.1},
{ 3,2363,2362, 2276,   0.399,   4.938,  325.349, -142.5,-154.0,  -0.2,  -2.6, -3.0,  0.1},
{ 3,2364,2363, 2276,   0.399,   4.938,  330.287, -145.1,-156.9,  -0.1,  -2.6, -3.0,  0.1},
{ 3,2365,2364, 2276,   0.399,   4.938,  335.225, -147.7,-159.9,   0.0,  -2.6, -3.0,  0.1},
{ 3,2366,2365, 2276,   0.399,   4.938,  340.164, -150.3,-162.9,   0.1,  -2.6, -3.0,  0.1},
{ 3,2367,2366, 2276,   0.399,   4.938,  345.102, -152.9,-165.8,   0.2,  -2.6, -3.0,  0.1},
{ 3,2368,   0, 2280,   0.720,   4.643,    2.322,   -1.1,  -8.8,  -0.5,   0.7, -2.3, -3.0},
{ 3,2369,2368, 2280,   0.720,   4.643,    6.965,   -0.5, -11.0,  -3.5,   0.7, -2.3, -3.0},
{ 3,2370,2369, 2280,   0.720,   4.643,   11.608,    0.2, -13.3,  -6.5,   0.7, -2.3, -3.0},
{ 3,2371,2370, 2386,   0.720,   4.565,   16.212,    1.5, -15.8,  -8.3,   1.9, -2.7, -0.6},
{ 3,2372,2371, 2386,   0.720,   4.565,   20.777,    3.4, -18.5,  -8.9,   1.9, -2.7, -0.6},
{ 3,2373,2372, 2386,   0.720,   4.565,   25.341,    5.3, -21.3,  -9.5,   1.9, -2.7, -0.6},
{ 3,2374,2373, 2386,   0.720,   4.565,   29.906,    7.2, -24.0, -10.1,   1.9, -2.7, -0.6},
{ 3,2375,2374, 2386,   0.720,   4.565,   34.471,    9.2, -26.7, -10.7,   1.9, -2.7, -0.6},
{ 3,2376,2375, 2386,   0.720,   4.565,   39.035,   11.1, -29.5, -11.4,   1.9, -2.7, -0.6},
{ 3,2377,2376, 2386,   0.720,   4.565,   43.600,   13.0, -32.2, -12.0,   1.9, -2.7, -0.6},
{ 3,2378,2377, 2386,   0.720,   4.565,   48.165,   15.0, -34.9, -12.6,   1.9, -2.7, -0.6},
{ 3,2379,2378, 2386,   0.720,   4.565,   52.730,   16.9, -37.6, -13.2,   1.9, -2.7, -0.6},
{ 3,2380,2379, 2390,   0.720,   4.099,   57.061,   18.7, -40.6, -13.0,   1.6, -3.2,  1.0},
{ 3,2381,2380, 2390,   0.720,   4.099,   61.160,   20.2, -43.7, -12.0,   1.6, -3.2,  1.0},
{ 3,2382,2381, 2390,   0.720,   4.099,   65.258,   21.8, -46.9, -11.0,   1.6, -3.2,  1.0},
{ 3,2383,2382, 2390,   0.720,   4.099,   69.357,   23.4, -50.1, -10.0,   1.6, -3.2,  1.0},
{ 3,2384,2383, 2393,   0.400,   4.852,   73.832,   25.5, -51.5, -11.3,   2.7,  0.2, -3.5},
{ 3,2385,2384, 2393,   0.400,   4.852,   78.684,   28.2, -51.3, -14.8,   2.7,  0.2, -3.5},
{ 3,2386,2385, 2393,   0.400,   4.852,   83.536,   30.9, -51.1, -18.3,   2.7,  0.2, -3.5},
{ 3,2387,2386, 2407,   0.240,   4.837,   88.380,   33.9, -51.2, -20.6,   3.3, -0.5, -1.2},
{ 3,2388,2387, 2407,   0.240,   4.837,   93.216,   37.2, -51.7, -21.7,   3.3, -0.5, -1.2},
{ 3,2389,2388, 2407,   0.240,   4.837,   98.053,   40.5, -52.2, -22.9,   3.3, -0.5, -1.2},
{ 3,2390,2389, 2407,   0.240,   4.837,  102.889,   43.8, -52.7, -24.1,   3.3, -0.5, -1.2},
{ 3,2391,2390, 2407,   0.240,   4.837,  107.726,   47.0, -53.2, -25.2,   3.3, -0.5, -1.2},
{ 3,2392,2391, 2407,   0.240,   4.837,  112.562,   50.3, -53.7, -26.4,   3.3, -0.5, -1.2},
{ 3,2393,2392, 2407,   0.240,   4.837,  117.399,   53.6, -54.1, -27.6,   3.3, -0.5, -1.2},
{ 3,2394,2393, 2407,   0.240,   4.837,  122.235,   56.9, -54.6, -28.7,   3.3, -0.5, -1.2},
{ 3,2395,2394, 2407,   0.240,   4.837,  127.072,   60.2, -55.1, -29.9,   3.3, -0.5, -1.2},
{ 3,2396,2395, 2407,   0.240,   4.837,  131.908,   63.5, -55.6, -31.0,   3.3, -0.5, -1.2},
{ 3,2397,2396, 2407,   0.240,   4.837,  136.745,   66.8, -56.1, -32.2,   3.3, -0.5, -1.2},
{ 3,2398,2397, 2407,   0.240,   4.837,  141.581,   70.1, -56.6, -33.4,   3.3, -0.5, -1.2},
{ 3,2399,2398, 2407,   0.240,   4.837,  146.418,   73.4, -57.1, -34.5,   3.3, -0.5, -1.2},
{ 3,2400,2399, 2407,   0.240,   4.837,  151.254,   76.6, -57.5, -35.7,   3.3, -0.5, -1.2},
{ 3,2401,2400, 2407,   0.240,   4.837,  156.091,   79.9, -58.0, -36.8,   3.3, -0.5, -1.2},
{ 3,2402,2401, 2407,   0.240,   4.837,  160.927,   83.2, -58.5, -38.0,   3.3, -0.5, -1.2},
{ 3,2403,2402, 2407,   0.240,   4.837,  165.764,   86.5, -59.0, -39.2,   3.3, -0.5, -1.2},
{ 3,2404,2386, 2416,   0.560,   4.656,   88.289,   33.0, -52.3, -19.8,   1.6, -2.6,  0.3},
{ 3,2405,2404, 2416,   0.560,   4.656,   92.945,   34.6, -54.9, -19.5,   1.6, -2.6,  0.3},
{ 3,2406,2405, 2416,   0.560,   4.656,   97.600,   36.2, -57.5, -19.2,   1.6, -2.6,  0.3},
{ 3,2407,2406, 2416,   0.560,   4.656,  102.256,   37.8, -60.0, -18.9,   1.6, -2.6,  0.3},
{ 3,2408,2407, 2416,   0.560,   4.656,  106.911,   39.4, -62.6, -18.6,   1.6, -2.6,  0.3},
{ 3,2409,2408, 2416,   0.560,   4.656,  111.567,   40.9, -65.2, -18.3,   1.6, -2.6,  0.3},
{ 3,2410,2409, 2416,   0.560,   4.656,  116.223,   42.5, -67.8, -17.9,   1.6, -2.6,  0.3},
{ 3,2411,2410, 2416,   0.560,   4.656,  120.878,   44.1, -70.4, -17.6,   1.6, -2.6,  0.3},
{ 3,2412,2411, 2416,   0.560,   4.656,  125.534,   45.7, -73.0, -17.3,   1.6, -2.6,  0.3},
{ 3,2413,2412, 2416,   0.560,   4.656,  130.189,   47.3, -75.6, -17.0,   1.6, -2.6,  0.3},
{ 3,2414,2413, 2416,   0.560,   4.656,  134.845,   48.8, -78.2, -16.7,   1.6, -2.6,  0.3},
{ 3,2415,2414, 2423,   0.560,   4.686,  139.516,   50.8, -79.7, -16.8,   2.4, -0.5, -0.6},
{ 3,2416,2415, 2423,   0.560,   4.686,  144.202,   53.3, -80.2, -17.4,   2.4, -0.5, -0.6},
{ 3,2417,2416, 2423,   0.560,   4.686,  148.888,   55.7, -80.7, -18.1,   2.4, -0.5, -0.6},
{ 3,2418,2417, 2423,   0.560,   4.686,  153.575,   58.1, -81.2, -18.7,   2.4, -0.5, -0.6},
{ 3,2419,2418, 2423,   0.560,   4.686,  158.261,   60.6, -81.6, -19.3,   2.4, -0.5, -0.6},
{ 3,2420,2419, 2423,   0.560,   4.686,  162.947,   63.0, -82.1, -19.9,   2.4, -0.5, -0.6},
{ 3,2421,2420, 2423,   0.560,   4.686,  167.634,   65.5, -82.6, -20.6,   2.4, -0.5, -0.6},
{ 3,2422,2421, 2423,   0.560,   4.686,  172.320,   67.9, -83.1, -21.2,   2.4, -0.5, -0.6},
{ 3,2423,2422, 2423,   0.560,   4.686,  177.006,   70.3, -83.6, -21.8,   2.4, -0.5, -0.6},
{ 3,2424,2423, 2423,   0.560,   4.686,  181.693,   72.8, -84.0, -22.4,   2.4, -0.5, -0.6},
{ 3,2425,2424, 2427,   0.410,   4.410,  186.241,   75.8, -85.1, -22.8,   3.6, -1.6,  0.0},
{ 3,2426,2425, 2427,   0.410,   4.410,  190.651,   79.4, -86.7, -22.8,   3.6, -1.6,  0.0},
{ 3,2427,2426, 2427,   0.410,   4.410,  195.061,   83.1, -88.3, -22.8,   3.6, -1.6,  0.0},
{ 3,2428,2427, 2427,   0.410,   4.410,  199.471,   86.7, -89.9, -22.8,   3.6, -1.6,  0.0},
{ 3,2429,2428, 2427,   0.410,   4.410,  203.881,   90.4, -91.5, -22.8,   3.6, -1.6,  0.0},
{ 3,2430,2429, 2447,   0.400,   5.027,  208.599,   93.5, -92.8, -23.2,   2.7, -0.9, -0.9},
{ 3,2431,2430, 2447,   0.400,   5.027,  213.626,   96.2, -93.7, -24.1,   2.7, -0.9, -0.9},
{ 3,2432,2431, 2447,   0.400,   5.027,  218.653,   98.9, -94.6, -25.0,   2.7, -0.9, -0.9},
{ 3,2433,2432, 2447,   0.400,   5.027,  223.680,  101.6, -95.5, -26.0,   2.7, -0.9, -0.9},
{ 3,2434,2433, 2447,   0.400,   5.027,  228.707,  104.3, -96.4, -26.9,   2.7, -0.9, -0.9},
{ 3,2435,2434, 2447,   0.400,   5.027,  233.735,  107.0, -97.3, -27.8,   2.7, -0.9, -0.9},
{ 3,2436,2435, 2447,   0.400,   5.027,  238.762,  109.7, -98.2, -28.7,   2.7, -0.9, -0.9},
{ 3,2437,2436, 2447,   0.400,   5.027,  243.789,  112.4, -99.1, -29.6,   2.7, -0.9, -0.9},
{ 3,2438,2437, 2447,   0.400,   5.027,  248.816,  115.1,-100.0, -30.5,   2.7, -0.9, -0.9},
{ 3,2439,2438, 2447,   0.400,   5.027,  253.843,  117.8,-100.9, -31.5,   2.7, -0.9, -0.9},
{ 3,2440,2439, 2447,   0.400,   5.027,  258.870,  120.5,-101.8, -32.4,   2.7, -0.9, -0.9},
{ 3,2441,2440, 2447,   0.400,   5.027,  263.897,  123.2,-102.7, -33.3,   2.7, -0.9, -0.9},
{ 3,2442,2441, 2447,   0.400,   5.027,  268.924,  125.9,-103.6, -34.2,   2.7, -0.9, -0.9},
{ 3,2443,2442, 2447,   0.400,   5.027,  273.951,  128.6,-104.5, -35.1,   2.7, -0.9, -0.9},
{ 3,2444,2443, 2447,   0.400,   5.027,  278.978,  131.3,-105.4, -36.0,   2.7, -0.9, -0.9},
{ 3,2445,2429, 2449,   0.400,   4.608,  208.390,   94.0, -93.6, -23.4,   3.6, -2.5, -1.4},
{ 3,2446,2445, 2449,   0.400,   4.608,  212.998,   97.6, -96.1, -24.8,   3.6, -2.5, -1.4},
{ 3,2447,2446, 2467,   0.247,   5.007,  217.806,  100.4, -97.8, -25.4,   2.1, -1.0,  0.1},
{ 3,2448,2447, 2467,   0.247,   5.007,  222.813,  102.5, -98.8, -25.3,   2.1, -1.0,  0.1},
{ 3,2449,2448, 2467,   0.247,   5.007,  227.819,  104.6, -99.7, -25.2,   2.1, -1.0,  0.1},
{ 3,2450,2449, 2467,   0.247,   5.007,  232.826,  106.6,-100.7, -25.0,   2.1, -1.0,  0.1},
{ 3,2451,2450, 2467,   0.247,   5.007,  237.833,  108.7,-101.6, -24.9,   2.1, -1.0,  0.1},
{ 3,2452,2451, 2467,   0.247,   5.007,  242.840,  110.8,-102.6, -24.8,   2.1, -1.0,  0.1},
{ 3,2453,2452, 2467,   0.247,   5.007,  247.847,  112.9,-103.6, -24.6,   2.1, -1.0,  0.1},
{ 3,2454,2453, 2467,   0.247,   5.007,  252.853,  115.0,-104.5, -24.5,   2.1, -1.0,  0.1},
{ 3,2455,2454, 2467,   0.247,   5.007,  257.860,  117.0,-105.5, -24.4,   2.1, -1.0,  0.1},
{ 3,2456,2455, 2467,   0.247,   5.007,  262.867,  119.1,-106.5, -24.2,   2.1, -1.0,  0.1},
{ 3,2457,2456, 2467,   0.247,   5.007,  267.874,  121.2,-107.4, -24.1,   2.1, -1.0,  0.1},
{ 3,2458,2457, 2467,   0.247,   5.007,  272.881,  123.3,-108.4, -24.0,   2.1, -1.0,  0.1},
{ 3,2459,2458, 2467,   0.247,   5.007,  277.887,  125.3,-109.3, -23.8,   2.1, -1.0,  0.1},
{ 3,2460,2459, 2467,   0.247,   5.007,  282.894,  127.4,-110.3, -23.7,   2.1, -1.0,  0.1},
{ 3,2461,2460, 2467,   0.247,   5.007,  287.901,  129.5,-111.3, -23.5,   2.1, -1.0,  0.1},
{ 3,2462,2461, 2467,   0.247,   5.007,  292.908,  131.6,-112.2, -23.4,   2.1, -1.0,  0.1},
{ 3,2463,2462, 2467,   0.247,   5.007,  297.914,  133.6,-113.2, -23.3,   2.1, -1.0,  0.1},
{ 3,2464,2463, 2467,   0.247,   5.007,  302.921,  135.7,-114.2, -23.1,   2.1, -1.0,  0.1},
{ 3,2465,2464, 2467,   0.247,   5.007,  307.928,  137.8,-115.1, -23.0,   2.1, -1.0,  0.1},
{ 3,2466,2465, 2467,   0.247,   5.007,  312.935,  139.9,-116.1, -22.9,   2.1, -1.0,  0.1},
{ 3,2467,2466, 2467,   0.247,   5.007,  317.942,  142.0,-117.0, -22.7,   2.1, -1.0,  0.1},
{ 3,2468,2467, 2467,   0.247,   5.007,  322.948,  144.0,-118.0, -22.6,   2.1, -1.0,  0.1},
{ 3,2469,2468, 2467,   0.247,   5.007,  327.955,  146.1,-119.0, -22.5,   2.1, -1.0,  0.1},
{ 3,2470,2469, 2467,   0.247,   5.007,  332.962,  148.2,-119.9, -22.3,   2.1, -1.0,  0.1},
{ 3,2471,2470, 2467,   0.247,   5.007,  337.969,  150.3,-120.9, -22.2,   2.1, -1.0,  0.1},
{ 3,2472,2471, 2467,   0.247,   5.007,  342.975,  152.3,-121.9, -22.1,   2.1, -1.0,  0.1},
{ 3,2473,2414, 2472,   0.560,   4.670,  139.508,   50.2, -81.1, -15.7,   1.2, -3.1,  1.5},
{ 3,2474,2473, 2472,   0.560,   4.670,  144.178,   51.4, -84.2, -14.2,   1.2, -3.1,  1.5},
{ 3,2475,2474, 2472,   0.560,   4.670,  148.849,   52.6, -87.3, -12.7,   1.2, -3.1,  1.5},
{ 3,2476,2475, 2472,   0.560,   4.670,  153.519,   53.9, -90.5, -11.1,   1.2, -3.1,  1.5},
{ 3,2477,2476, 2472,   0.560,   4.670,  158.189,   55.1, -93.6,  -9.6,   1.2, -3.1,  1.5},
{ 3,2478,2477, 2472,   0.560,   4.670,  162.860,   56.3, -96.7,  -8.1,   1.2, -3.1,  1.5},
{ 3,2479,2478, 2472,   0.560,   4.670,  167.530,   57.5, -99.9,  -6.5,   1.2, -3.1,  1.5},
{ 3,2480,2479, 2494,   0.405,   4.992,  172.361,   59.3,-103.3,  -6.1,   2.5, -3.7, -0.6},
{ 3,2481,2480, 2494,   0.405,   4.992,  177.353,   61.8,-107.0,  -6.7,   2.5, -3.7, -0.6},
{ 3,2482,2481, 2494,   0.405,   4.992,  182.345,   64.3,-110.8,  -7.3,   2.5, -3.7, -0.6},
{ 3,2483,2482, 2494,   0.405,   4.992,  187.336,   66.7,-114.5,  -8.0,   2.5, -3.7, -0.6},
{ 3,2484,2483, 2494,   0.405,   4.992,  192.328,   69.2,-118.2,  -8.6,   2.5, -3.7, -0.6},
{ 3,2485,2484, 2494,   0.405,   4.992,  197.320,   71.7,-121.9,  -9.2,   2.5, -3.7, -0.6},
{ 3,2486,2485, 2494,   0.405,   4.992,  202.311,   74.1,-125.7,  -9.8,   2.5, -3.7, -0.6},
{ 3,2487,2486, 2494,   0.405,   4.992,  207.303,   76.6,-129.4, -10.5,   2.5, -3.7, -0.6},
{ 3,2488,2487, 2494,   0.405,   4.992,  212.294,   79.1,-133.1, -11.1,   2.5, -3.7, -0.6},
{ 3,2489,2488, 2494,   0.405,   4.992,  217.286,   81.5,-136.9, -11.7,   2.5, -3.7, -0.6},
{ 3,2490,2489, 2494,   0.405,   4.992,  222.278,   84.0,-140.6, -12.4,   2.5, -3.7, -0.6},
{ 3,2491,2490, 2494,   0.405,   4.992,  227.269,   86.4,-144.3, -13.0,   2.5, -3.7, -0.6},
{ 3,2492,2491, 2494,   0.405,   4.992,  232.261,   88.9,-148.0, -13.6,   2.5, -3.7, -0.6},
{ 3,2493,2492, 2494,   0.405,   4.992,  237.253,   91.4,-151.8, -14.2,   2.5, -3.7, -0.6},
{ 3,2494,2493, 2494,   0.405,   4.992,  242.244,   93.8,-155.5, -14.9,   2.5, -3.7, -0.6},
{ 3,2495,2494, 2494,   0.405,   4.992,  247.236,   96.3,-159.2, -15.5,   2.5, -3.7, -0.6},
{ 3,2496,2495, 2494,   0.405,   4.992,  252.228,   98.8,-162.9, -16.1,   2.5, -3.7, -0.6},
{ 3,2497,2496, 2494,   0.405,   4.992,  257.219,  101.2,-166.7, -16.8,   2.5, -3.7, -0.6},
{ 3,2498,2497, 2494,   0.405,   4.992,  262.211,  103.7,-170.4, -17.4,   2.5, -3.7, -0.6},
{ 3,2499,2498, 2494,   0.405,   4.992,  267.203,  106.2,-174.1, -18.0,   2.5, -3.7, -0.6},
{ 3,2500,2499, 2494,   0.405,   4.992,  272.194,  108.6,-177.8, -18.6,   2.5, -3.7, -0.6},
{ 3,2501,2500, 2494,   0.405,   4.992,  277.186,  111.1,-181.6, -19.3,   2.5, -3.7, -0.6},
{ 3,2502,2501, 2494,   0.405,   4.992,  282.178,  113.6,-185.3, -19.9,   2.5, -3.7, -0.6},
{ 3,2503,2502, 2494,   0.405,   4.992,  287.169,  116.0,-189.0, -20.5,   2.5, -3.7, -0.6},
{ 3,2504,2503, 2494,   0.405,   4.992,  292.161,  118.5,-192.7, -21.2,   2.5, -3.7, -0.6},
{ 3,2505,2504, 2494,   0.405,   4.992,  297.153,  121.0,-196.5, -21.8,   2.5, -3.7, -0.6},
{ 3,2506,2505, 2494,   0.405,   4.992,  302.144,  123.4,-200.2, -22.4,   2.5, -3.7, -0.6},
{ 3,2507,2506, 2494,   0.405,   4.992,  307.136,  125.9,-203.9, -23.0,   2.5, -3.7, -0.6},
{ 3,2508,2507, 2494,   0.405,   4.992,  312.127,  128.4,-207.6, -23.7,   2.5, -3.7, -0.6},
{ 3,2509,2508, 2494,   0.405,   4.992,  317.119,  130.8,-211.4, -24.3,   2.5, -3.7, -0.6},
{ 3,2510,2509, 2494,   0.405,   4.992,  322.111,  133.3,-215.1, -24.9,   2.5, -3.7, -0.6},
{ 3,2511,2383, 2505,   0.569,   4.839,   73.826,   24.6, -53.4,  -9.3,   0.8, -3.5,  0.5},
{ 3,2512,2511, 2505,   0.569,   4.839,   78.665,   25.4, -56.9,  -8.8,   0.8, -3.5,  0.5},
{ 3,2513,2512, 2505,   0.569,   4.839,   83.504,   26.2, -60.5,  -8.3,   0.8, -3.5,  0.5},
{ 3,2514,2513, 2505,   0.569,   4.839,   88.344,   27.0, -64.0,  -7.8,   0.8, -3.5,  0.5},
{ 3,2515,2514, 2505,   0.569,   4.839,   93.183,   27.7, -67.6,  -7.3,   0.8, -3.5,  0.5},
{ 3,2516,2515, 2505,   0.569,   4.839,   98.022,   28.5, -71.1,  -6.8,   0.8, -3.5,  0.5},
{ 3,2517,2516, 2505,   0.569,   4.839,  102.862,   29.3, -74.7,  -6.3,   0.8, -3.5,  0.5},
{ 3,2518,2517, 2510,   0.400,   4.480,  107.521,   31.1, -78.0,  -6.3,   2.8, -3.0, -0.6},
{ 3,2519,2518, 2510,   0.400,   4.480,  112.001,   33.9, -81.0,  -7.0,   2.8, -3.0, -0.6},
{ 3,2520,2519, 2510,   0.400,   4.480,  116.481,   36.7, -84.0,  -7.6,   2.8, -3.0, -0.6},
{ 3,2521,2520, 2510,   0.400,   4.480,  120.961,   39.5, -87.1,  -8.3,   2.8, -3.0, -0.6},
{ 3,2522,2521, 2510,   0.400,   4.480,  125.440,   42.3, -90.1,  -8.9,   2.8, -3.0, -0.6},
{ 3,2523,2522, 2537,   0.244,   4.895,  130.128,   44.4, -93.1,  -9.5,   1.3, -2.9, -0.5},
{ 3,2524,2523, 2537,   0.244,   4.895,  135.023,   45.7, -96.0, -10.1,   1.3, -2.9, -0.5},
{ 3,2525,2524, 2537,   0.244,   4.895,  139.917,   47.1, -98.8, -10.6,   1.3, -2.9, -0.5},
{ 3,2526,2525, 2537,   0.244,   4.895,  144.812,   48.4,-101.7, -11.1,   1.3, -2.9, -0.5},
{ 3,2527,2526, 2537,   0.244,   4.895,  149.707,   49.8,-104.6, -11.7,   1.3, -2.9, -0.5},
{ 3,2528,2527, 2537,   0.244,   4.895,  154.602,   51.1,-107.4, -12.2,   1.3, -2.9, -0.5},
{ 3,2529,2528, 2537,   0.244,   4.895,  159.496,   52.5,-110.3, -12.7,   1.3, -2.9, -0.5},
{ 3,2530,2529, 2537,   0.244,   4.895,  164.391,   53.8,-113.2, -13.3,   1.3, -2.9, -0.5},
{ 3,2531,2530, 2537,   0.244,   4.895,  169.286,   55.1,-116.0, -13.8,   1.3, -2.9, -0.5},
{ 3,2532,2531, 2537,   0.244,   4.895,  174.181,   56.5,-118.9, -14.3,   1.3, -2.9, -0.5},
{ 3,2533,2532, 2537,   0.244,   4.895,  179.075,   57.8,-121.8, -14.9,   1.3, -2.9, -0.5},
{ 3,2534,2533, 2537,   0.244,   4.895,  183.970,   59.2,-124.6, -15.4,   1.3, -2.9, -0.5},
{ 3,2535,2534, 2537,   0.244,   4.895,  188.865,   60.5,-127.5, -16.0,   1.3, -2.9, -0.5},
{ 3,2536,2535, 2537,   0.244,   4.895,  193.760,   61.9,-130.3, -16.5,   1.3, -2.9, -0.5},
{ 3,2537,2536, 2537,   0.244,   4.895,  198.654,   63.2,-133.2, -17.0,   1.3, -2.9, -0.5},
{ 3,2538,2537, 2537,   0.244,   4.895,  203.549,   64.6,-136.1, -17.6,   1.3, -2.9, -0.5},
{ 3,2539,2538, 2537,   0.244,   4.895,  208.444,   65.9,-138.9, -18.1,   1.3, -2.9, -0.5},
{ 3,2540,2539, 2537,   0.244,   4.895,  213.339,   67.3,-141.8, -18.6,   1.3, -2.9, -0.5},
{ 3,2541,2540, 2537,   0.244,   4.895,  218.233,   68.6,-144.7, -19.2,   1.3, -2.9, -0.5},
{ 3,2542,2541, 2537,   0.244,   4.895,  223.128,   70.0,-147.5, -19.7,   1.3, -2.9, -0.5},
{ 3,2543,2542, 2537,   0.244,   4.895,  228.023,   71.3,-150.4, -20.3,   1.3, -2.9, -0.5},
{ 3,2544,2543, 2537,   0.244,   4.895,  232.918,   72.7,-153.3, -20.8,   1.3, -2.9, -0.5},
{ 3,2545,2544, 2537,   0.244,   4.895,  237.813,   74.0,-156.1, -21.3,   1.3, -2.9, -0.5},
{ 3,2546,2545, 2537,   0.244,   4.895,  242.707,   75.4,-159.0, -21.9,   1.3, -2.9, -0.5},
{ 3,2547,2546, 2537,   0.244,   4.895,  247.602,   76.7,-161.9, -22.4,   1.3, -2.9, -0.5},
{ 3,2548,2547, 2537,   0.244,   4.895,  252.497,   78.1,-164.7, -22.9,   1.3, -2.9, -0.5},
{ 3,2549,2548, 2537,   0.244,   4.895,  257.392,   79.4,-167.6, -23.5,   1.3, -2.9, -0.5},
{ 3,2550,2549, 2537,   0.244,   4.895,  262.286,   80.8,-170.5, -24.0,   1.3, -2.9, -0.5},
{ 3,2551,2550, 2537,   0.244,   4.895,  267.181,   82.1,-173.3, -24.5,   1.3, -2.9, -0.5},
{ 3,2552,2551, 2537,   0.244,   4.895,  272.076,   83.5,-176.2, -25.1,   1.3, -2.9, -0.5},
{ 3,2553,2552, 2537,   0.244,   4.895,  276.971,   84.8,-179.1, -25.6,   1.3, -2.9, -0.5},
{ 3,2554,2553, 2537,   0.244,   4.895,  281.865,   86.1,-181.9, -26.2,   1.3, -2.9, -0.5},
{ 3,2555,2554, 2537,   0.244,   4.895,  286.760,   87.5,-184.8, -26.7,   1.3, -2.9, -0.5},
{ 3,2556,2555, 2537,   0.244,   4.895,  291.655,   88.8,-187.7, -27.2,   1.3, -2.9, -0.5},
{ 3,2557,2556, 2537,   0.244,   4.895,  296.550,   90.2,-190.5, -27.8,   1.3, -2.9, -0.5},
{ 3,2558,2557, 2537,   0.244,   4.895,  301.444,   91.5,-193.4, -28.3,   1.3, -2.9, -0.5},
{ 3,2559,2558, 2537,   0.244,   4.895,  306.339,   92.9,-196.2, -28.8,   1.3, -2.9, -0.5},
{ 3,2560,2559, 2537,   0.244,   4.895,  311.234,   94.2,-199.1, -29.4,   1.3, -2.9, -0.5},
{ 3,2561,2560, 2537,   0.244,   4.895,  316.129,   95.6,-202.0, -29.9,   1.3, -2.9, -0.5},
{ 3,2562,2561, 2537,   0.244,   4.895,  321.023,   96.9,-204.8, -30.4,   1.3, -2.9, -0.5},
{ 3,2563,2562, 2537,   0.244,   4.895,  325.918,   98.3,-207.7, -31.0,   1.3, -2.9, -0.5},
{ 3,2564,2517, 2541,   0.443,   3.893,  107.228,   30.3, -77.8,  -5.4,   1.2, -2.8,  1.3},
{ 3,2565,2564, 2541,   0.443,   3.893,  111.121,   31.5, -80.7,  -4.1,   1.2, -2.8,  1.3},
{ 3,2566,2565, 2575,   0.400,   4.893,  115.514,   32.1, -84.1,  -3.3,  -0.0, -4.0,  0.5},
{ 3,2567,2566, 2575,   0.400,   4.893,  120.407,   32.0, -88.1,  -2.8,  -0.0, -4.0,  0.5},
{ 3,2568,2567, 2575,   0.400,   4.893,  125.300,   32.0, -92.2,  -2.4,  -0.0, -4.0,  0.5},
{ 3,2569,2568, 2575,   0.400,   4.893,  130.193,   32.0, -96.2,  -1.9,  -0.0, -4.0,  0.5},
{ 3,2570,2569, 2575,   0.400,   4.893,  135.086,   32.0,-100.2,  -1.4,  -0.0, -4.0,  0.5},
{ 3,2571,2570, 2575,   0.400,   4.893,  139.979,   32.0,-104.3,  -1.0,  -0.0, -4.0,  0.5},
{ 3,2572,2571, 2575,   0.400,   4.893,  144.872,   32.0,-108.3,  -0.5,  -0.0, -4.0,  0.5},
{ 3,2573,2572, 2575,   0.400,   4.893,  149.765,   32.0,-112.3,  -0.1,  -0.0, -4.0,  0.5},
{ 3,2574,2573, 2575,   0.400,   4.893,  154.657,   32.0,-116.3,   0.4,  -0.0, -4.0,  0.5},
{ 3,2575,2574, 2575,   0.400,   4.893,  159.550,   32.0,-120.4,   0.9,  -0.0, -4.0,  0.5},
{ 3,2576,2575, 2575,   0.400,   4.893,  164.443,   32.0,-124.4,   1.3,  -0.0, -4.0,  0.5},
{ 3,2577,2576, 2575,   0.400,   4.893,  169.336,   32.0,-128.4,   1.8,  -0.0, -4.0,  0.5},
{ 3,2578,2577, 2575,   0.400,   4.893,  174.229,   32.0,-132.5,   2.2,  -0.0, -4.0,  0.5},
{ 3,2579,2578, 2575,   0.400,   4.893,  179.122,   32.0,-136.5,   2.7,  -0.0, -4.0,  0.5},
{ 3,2580,2579, 2575,   0.400,   4.893,  184.015,   32.0,-140.5,   3.1,  -0.0, -4.0,  0.5},
{ 3,2581,2580, 2575,   0.400,   4.893,  188.908,   32.0,-144.6,   3.6,  -0.0, -4.0,  0.5},
{ 3,2582,2581, 2575,   0.400,   4.893,  193.801,   32.0,-148.6,   4.1,  -0.0, -4.0,  0.5},
{ 3,2583,2582, 2575,   0.400,   4.893,  198.693,   31.9,-152.6,   4.5,  -0.0, -4.0,  0.5},
{ 3,2584,2583, 2575,   0.400,   4.893,  203.586,   31.9,-156.7,   5.0,  -0.0, -4.0,  0.5},
{ 3,2585,2584, 2575,   0.400,   4.893,  208.479,   31.9,-160.7,   5.4,  -0.0, -4.0,  0.5},
{ 3,2586,2585, 2575,   0.400,   4.893,  213.372,   31.9,-164.7,   5.9,  -0.0, -4.0,  0.5},
{ 3,2587,2586, 2575,   0.400,   4.893,  218.265,   31.9,-168.7,   6.4,  -0.0, -4.0,  0.5},
{ 3,2588,2587, 2575,   0.400,   4.893,  223.158,   31.9,-172.8,   6.8,  -0.0, -4.0,  0.5},
{ 3,2589,2588, 2575,   0.400,   4.893,  228.051,   31.9,-176.8,   7.3,  -0.0, -4.0,  0.5},
{ 3,2590,2589, 2575,   0.400,   4.893,  232.944,   31.9,-180.8,   7.7,  -0.0, -4.0,  0.5},
{ 3,2591,2590, 2575,   0.400,   4.893,  237.837,   31.9,-184.9,   8.2,  -0.0, -4.0,  0.5},
{ 3,2592,2591, 2575,   0.400,   4.893,  242.729,   31.9,-188.9,   8.6,  -0.0, -4.0,  0.5},
{ 3,2593,2592, 2575,   0.400,   4.893,  247.622,   31.9,-192.9,   9.1,  -0.0, -4.0,  0.5},
{ 3,2594,2593, 2575,   0.400,   4.893,  252.515,   31.9,-197.0,   9.6,  -0.0, -4.0,  0.5},
{ 3,2595,2594, 2575,   0.400,   4.893,  257.408,   31.9,-201.0,  10.0,  -0.0, -4.0,  0.5},
{ 3,2596,2595, 2575,   0.400,   4.893,  262.301,   31.9,-205.0,  10.5,  -0.0, -4.0,  0.5},
{ 3,2597,2596, 2575,   0.400,   4.893,  267.194,   31.9,-209.1,  10.9,  -0.0, -4.0,  0.5},
{ 3,2598,2597, 2575,   0.400,   4.893,  272.087,   31.9,-213.1,  11.4,  -0.0, -4.0,  0.5},
{ 3,2599,2598, 2575,   0.400,   4.893,  276.980,   31.9,-217.1,  11.9,  -0.0, -4.0,  0.5},
{ 3,2600,2599, 2575,   0.400,   4.893,  281.873,   31.8,-221.1,  12.3,  -0.0, -4.0,  0.5},
{ 3,2601,2600, 2575,   0.400,   4.893,  286.765,   31.8,-225.2,  12.8,  -0.0, -4.0,  0.5},
{ 3,2602,2379, 2578,   0.560,   4.182,   57.103,   19.5, -40.0, -13.0,   3.3, -1.9,  1.1},
{ 3,2603,2602, 2578,   0.560,   4.182,   61.286,   22.8, -41.8, -11.9,   3.3, -1.9,  1.1},
{ 3,2604,2603, 2578,   0.560,   4.182,   65.468,   26.2, -43.7, -10.8,   3.3, -1.9,  1.1},
{ 3,2605,2604, 2586,   0.426,   4.712,   69.915,   29.2, -45.4, -11.2,   2.7, -1.4, -1.9},
{ 3,2606,2605, 2586,   0.426,   4.712,   74.627,   31.9, -46.8, -13.0,   2.7, -1.4, -1.9},
{ 3,2607,2606, 2586,   0.426,   4.712,   79.339,   34.6, -48.2, -14.9,   2.7, -1.4, -1.9},
{ 3,2608,2607, 2586,   0.426,   4.712,   84.051,   37.3, -49.6, -16.7,   2.7, -1.4, -1.9},
{ 3,2609,2608, 2586,   0.426,   4.712,   88.763,   40.0, -51.0, -18.6,   2.7, -1.4, -1.9},
{ 3,2610,2609, 2586,   0.426,   4.712,   93.475,   42.7, -52.5, -20.4,   2.7, -1.4, -1.9},
{ 3,2611,2610, 2586,   0.426,   4.712,   98.187,   45.4, -53.9, -22.3,   2.7, -1.4, -1.9},
{ 3,2612,2611, 2586,   0.426,   4.712,  102.899,   48.1, -55.3, -24.1,   2.7, -1.4, -1.9},
{ 3,2613,2612, 2586,   0.426,   4.712,  107.612,   50.9, -56.7, -26.0,   2.7, -1.4, -1.9},
{ 3,2614,2613, 2586,   0.426,   4.712,  112.324,   53.6, -58.1, -27.8,   2.7, -1.4, -1.9},
{ 3,2615,2614, 2590,   0.552,   4.684,  117.022,   55.4, -58.9, -28.4,   0.9, -0.3,  0.8},
{ 3,2616,2615, 2590,   0.552,   4.684,  121.706,   56.2, -59.2, -27.6,   0.9, -0.3,  0.8},
{ 3,2617,2616, 2590,   0.552,   4.684,  126.391,   57.1, -59.5, -26.8,   0.9, -0.3,  0.8},
{ 3,2618,2617, 2590,   0.552,   4.684,  131.075,   58.0, -59.8, -26.1,   0.9, -0.3,  0.8},
{ 3,2619,2618, 2590,   0.552,   4.684,  135.760,   58.9, -60.0, -25.3,   0.9, -0.3,  0.8},
{ 3,2620,2619, 2590,   0.552,   4.684,  140.444,   59.8, -60.3, -24.5,   0.9, -0.3,  0.8},
{ 3,2621,2620, 2590,   0.552,   4.684,  145.129,   60.6, -60.6, -23.7,   0.9, -0.3,  0.8},
{ 3,2622,2621, 2590,   0.552,   4.684,  149.813,   61.5, -60.8, -23.0,   0.9, -0.3,  0.8},
{ 3,2623,2622, 2590,   0.552,   4.684,  154.498,   62.4, -61.1, -22.2,   0.9, -0.3,  0.8},
{ 3,2624,2623, 2590,   0.552,   4.684,  159.182,   63.3, -61.4, -21.4,   0.9, -0.3,  0.8},
{ 3,2625,2624, 2590,   0.552,   4.684,  163.867,   64.2, -61.6, -20.7,   0.9, -0.3,  0.8},
{ 3,2626,2625, 2590,   0.552,   4.684,  168.551,   65.0, -61.9, -19.9,   0.9, -0.3,  0.8},
{ 3,2627,2626, 2593,   0.560,   4.561,  173.174,   66.8, -62.8, -20.5,   2.6, -1.6, -2.0},
{ 3,2628,2627, 2593,   0.560,   4.561,  177.735,   69.3, -64.4, -22.4,   2.6, -1.6, -2.0},
{ 3,2629,2628, 2593,   0.560,   4.561,  182.296,   71.9, -66.1, -24.4,   2.6, -1.6, -2.0},
{ 3,2630,2629, 2593,   0.560,   4.561,  186.857,   74.5, -67.7, -26.3,   2.6, -1.6, -2.0},
{ 3,2631,2630, 2593,   0.560,   4.561,  191.418,   77.1, -69.3, -28.3,   2.6, -1.6, -2.0},
{ 3,2632,2631, 2619,   0.400,   4.966,  196.182,   79.5, -70.8, -30.0,   2.3, -1.5, -1.5},
{ 3,2633,2632, 2619,   0.400,   4.966,  201.148,   81.9, -72.3, -31.5,   2.3, -1.5, -1.5},
{ 3,2634,2633, 2619,   0.400,   4.966,  206.114,   84.2, -73.8, -33.0,   2.3, -1.5, -1.5},
{ 3,2635,2634, 2619,   0.400,   4.966,  211.080,   86.5, -75.3, -34.6,   2.3, -1.5, -1.5},
{ 3,2636,2635, 2619,   0.400,   4.966,  216.047,   88.9, -76.8, -36.1,   2.3, -1.5, -1.5},
{ 3,2637,2636, 2619,   0.400,   4.966,  221.013,   91.2, -78.3, -37.6,   2.3, -1.5, -1.5},
{ 3,2638,2637, 2619,   0.400,   4.966,  225.979,   93.5, -79.8, -39.1,   2.3, -1.5, -1.5},
{ 3,2639,2638, 2619,   0.400,   4.966,  230.945,   95.9, -81.3, -40.7,   2.3, -1.5, -1.5},
{ 3,2640,2639, 2619,   0.400,   4.966,  235.911,   98.2, -82.8, -42.2,   2.3, -1.5, -1.5},
{ 3,2641,2640, 2619,   0.400,   4.966,  240.877,  100.5, -84.3, -43.7,   2.3, -1.5, -1.5},
{ 3,2642,2641, 2619,   0.400,   4.966,  245.844,  102.9, -85.8, -45.2,   2.3, -1.5, -1.5},
{ 3,2643,2642, 2619,   0.400,   4.966,  250.810,  105.2, -87.3, -46.7,   2.3, -1.5, -1.5},
{ 3,2644,2643, 2619,   0.400,   4.966,  255.776,  107.5, -88.8, -48.3,   2.3, -1.5, -1.5},
{ 3,2645,2644, 2619,   0.400,   4.966,  260.742,  109.9, -90.2, -49.8,   2.3, -1.5, -1.5},
{ 3,2646,2645, 2619,   0.400,   4.966,  265.708,  112.2, -91.7, -51.3,   2.3, -1.5, -1.5},
{ 3,2647,2646, 2619,   0.400,   4.966,  270.675,  114.5, -93.2, -52.8,   2.3, -1.5, -1.5},
{ 3,2648,2647, 2619,   0.400,   4.966,  275.641,  116.9, -94.7, -54.3,   2.3, -1.5, -1.5},
{ 3,2649,2648, 2619,   0.400,   4.966,  280.607,  119.2, -96.2, -55.9,   2.3, -1.5, -1.5},
{ 3,2650,2649, 2619,   0.400,   4.966,  285.573,  121.6, -97.7, -57.4,   2.3, -1.5, -1.5},
{ 3,2651,2650, 2619,   0.400,   4.966,  290.539,  123.9, -99.2, -58.9,   2.3, -1.5, -1.5},
{ 3,2652,2651, 2619,   0.400,   4.966,  295.505,  126.2,-100.7, -60.4,   2.3, -1.5, -1.5},
{ 3,2653,2652, 2619,   0.400,   4.966,  300.472,  128.6,-102.2, -61.9,   2.3, -1.5, -1.5},
{ 3,2654,2653, 2619,   0.400,   4.966,  305.438,  130.9,-103.7, -63.5,   2.3, -1.5, -1.5},
{ 3,2655,2654, 2619,   0.400,   4.966,  310.404,  133.2,-105.2, -65.0,   2.3, -1.5, -1.5},
{ 3,2656,2655, 2619,   0.400,   4.966,  315.370,  135.6,-106.7, -66.5,   2.3, -1.5, -1.5},
{ 3,2657,2656, 2619,   0.400,   4.966,  320.336,  137.9,-108.2, -68.0,   2.3, -1.5, -1.5},
{ 3,2658,2657, 2619,   0.400,   4.966,  325.303,  140.2,-109.7, -69.5,   2.3, -1.5, -1.5},
{ 3,2659,2658, 2619,   0.400,   4.966,  330.269,  142.6,-111.2, -71.0,   2.3, -1.5, -1.5},
{ 3,2660,2659, 2619,   0.400,   4.966,  335.235,  144.9,-112.7, -72.6,   2.3, -1.5, -1.5},
{ 3,2661,2660, 2619,   0.400,   4.966,  340.201,  147.2,-114.1, -74.1,   2.3, -1.5, -1.5},
{ 3,2662,2661, 2619,   0.400,   4.966,  345.167,  149.6,-115.6, -75.6,   2.3, -1.5, -1.5},
{ 3,2663,2662, 2619,   0.400,   4.966,  350.134,  151.9,-117.1, -77.1,   2.3, -1.5, -1.5},
{ 3,2664,2663, 2619,   0.400,   4.966,  355.100,  154.2,-118.6, -78.6,   2.3, -1.5, -1.5},
{ 3,2665,2664, 2619,   0.400,   4.966,  360.066,  156.6,-120.1, -80.2,   2.3, -1.5, -1.5},
{ 3,2666,2665, 2619,   0.400,   4.966,  365.032,  158.9,-121.6, -81.7,   2.3, -1.5, -1.5},
{ 3,2667,2666, 2619,   0.400,   4.966,  369.998,  161.3,-123.1, -83.2,   2.3, -1.5, -1.5},
{ 3,2668,2667, 2619,   0.400,   4.966,  374.964,  163.6,-124.6, -84.7,   2.3, -1.5, -1.5},
{ 3,2669,2668, 2619,   0.400,   4.966,  379.931,  165.9,-126.1, -86.2,   2.3, -1.5, -1.5},
{ 3,2670,2669, 2619,   0.400,   4.966,  384.897,  168.3,-127.6, -87.8,   2.3, -1.5, -1.5},
{ 3,2671,2670, 2619,   0.400,   4.966,  389.863,  170.6,-129.1, -89.3,   2.3, -1.5, -1.5},
{ 3,2672,2671, 2619,   0.400,   4.966,  394.829,  172.9,-130.6, -90.8,   2.3, -1.5, -1.5},
{ 3,2673,2672, 2619,   0.400,   4.966,  399.795,  175.3,-132.1, -92.3,   2.3, -1.5, -1.5},
{ 3,2674,2673, 2619,   0.400,   4.966,  404.762,  177.6,-133.6, -93.8,   2.3, -1.5, -1.5},
{ 3,2675,2674, 2619,   0.400,   4.966,  409.728,  179.9,-135.1, -95.4,   2.3, -1.5, -1.5},
{ 3,2676,2675, 2619,   0.400,   4.966,  414.694,  182.3,-136.6, -96.9,   2.3, -1.5, -1.5},
{ 3,2677,2676, 2619,   0.400,   4.966,  419.660,  184.6,-138.0, -98.4,   2.3, -1.5, -1.5},
{ 3,2678,2677, 2619,   0.400,   4.966,  424.626,  186.9,-139.5, -99.9,   2.3, -1.5, -1.5},
{ 3,2679,2678, 2619,   0.400,   4.966,  429.592,  189.3,-141.0,-101.4,   2.3, -1.5, -1.5},
{ 3,2680,2679, 2619,   0.400,   4.966,  434.559,  191.6,-142.5,-103.0,   2.3, -1.5, -1.5},
{ 3,2681,2680, 2619,   0.400,   4.966,  439.525,  193.9,-144.0,-104.5,   2.3, -1.5, -1.5},
{ 3,2682,2631, 2621,   0.483,   2.938,  195.168,   78.1, -70.8, -30.4,  -0.4, -1.5, -2.4},
{ 3,2683,2682, 2621,   0.483,   2.938,  198.105,   77.7, -72.4, -32.8,  -0.4, -1.5, -2.4},
{ 3,2684,2683, 2636,   0.400,   4.965,  202.057,   78.9, -73.8, -35.3,   2.7, -1.4, -2.6},
{ 3,2685,2684, 2636,   0.400,   4.965,  207.022,   81.5, -75.2, -38.0,   2.7, -1.4, -2.6},
{ 3,2686,2685, 2636,   0.400,   4.965,  211.987,   84.2, -76.6, -40.6,   2.7, -1.4, -2.6},
{ 3,2687,2686, 2636,   0.400,   4.965,  216.952,   86.8, -77.9, -43.2,   2.7, -1.4, -2.6},
{ 3,2688,2687, 2636,   0.400,   4.965,  221.917,   89.5, -79.3, -45.9,   2.7, -1.4, -2.6},
{ 3,2689,2688, 2636,   0.400,   4.965,  226.883,   92.1, -80.6, -48.5,   2.7, -1.4, -2.6},
{ 3,2690,2689, 2636,   0.400,   4.965,  231.848,   94.8, -82.0, -51.2,   2.7, -1.4, -2.6},
{ 3,2691,2690, 2636,   0.400,   4.965,  236.813,   97.4, -83.4, -53.8,   2.7, -1.4, -2.6},
{ 3,2692,2691, 2636,   0.400,   4.965,  241.778,  100.1, -84.7, -56.4,   2.7, -1.4, -2.6},
{ 3,2693,2692, 2636,   0.400,   4.965,  246.743,  102.7, -86.1, -59.1,   2.7, -1.4, -2.6},
{ 3,2694,2693, 2636,   0.400,   4.965,  251.708,  105.4, -87.4, -61.7,   2.7, -1.4, -2.6},
{ 3,2695,2694, 2636,   0.400,   4.965,  256.673,  108.0, -88.8, -64.4,   2.7, -1.4, -2.6},
{ 3,2696,2695, 2636,   0.400,   4.965,  261.638,  110.7, -90.1, -67.0,   2.7, -1.4, -2.6},
{ 3,2697,2696, 2636,   0.400,   4.965,  266.603,  113.3, -91.5, -69.6,   2.7, -1.4, -2.6},
{ 3,2698,2697, 2636,   0.400,   4.965,  271.569,  116.0, -92.9, -72.3,   2.7, -1.4, -2.6},
{ 3,2699,2698, 2636,   0.400,   4.965,  276.534,  118.6, -94.2, -74.9,   2.7, -1.4, -2.6},
{ 3,2700,2699, 2650,   0.250,   4.814,  281.423,  121.5, -95.8, -77.1,   3.2, -1.8, -1.8},
{ 3,2701,2700, 2650,   0.250,   4.814,  286.238,  124.7, -97.6, -78.9,   3.2, -1.8, -1.8},
{ 3,2702,2701, 2650,   0.250,   4.814,  291.052,  127.9, -99.4, -80.7,   3.2, -1.8, -1.8},
{ 3,2703,2702, 2650,   0.250,   4.814,  295.867,  131.1,-101.3, -82.4,   3.2, -1.8, -1.8},
{ 3,2704,2703, 2650,   0.250,   4.814,  300.681,  134.3,-103.1, -84.2,   3.2, -1.8, -1.8},
{ 3,2705,2704, 2650,   0.250,   4.814,  305.495,  137.5,-104.9, -85.9,   3.2, -1.8, -1.8},
{ 3,2706,2705, 2650,   0.250,   4.814,  310.310,  140.7,-106.7, -87.7,   3.2, -1.8, -1.8},
{ 3,2707,2706, 2650,   0.250,   4.814,  315.124,  143.9,-108.5, -89.5,   3.2, -1.8, -1.8},
{ 3,2708,2707, 2650,   0.250,   4.814,  319.939,  147.1,-110.4, -91.2,   3.2, -1.8, -1.8},
{ 3,2709,2708, 2650,   0.250,   4.814,  324.753,  150.3,-112.2, -93.0,   3.2, -1.8, -1.8},
{ 3,2710,2709, 2650,   0.250,   4.814,  329.567,  153.4,-114.0, -94.8,   3.2, -1.8, -1.8},
{ 3,2711,2710, 2650,   0.250,   4.814,  334.382,  156.6,-115.8, -96.5,   3.2, -1.8, -1.8},
{ 3,2712,2711, 2650,   0.250,   4.814,  339.196,  159.8,-117.6, -98.3,   3.2, -1.8, -1.8},
{ 3,2713,2712, 2650,   0.250,   4.814,  344.011,  163.0,-119.5,-100.1,   3.2, -1.8, -1.8},
{ 3,2714,2713, 2650,   0.250,   4.814,  348.825,  166.2,-121.3,-101.8,   3.2, -1.8, -1.8},
{ 3,2715,2714, 2650,   0.250,   4.814,  353.639,  169.4,-123.1,-103.6,   3.2, -1.8, -1.8},
{ 3,2716,2715, 2650,   0.250,   4.814,  358.454,  172.6,-124.9,-105.3,   3.2, -1.8, -1.8},
{ 3,2717,2716, 2650,   0.250,   4.814,  363.268,  175.8,-126.7,-107.1,   3.2, -1.8, -1.8},
{ 3,2718,2717, 2650,   0.250,   4.814,  368.082,  179.0,-128.6,-108.9,   3.2, -1.8, -1.8},
{ 3,2719,2626, 2663,   0.400,   4.926,  173.357,   66.9, -61.5, -21.5,   2.7,  1.2, -4.1},
{ 3,2720,2719, 2663,   0.400,   4.926,  178.283,   69.6, -60.3, -25.6,   2.7,  1.2, -4.1},
{ 3,2721,2720, 2663,   0.400,   4.926,  183.209,   72.3, -59.1, -29.6,   2.7,  1.2, -4.1},
{ 3,2722,2721, 2663,   0.400,   4.926,  188.135,   75.1, -58.0, -33.7,   2.7,  1.2, -4.1},
{ 3,2723,2722, 2663,   0.400,   4.926,  193.062,   77.8, -56.8, -37.8,   2.7,  1.2, -4.1},
{ 3,2724,2723, 2663,   0.400,   4.926,  197.988,   80.6, -55.7, -41.8,   2.7,  1.2, -4.1},
{ 3,2725,2724, 2663,   0.400,   4.926,  202.914,   83.3, -54.5, -45.9,   2.7,  1.2, -4.1},
{ 3,2726,2725, 2663,   0.400,   4.926,  207.840,   86.0, -53.4, -49.9,   2.7,  1.2, -4.1},
{ 3,2727,2726, 2663,   0.400,   4.926,  212.766,   88.8, -52.2, -54.0,   2.7,  1.2, -4.1},
{ 3,2728,2727, 2677,   0.400,   5.015,  217.737,   91.4, -51.2, -55.7,   2.6,  0.9,  0.6},
{ 3,2729,2728, 2677,   0.400,   5.015,  222.752,   94.0, -50.3, -55.1,   2.6,  0.9,  0.6},
{ 3,2730,2729, 2677,   0.400,   5.015,  227.767,   96.6, -49.5, -54.4,   2.6,  0.9,  0.6},
{ 3,2731,2730, 2677,   0.400,   5.015,  232.781,   99.2, -48.6, -53.8,   2.6,  0.9,  0.6},
{ 3,2732,2731, 2677,   0.400,   5.015,  237.796,  101.8, -47.7, -53.2,   2.6,  0.9,  0.6},
{ 3,2733,2732, 2677,   0.400,   5.015,  242.811,  104.3, -46.8, -52.5,   2.6,  0.9,  0.6},
{ 3,2734,2733, 2677,   0.400,   5.015,  247.826,  106.9, -45.9, -51.9,   2.6,  0.9,  0.6},
{ 3,2735,2734, 2677,   0.400,   5.015,  252.840,  109.5, -45.0, -51.3,   2.6,  0.9,  0.6},
{ 3,2736,2735, 2677,   0.400,   5.015,  257.855,  112.1, -44.2, -50.6,   2.6,  0.9,  0.6},
{ 3,2737,2736, 2677,   0.400,   5.015,  262.870,  114.6, -43.3, -50.0,   2.6,  0.9,  0.6},
{ 3,2738,2737, 2677,   0.400,   5.015,  267.885,  117.2, -42.4, -49.4,   2.6,  0.9,  0.6},
{ 3,2739,2738, 2677,   0.400,   5.015,  272.900,  119.8, -41.5, -48.7,   2.6,  0.9,  0.6},
{ 3,2740,2739, 2677,   0.400,   5.015,  277.914,  122.4, -40.6, -48.1,   2.6,  0.9,  0.6},
{ 3,2741,2740, 2677,   0.400,   5.015,  282.929,  125.0, -39.8, -47.5,   2.6,  0.9,  0.6},
{ 3,2742,2741, 2677,   0.400,   5.015,  287.944,  127.5, -38.9, -46.8,   2.6,  0.9,  0.6},
{ 3,2743,2742, 2677,   0.400,   5.015,  292.959,  130.1, -38.0, -46.2,   2.6,  0.9,  0.6},
{ 3,2744,2743, 2677,   0.400,   5.015,  297.974,  132.7, -37.1, -45.6,   2.6,  0.9,  0.6},
{ 3,2745,2744, 2677,   0.400,   5.015,  302.988,  135.3, -36.2, -44.9,   2.6,  0.9,  0.6},
{ 3,2746,2745, 2677,   0.400,   5.015,  308.003,  137.8, -35.4, -44.3,   2.6,  0.9,  0.6},
{ 3,2747,2727, 2685,   0.400,   4.756,  217.608,   90.7, -51.3, -56.2,   1.1,  0.7, -0.4},
{ 3,2748,2747, 2685,   0.400,   4.756,  222.364,   91.8, -50.6, -56.6,   1.1,  0.7, -0.4},
{ 3,2749,2748, 2685,   0.400,   4.756,  227.120,   92.8, -50.0, -56.9,   1.1,  0.7, -0.4},
{ 3,2750,2749, 2685,   0.400,   4.756,  231.876,   93.9, -49.3, -57.3,   1.1,  0.7, -0.4},
{ 3,2751,2750, 2685,   0.400,   4.756,  236.632,   95.0, -48.6, -57.7,   1.1,  0.7, -0.4},
{ 3,2752,2751, 2685,   0.400,   4.756,  241.388,   96.0, -48.0, -58.1,   1.1,  0.7, -0.4},
{ 3,2753,2752, 2685,   0.400,   4.756,  246.144,   97.1, -47.3, -58.4,   1.1,  0.7, -0.4},
{ 3,2754,2753, 2685,   0.400,   4.756,  250.901,   98.2, -46.6, -58.8,   1.1,  0.7, -0.4},
{ 3,2755,2754, 2685,   0.400,   4.756,  255.657,   99.2, -45.9, -59.2,   1.1,  0.7, -0.4},
{ 3,2756,2755, 2685,   0.400,   4.756,  260.413,  100.3, -45.3, -59.6,   1.1,  0.7, -0.4},
{ 3,2757,2756, 2685,   0.400,   4.756,  265.169,  101.4, -44.6, -59.9,   1.1,  0.7, -0.4},
{ 3,2758,2757, 2685,   0.400,   4.756,  269.925,  102.4, -43.9, -60.3,   1.1,  0.7, -0.4},
{ 3,2759,2758, 2685,   0.400,   4.756,  274.681,  103.5, -43.3, -60.7,   1.1,  0.7, -0.4},
{ 3,2760,2759, 2685,   0.400,   4.756,  279.437,  104.6, -42.6, -61.1,   1.1,  0.7, -0.4},
{ 3,2761,2760, 2685,   0.400,   4.756,  284.194,  105.6, -41.9, -61.4,   1.1,  0.7, -0.4},
{ 3,2762,2761, 2685,   0.400,   4.756,  288.950,  106.7, -41.2, -61.8,   1.1,  0.7, -0.4},
{ 3,2763,2762, 2685,   0.400,   4.756,  293.706,  107.8, -40.6, -62.2,   1.1,  0.7, -0.4},
{ 3,2764,2763, 2685,   0.400,   4.756,  298.462,  108.8, -39.9, -62.6,   1.1,  0.7, -0.4},

// end marker
{-1,  -1,  -1,   -1,   0.000,   0.000,    0.000,    0.0,   0.0,   0.0,   0.0,  0.0,  0.0}


};	// end of table

return cellMorphology;

}	// end of function
